/* 
 * Licensed to Aduna under one or more contributor license agreements.  
 * See the NOTICE.txt file distributed with this work for additional 
 * information regarding copyright ownership. 
 *
 * Aduna licenses this file to you under the terms of the Aduna BSD 
 * License (the "License"); you may not use this file except in compliance 
 * with the License. See the LICENSE.txt file distributed with this work 
 * for the full License.
 *
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package org.openrdf.sail.rdbms.evaluation;

import java.util.List;

/**
 * Facilitates the building of a SQL query.
 * 
 * @author James Leigh
 * 
 */
public class SqlQueryBuilder {

	private QueryBuilderFactory factory;

	protected boolean distinct;

	protected SqlExprBuilder select;

	protected SqlJoinBuilder from;

	protected StringBuilder group = new StringBuilder();

	protected SqlExprBuilder order;

	protected SqlQueryBuilder union;

	protected Long offset;

	protected Long limit;

	public SqlQueryBuilder(QueryBuilderFactory factory) {
		super();
		this.factory = factory;
		select = factory.createSqlExprBuilder();
		order = factory.createSqlExprBuilder();
	}

	public List<Object> findParameters(List<Object> parameters) {
		parameters.addAll(select.getParameters());
		if (from != null) {
			from.findParameters(parameters);
		}
		if (union != null) {
			union.findParameters(parameters);
		}
		parameters.addAll(order.getParameters());
		return parameters;
	}

	public void distinct() {
		distinct = true;
	}

	public SqlExprBuilder select() {
		if (!select.isEmpty())
			select.append(",\n ");
		return select;
	}

	public SqlJoinBuilder from(String table, String alias) {
		assert from == null : alias;
		return from = factory.createSqlJoinBuilder(table, alias);
	}

	public SqlJoinBuilder from(String alias) {
		assert from == null : alias;
		return from = factory.createSqlJoinBuilder(null, alias);
	}

	public SqlExprBuilder filter() {
		assert from != null;
		return from.on();
	}

	public SqlQueryBuilder groupBy(String... expressions) {
		for (String expr : expressions) {
			if (group.length() == 0) {
				group.append("\nGROUP BY ");
			}
			else {
				group.append(", ");
			}
			group.append(expr);
		}
		return this;
	}

	public SqlQueryBuilder union() {
		assert union == null : union;
		return union = factory.createSqlQueryBuilder();
	}

	public boolean isEmpty() {
		return select.isEmpty() && from == null;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT ");
		if (distinct) {
			sb.append("DISTINCT ");
		}
		if (select.isEmpty()) {
			sb.append("*");
		}
		else {
			sb.append(select.toSql());
		}
		if (from != null) {
			sb.append("\nFROM ").append(from.getFromClause());
			if (!from.on().isEmpty()) {
				sb.append("\nWHERE ");
				sb.append(from.on().toSql());
			}
		}
		sb.append(group);
		if (union != null && !union.isEmpty()) {
			sb.append("\nUNION ALL ");
			sb.append(union.toString());
		}
		if (!order.isEmpty()) {
			sb.append("\nORDER BY ").append(order.toSql());
		}
		if (limit != null) {
			sb.append("\nLIMIT ").append(limit);
		}
		if (offset != null) {
			sb.append("\nOFFSET ").append(offset);
		}
		return sb.toString();
	}

	public SqlExprBuilder orderBy() {
		if (!order.isEmpty())
			order.append(",\n ");
		return order;
	}

	public void offset(Long offset) {
		this.offset = offset;
		if (limit == null) {
			limit = Long.MAX_VALUE;
		}
	}

	public void limit(Long limit) {
		this.limit = limit;
	}
}
