/*
 * Copyright 2021 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.java.dependencies;

import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.EqualsAndHashCode;
import lombok.Value;
import lombok.experimental.NonFinal;
import org.jspecify.annotations.Nullable;
import org.openrewrite.*;
import org.openrewrite.gradle.UpgradeTransitiveDependencyVersion;
import org.openrewrite.gradle.marker.GradleDependencyConfiguration;
import org.openrewrite.gradle.marker.GradleProject;
import org.openrewrite.groovy.GroovyIsoVisitor;
import org.openrewrite.groovy.GroovyVisitor;
import org.openrewrite.groovy.tree.G;
import org.openrewrite.internal.StringUtils;
import org.openrewrite.java.dependencies.internal.StaticVersionComparator;
import org.openrewrite.java.dependencies.internal.Version;
import org.openrewrite.java.dependencies.internal.VersionParser;
import org.openrewrite.java.dependencies.table.VulnerabilityReport;
import org.openrewrite.java.marker.JavaProject;
import org.openrewrite.marker.CommitMessage;
import org.openrewrite.marker.SearchResult;
import org.openrewrite.maven.AddManagedDependency;
import org.openrewrite.maven.MavenIsoVisitor;
import org.openrewrite.maven.MavenVisitor;
import org.openrewrite.maven.UpgradeDependencyVersion;
import org.openrewrite.maven.tree.*;
import org.openrewrite.semver.LatestPatch;
import org.openrewrite.xml.tree.Xml;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

@Value
@EqualsAndHashCode(callSuper = false)
public class DependencyVulnerabilityCheck extends ScanningRecipe<DependencyVulnerabilityCheck.Accumulator> {
    transient VersionParser versionParser = new VersionParser();
    transient VulnerabilityReport report = new VulnerabilityReport(this);

    @Option(displayName = "Scope",
            description = "Match dependencies with the specified scope. Default is `compile`.",
            valid = {"compile", "test", "runtime", "provided"},
            example = "compile",
            required = false)
    @Nullable
    String scope;

    @Option(displayName = "Override transitives",
            description = "When enabled transitive dependencies with vulnerabilities will have their versions overridden. " +
                          "By default only direct dependencies have their version numbers upgraded.",
            example = "false",
            required = false)
    @Nullable
    Boolean overrideTransitive;

    @Option(displayName = "Add search markers",
            description = "Report each vulnerability as search result markers. " +
                          "When enabled you can see which dependencies are bringing in vulnerable transitives in the diff view. " +
                          "By default these markers are omitted, making it easier to see version upgrades within the diff.",
            required = false)
    @Nullable
    Boolean addMarkers;

    @Override
    public String getDisplayName() {
        return "Find and fix vulnerable dependencies";
    }

    @Override
    public String getDescription() {
        //language=markdown
        return "This software composition analysis (SCA) tool detects and upgrades dependencies with publicly disclosed vulnerabilities. " +
               "This recipe both generates a report of vulnerable dependencies and upgrades to newer versions with fixes. " +
               "This recipe **only** upgrades to the latest **patch** version.  If a minor or major upgrade is required to reach the fixed version, this recipe will not make any changes. " +
               "Vulnerability information comes from the [GitHub Security Advisory Database](https://docs.github.com/en/code-security/security-advisories/global-security-advisories/about-the-github-advisory-database), " +
               "which aggregates vulnerability data from several public databases, including the [National Vulnerability Database](https://nvd.nist.gov/) maintained by the United States government. " +
               "Dependencies following [Semantic Versioning](https://semver.org/) will see their _patch_ version updated where applicable.";
    }

    @Override
    public Validated<Object> validate() {
        return super.validate().and(Validated.test("scope", "scope is a valid Maven scope", scope, s -> {
            try {
                Scope.fromName(s);
                return true;
            } catch (Throwable t) {
                return false;
            }
        }));
    }

    @Value
    public static class Accumulator {
        Map<GroupArtifact, List<Vulnerability>> db;
        Map<String, Vulnerabilities> projectToVulnerabilities;
        Scope scope;
        org.openrewrite.maven.UpgradeDependencyVersion.Accumulator mavenUdvAcc;
        org.openrewrite.gradle.UpgradeDependencyVersion.DependencyVersionState gradleUdvAcc;
    }

    @Value
    public static class Vulnerabilities {
        public static Vulnerabilities NONE = new Vulnerabilities(Collections.emptyMap());

        Map<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> gavToVulnerabilities;

        public Set<MinimumDepthVulnerability> computeIfAbsent(ResolvedGroupArtifactVersion gav, Function<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> mappingFunction) {
            return gavToVulnerabilities.computeIfAbsent(gav, mappingFunction);
        }
    }

    @Override
    public Accumulator getInitialValue(ExecutionContext ctx) {
        Scope parsedScope = Scope.fromName(scope);
        CsvMapper csvMapper = new CsvMapper();
        csvMapper.registerModule(new JavaTimeModule());
        Map<GroupArtifact, List<Vulnerability>> db = new HashMap<>();

        try (InputStream resourceAsStream = DependencyVulnerabilityCheck.class.getResourceAsStream("/advisories-maven.csv");
             MappingIterator<Vulnerability> vs = csvMapper.readerWithSchemaFor(Vulnerability.class).readValues(resourceAsStream)) {
            vs.forEachRemaining(v -> {
                String[] ga = v.getGroupArtifact().split(":");
                db.computeIfAbsent(new GroupArtifact(ga[0], ga[1]), g -> new ArrayList<>()).add(v);
            });
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return new Accumulator(db, new LinkedHashMap<>(), parsedScope,
                new org.openrewrite.maven.UpgradeDependencyVersion.Accumulator(),
                new org.openrewrite.gradle.UpgradeDependencyVersion.DependencyVersionState());
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getScanner(Accumulator acc) {
        return new TreeVisitor<Tree, ExecutionContext>() {
            @Override
            public @Nullable Tree visit(@Nullable Tree tree, ExecutionContext ctx) {
                if (tree == null) {
                    return null;
                }
                scanMaven(acc.getDb(), acc.getProjectToVulnerabilities(), acc.getScope()).visitNonNull(tree, ctx);
                scanGradleGroovy(acc.getDb(), acc.getProjectToVulnerabilities(), acc.getScope()).visitNonNull(tree, ctx);
                new UpgradeDependencyVersion("", "", "", null, null, null)
                        .getScanner(acc.getMavenUdvAcc())
                        .visit(tree, ctx);
                new org.openrewrite.gradle.UpgradeDependencyVersion("", "", null, null)
                        .getScanner(acc.getGradleUdvAcc())
                        .visit(tree, ctx);
                return tree;
            }
        };
    }

    @Override
    public Collection<SourceFile> generate(Accumulator acc, ExecutionContext ctx) {
        for (Map.Entry<String, Vulnerabilities> projectToVulnerabilities : acc.getProjectToVulnerabilities().entrySet()) {
            String projectName = projectToVulnerabilities.getKey();
            for (Map.Entry<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> vulnerabilitiesByGav : projectToVulnerabilities.getValue().getGavToVulnerabilities().entrySet()) {
                for (MinimumDepthVulnerability vDepth : vulnerabilitiesByGav.getValue()) {
                    Vulnerability v = vDepth.getVulnerability();
                    ResolvedGroupArtifactVersion gav = vulnerabilitiesByGav.getKey();
                    boolean fixWithVersionUpdateOnly = (vDepth.getMinDepth() == 0 || Boolean.TRUE.equals(overrideTransitive))
                                                       && new LatestPatch(null).isValid(gav.getVersion(), v.getFixedVersion());
                    report.insertRow(ctx, new VulnerabilityReport.Row(
                            projectName,
                            v.getCve(),
                            gav.getGroupId(),
                            gav.getArtifactId(),
                            gav.getVersion(),
                            v.getFixedVersion(),
                            fixWithVersionUpdateOnly,
                            v.getSummary(),
                            v.getSeverity().toString(),
                            vDepth.getMinDepth(),
                            v.getCwes()
                    ));
                }
            }
        }
        return Collections.emptyList();
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor(Accumulator acc) {
        return new TreeVisitor<Tree, ExecutionContext>() {
            @Override
            public Tree visit(@Nullable Tree tree, ExecutionContext ctx) {
                if (tree == null) {
                    return null;
                }
                Tree t = tree;

                if (t.getMarkers().findFirst(MavenResolutionResult.class).isPresent()) {
                    Map<GroupArtifact, VersionBecause> fixes = new LinkedHashMap<>();
                    Vulnerabilities vulnerabilities = acc.getProjectToVulnerabilities().getOrDefault(projectName(t), Vulnerabilities.NONE);
                    if (!vulnerabilities.getGavToVulnerabilities().isEmpty()) {
                        t = enumerateFixesMaven(vulnerabilities.getGavToVulnerabilities(), acc.getScope(), fixes).visitNonNull(t, ctx);
                        for (Map.Entry<GroupArtifact, VersionBecause> gav : fixes.entrySet()) {
                            Tree t2 = new UpgradeDependencyVersion(gav.getKey().getGroupId(), gav.getKey().getArtifactId(),
                                    gav.getValue().getVersion(), null, overrideTransitive, null)
                                    .getVisitor(acc.getMavenUdvAcc())
                                    .visitNonNull(t, ctx);
                            if (t == t2 && gav.getValue().isDotReleaseAmbiguous()) {
                                t2 = new UpgradeDependencyVersion(gav.getKey().getGroupId(), gav.getKey().getArtifactId(),
                                        gav.getValue().getVersion() + ".RELEASE", null, overrideTransitive, null)
                                        .getVisitor(acc.getMavenUdvAcc())
                                        .visitNonNull(t, ctx);
                            }
                            if (t == t2) {
                                if (Boolean.TRUE.equals(overrideTransitive)) {
                                    AddManagedDependency amd = new AddManagedDependency(gav.getKey().getGroupId(), gav.getKey().getArtifactId(), gav.getValue().getVersion(), null, null, null, null, null, null, null);
                                    t2 = amd.getVisitor(amd.getInitialValue(ctx))
                                            .visitNonNull(t, ctx);
                                    if (t == t2 && gav.getValue().isDotReleaseAmbiguous()) {
                                        amd = new AddManagedDependency(gav.getKey().getGroupId(), gav.getKey().getArtifactId(), gav.getValue().getVersion() + ".RELEASE", null, null, null, null, null, null, null);
                                        t2 = amd.getVisitor(amd.getInitialValue(ctx))
                                                .visitNonNull(t, ctx);
                                    }
                                    if (t != t2) {
                                        t = CommitMessage.message(t2, DependencyVulnerabilityCheck.this, requireNonNull(gav.getValue().getBecause()));
                                    }
                                }
                            } else {
                                t = CommitMessage.message(t2, DependencyVulnerabilityCheck.this, requireNonNull(gav.getValue().getBecause()));
                            }
                        }
                    }
                } else if (t.getMarkers().findFirst(GradleProject.class).isPresent()) {
                    Map<GroupArtifact, VersionBecause> fixes = new LinkedHashMap<>();
                    Vulnerabilities vulnerabilities = acc.getProjectToVulnerabilities().getOrDefault(projectName(t), Vulnerabilities.NONE);
                    if (!vulnerabilities.getGavToVulnerabilities().isEmpty()) {
                        t = enumerateFixesGradleGroovy(vulnerabilities.getGavToVulnerabilities(), acc.getScope(), fixes).visitNonNull(t, ctx);
                        for (Map.Entry<GroupArtifact, VersionBecause> gaToVb : fixes.entrySet()) {
                            VersionBecause vb = gaToVb.getValue();
                            Tree t2 = new org.openrewrite.gradle.UpgradeDependencyVersion(gaToVb.getKey().getGroupId(), gaToVb.getKey().getArtifactId(),
                                    vb.getVersion(), null)
                                    .getVisitor(acc.getGradleUdvAcc())
                                    .visitNonNull(t, ctx);
                            if (t == t2 && vb.isDotReleaseAmbiguous()) {
                                t2 = new org.openrewrite.gradle.UpgradeDependencyVersion(gaToVb.getKey().getGroupId(), gaToVb.getKey().getArtifactId(),
                                        vb.getVersion() + ".RELEASE", null)
                                        .getVisitor(acc.getGradleUdvAcc())
                                        .visitNonNull(t, ctx);
                            }
                            if (t == t2) {
                                if (Boolean.TRUE.equals(overrideTransitive)) {
                                    t2 = new UpgradeTransitiveDependencyVersion(gaToVb.getKey().getGroupId(), gaToVb.getKey().getArtifactId(),
                                            vb.getVersion(), null, vb.getBecause(), null)
                                            .getVisitor()
                                            .visitNonNull(t, ctx);
                                    if(t == t2 && vb.isDotReleaseAmbiguous()) {
                                        t2 = new UpgradeTransitiveDependencyVersion(gaToVb.getKey().getGroupId(), gaToVb.getKey().getArtifactId(),
                                                vb.getVersion() + ".RELEASE", null, vb.getBecause(), null)
                                                .getVisitor()
                                                .visitNonNull(t, ctx);
                                    }
                                    if (t != t2) {
                                        t = CommitMessage.message(t2, DependencyVulnerabilityCheck.this, vb.getBecause());
                                    }
                                }
                            } else {
                                t = CommitMessage.message(t2, DependencyVulnerabilityCheck.this, vb.getBecause());
                            }
                        }
                    }
                }

                return t;
            }
        };
    }

    private MavenVisitor<ExecutionContext> scanMaven(
            Map<GroupArtifact, List<Vulnerability>> db,
            Map<String, Vulnerabilities> projectToVulnerabilities,
            Scope aScope) {
        return new MavenIsoVisitor<ExecutionContext>() {
            @Override
            public Xml.Document visitDocument(Xml.Document document, ExecutionContext ctx) {
                List<ResolvedDependency> scopeDependencies = getResolutionResult().getDependencies().get(aScope);
                if (scopeDependencies != null) {
                    String projectName = projectName(document);
                    for (ResolvedDependency resolvedDependency : scopeDependencies) {
                        analyzeDependency(db,
                                projectToVulnerabilities.computeIfAbsent(projectName, p -> new Vulnerabilities(new LinkedHashMap<>())),
                                resolvedDependency);
                    }
                }
                return document;
            }
        };
    }

    private static String projectName(Tree t) {
        return t.getMarkers().findFirst(JavaProject.class)
                .map(JavaProject::getProjectName)
                .orElse("");
    }

    private static boolean scopeExcludesConfiguration(GradleDependencyConfiguration configuration, Scope scope) {
        switch (scope) {
            case Test:
                return !configuration.getName().contains("test");
            case Compile:
            case Runtime:
                return configuration.getName().contains("test");
            case Provided:
                return !configuration.getName().contains("provided") && !configuration.getName().contains("compileOnly");
            default:
                return false;
        }
    }

    private GroovyVisitor<ExecutionContext> scanGradleGroovy(
            Map<GroupArtifact, List<Vulnerability>> db,
            Map<String, Vulnerabilities> projectToVulnerabilities,
            Scope aScope) {
        return new GroovyIsoVisitor<ExecutionContext>() {
            @Override
            public G.CompilationUnit visitCompilationUnit(G.CompilationUnit cu, ExecutionContext ctx) {
                cu.getMarkers().findFirst(GradleProject.class).ifPresent(gradleProject -> {
                    String projectName = projectName(cu);
                    for (GradleDependencyConfiguration configuration : gradleProject.getConfigurations()) {
                        if (scopeExcludesConfiguration(configuration, aScope)) {
                            continue;
                        }
                        for (ResolvedDependency resolvedDependency : configuration.getResolved()) {
                            if (!StringUtils.isBlank(resolvedDependency.getVersion())) {
                                analyzeDependency(db,
                                        projectToVulnerabilities.computeIfAbsent(projectName, p -> new Vulnerabilities(new LinkedHashMap<>())),
                                        resolvedDependency);
                            }
                        }
                    }
                });
                return cu;
            }
        };
    }

    @Value
    private static class VersionBecause {
        String version;

        @Nullable
        String because;

        boolean dotReleaseAmbiguous;
    }

    private void analyzeDependency(
            Map<GroupArtifact, List<Vulnerability>> db,
            Vulnerabilities vulnerabilities,
            ResolvedDependency resolvedDependency) {
        List<Vulnerability> vs = db.get(new GroupArtifact(resolvedDependency.getGroupId(), resolvedDependency.getArtifactId()));
        if (vs != null) {
            Set<MinimumDepthVulnerability> gavVs = null;
            Comparator<Version> vc = new StaticVersionComparator();

            nextVulnerability:
            for (Vulnerability v : vs) {
                // Some dependencies have a ".RELEASE" suffix.
                // For example spring-security-core had a .RELEASE suffix for versions >=2.0.5 and <5.4.0. No suffixes since then
                // The vulnerability database is inconsistent about whether the ".RELEASE" is included in the fixed version
                // This inconsistency complicates comparisons because "5.3.0" != "5.3.0.RELEASE"
                // This inconsistency complicates dependency upgrade since we don't know which version number format to request
                // Therefore ignore the suffix during comparison but record it so that version upgrades can try both with and without the suffix
                // The edge case of ".RELEASE" being introduced into a version scheme between patch versions is possible but hopefully rare
                boolean dotReleaseAmbiguous = resolvedDependency.getVersion().endsWith(".RELEASE") && !v.getFixedVersion().endsWith(".RELEASE");
                boolean isLessThanFixed = StringUtils.isBlank(v.getFixedVersion());
                if (!isLessThanFixed
                    && vc.compare(
                        versionParser.transform(stripExtraneousVersionSuffix(v.getFixedVersion())),
                        versionParser.transform(stripExtraneousVersionSuffix(resolvedDependency.getVersion()))) > 0) {
                    isLessThanFixed = true;
                }

                if (isLessThanFixed
                    && vc.compare(
                        versionParser.transform(stripExtraneousVersionSuffix(v.getIntroducedVersion())),
                        versionParser.transform(stripExtraneousVersionSuffix(resolvedDependency.getVersion()))) <= 0) {
                    if (gavVs == null) {
                        gavVs = vulnerabilities.computeIfAbsent(resolvedDependency.getGav(), ga -> new TreeSet<>(
                                Comparator.comparing((MinimumDepthVulnerability vDep) -> vDep.getVulnerability().getSeverity()).reversed()
                                        .thenComparing((MinimumDepthVulnerability vDep) -> vDep.getVulnerability().getCve())));
                    }

                    for (MinimumDepthVulnerability vDep : gavVs) {
                        if (vDep.getVulnerability().equals(v)) {
                            vDep.minDepth = Math.min(vDep.minDepth, resolvedDependency.getDepth());
                            continue nextVulnerability;
                        }
                    }

                    gavVs.add(new MinimumDepthVulnerability(resolvedDependency.getDepth(), v, dotReleaseAmbiguous));
                }
            }
        }
    }

    private MavenVisitor<ExecutionContext> enumerateFixesMaven(
            Map<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> vulnerabilities,
            Scope aScope,
            Map<GroupArtifact, VersionBecause> fixes
    ) {
        return new MavenIsoVisitor<ExecutionContext>() {
            @Override
            public Xml.Tag visitTag(Xml.Tag tag, ExecutionContext ctx) {
                if (isDependencyTag()) {
                    ResolvedDependency resolved = findDependency(tag, aScope);
                    if (resolved != null) {
                        for (Map.Entry<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> vulnerabilitiesByGav : vulnerabilities.entrySet()) {
                            ResolvedGroupArtifactVersion gav = vulnerabilitiesByGav.getKey();
                            ResolvedDependency match = resolved.findDependency(requireNonNull(gav.getGroupId()), gav.getArtifactId());
                            if (match != null) {
                                boolean vulnerable = false;

                                for (MinimumDepthVulnerability vDepth : vulnerabilitiesByGav.getValue()) {
                                    Vulnerability v = vDepth.getVulnerability();
                                    vulnerable = true;

                                    GroupArtifact ga = new GroupArtifact(gav.getGroupId(), gav.getArtifactId());
                                    String fixVersion = (fixes.get(ga) == null) ? null : fixes.get(ga).getVersion();
                                    if (!StringUtils.isBlank(v.getFixedVersion()) &&
                                        new LatestPatch(null).isValid(gav.getVersion(), v.getFixedVersion())) {
                                        if (fixVersion == null || new StaticVersionComparator().compare(versionParser.transform(v.getFixedVersion()), versionParser.transform(fixVersion)) > 0) {
                                            fixes.put(ga, new VersionBecause(v.getFixedVersion(), v.getCve(), vDepth.dotReleaseAmbiguous));
                                        }
                                    }
                                }

                                if (vulnerable && Boolean.TRUE.equals(addMarkers)) {
                                    return SearchResult.found(tag, "This dependency includes " + gav + " which has the following vulnerabilities:\n" +
                                                                   vulnerabilitiesByGav.getValue().stream()
                                                                           .map(vDepth -> {
                                                                               Vulnerability v = vDepth.getVulnerability();
                                                                               return v.getCve() + " (" + v.getSeverity() + " severity" +
                                                                                      (StringUtils.isBlank(v.getFixedVersion()) ? "" : ", fixed in " + v.getFixedVersion()) +
                                                                                      ") - " + v.getSummary();
                                                                           })
                                                                           .collect(Collectors.joining("\n")));
                                }
                            }
                        }
                    }
                }
                return super.visitTag(tag, ctx);
            }
        };
    }

    private GroovyVisitor<ExecutionContext> enumerateFixesGradleGroovy(
            Map<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> vulnerabilities,
            Scope aScope,
            Map<GroupArtifact, VersionBecause> fixes
    ) {
        return new GroovyIsoVisitor<ExecutionContext>() {
            @Override
            public G.CompilationUnit visitCompilationUnit(G.CompilationUnit cu, ExecutionContext ctx) {
                G.CompilationUnit c = cu;
                //noinspection OptionalGetWithoutIsPresent
                GradleProject gp = c.getMarkers().findFirst(GradleProject.class).get();
                for (Map.Entry<ResolvedGroupArtifactVersion, Set<MinimumDepthVulnerability>> vulnerabilitiesByGav : vulnerabilities.entrySet()) {
                    ResolvedGroupArtifactVersion gav = vulnerabilitiesByGav.getKey();
                    if (Boolean.TRUE.equals(addMarkers)) {
                        c = SearchResult.found(c, "This project has the following vulnerabilities:\n" +
                                                  vulnerabilitiesByGav.getValue().stream()
                                                          .map(vDepth -> {
                                                              Vulnerability v = vDepth.getVulnerability();
                                                              return "Dependency " + gav + " has " + v.getCve() + " (" + v.getSeverity() + " severity" +
                                                                     (StringUtils.isBlank(v.getFixedVersion()) ? "" : ", fixed in " + v.getFixedVersion()) +
                                                                     ") - " + v.getSummary();
                                                          })
                                                          .collect(Collectors.joining("\n")));
                    }
                    for (GradleDependencyConfiguration configuration : gp.getConfigurations()) {
                        if (scopeExcludesConfiguration(configuration, aScope) || !configuration.isCanBeResolved()) {
                            continue;
                        }
                        List<ResolvedDependency> resolved = configuration.getResolved();
                        for (ResolvedDependency resolvedDependency : resolved) {
                            if (Objects.equals(resolvedDependency.getGroupId(), gav.getGroupId())
                                && Objects.equals(resolvedDependency.getArtifactId(), gav.getArtifactId())) {

                                for (MinimumDepthVulnerability vDepth : vulnerabilitiesByGav.getValue()) {
                                    Vulnerability v = vDepth.getVulnerability();

                                    GroupArtifact ga = new GroupArtifact(gav.getGroupId(), gav.getArtifactId());
                                    String fixVersion = (fixes.get(ga) == null) ? null : fixes.get(ga).getVersion();
                                    if (!StringUtils.isBlank(v.getFixedVersion()) &&
                                        new LatestPatch(null).isValid(gav.getVersion(), v.getFixedVersion())) {
                                        if (fixVersion == null || new StaticVersionComparator().compare(versionParser.transform(v.getFixedVersion()), versionParser.transform(fixVersion)) > 0) {
                                            fixes.put(ga, new VersionBecause(v.getFixedVersion(), v.getCve(), vDepth.dotReleaseAmbiguous));
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                return c;
            }
        };
    }

    @Value
    public static class MinimumDepthVulnerability {
        @NonFinal
        int minDepth;

        Vulnerability vulnerability;
        boolean dotReleaseAmbiguous;
    }

    private static String stripExtraneousVersionSuffix(String version) {
        if (version.endsWith(".RELEASE")) {
            return version.substring(0, version.length() - ".RELEASE".length());
        }
        return version;
    }
}
