/*
 * Copyright 2021 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.java.dependencies.table;

import com.fasterxml.jackson.annotation.JsonIgnoreType;
import lombok.Value;
import org.openrewrite.Column;
import org.openrewrite.DataTable;
import org.openrewrite.Recipe;

@JsonIgnoreType
public class VulnerabilityReport extends DataTable<VulnerabilityReport.Row> {

    public VulnerabilityReport(Recipe recipe) {
        super(recipe,
                "Vulnerability report",
                "A vulnerability report that includes detailed information about the affected artifact and the corresponding CVEs.");
    }

    @Value
    public static class Row {
        @Column(displayName = "Project",
                description = "The name of the project / module taking the dependency. Relevant in repositories with multiple modules.")
        String projectName;

        @Column(displayName = "CVE",
                description = "The CVE number.")
        String cve;

        @Column(displayName = "Group",
                description = "The first part of a dependency coordinate `com.google.guava:guava:VERSION`.")
        String groupId;

        @Column(displayName = "Artifact",
                description = "The second part of a dependency coordinate `com.google.guava:guava:VERSION`.")
        String artifactId;

        @Column(displayName = "Version",
                description = "The resolved version.")
        String version;

        @Column(displayName = "Fixed in version",
                description = "The minimum version that is no longer vulnerable.")
        String fixedVersion;

        @Column(displayName = "Fixable with version update only",
                //language=markdown
                description = "Whether the vulnerability is likely to be fixed by increasing the dependency version only, " +
                              "with no code modifications required. This is a heuristic which assumes that the dependency " +
                              "is accurately versioned according to [semver](https://semver.org/).")
        boolean fixWithVersionUpdateOnly;

        @Column(displayName = "Summary",
                description = "The summary of the CVE.")
        String summary;

        @Column(displayName = "Base score",
                description = "The calculated base score.")
        String severity;

        @Column(displayName = "Depth",
                description = "Zero for direct dependencies.")
        Integer depth;

        @Column(displayName = "CWEs",
                description = "Common Weakness Enumeration (CWE) identifiers; semicolon separated.")
        String CWEs;
    }
}
