/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.io;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IChemObject;
import org.openscience.cdk.interfaces.IChemObjectBuilder;
import org.openscience.cdk.io.CharIter;
import org.openscience.cdk.io.DefaultChemObjectReader;
import org.openscience.cdk.io.MDLRXNV2000Reader;
import org.openscience.cdk.io.MDLRXNV3000Reader;
import org.openscience.cdk.io.MDLV2000Reader;
import org.openscience.cdk.io.MDLV3000Reader;
import org.openscience.cdk.io.RdfileRecord;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;

public final class RdfileReader
implements Closeable,
Iterator<RdfileRecord> {
    private static final ILoggingTool LOGGER = LoggingToolFactory.createLoggingTool(RdfileReader.class);
    static final String RDFILE_VERSION_1 = "$RDFILE 1";
    static final String DATM = "$DATM";
    static final String REACTION_FMT = "$RFMT";
    static final String REACTION_INT_REG = "$RIREG";
    static final String REACTION_EXT_REG = "$REREG";
    static final String RXNFILE_START = "$RXN";
    static final String MOLECULE_FMT = "$MFMT";
    static final String MOLECULE_INT_REG = "$MIREG";
    static final String MOLECULE_EXT_REG = "$MEREG";
    static final String MOLFILE_START = "$MOL";
    static final String M_END = "M  END";
    static final String DTYPE = "$DTYPE";
    static final String DATUM = "$DATUM";
    static final String LINE_SEPARATOR_NEWLINE = "\n";
    static final String PLUS_SIGN = "+";
    private static final Pattern DTYPE_KEY = Pattern.compile("\\A\\$DTYPE (.+)\\Z");
    private static final Pattern DATUM_VALUE_FIRSTLINE = Pattern.compile("\\A\\$DATUM (.{73}\\+|.{1,72})\\Z");
    private static final Pattern MDL_CTAB_VERSION = Pattern.compile("(V[23]0{3})\\Z");
    private static final Pattern MDL_RXN_VERSION = Pattern.compile("\\A\\$RXN ?(V30{3})?\\Z");
    private boolean headerRead;
    private String previousLine;
    private int lineCounter;
    private RdfileRecord nextRecord;
    private boolean endOfFile;
    private final boolean continueOnError;
    private final BufferedReader bufferedReader;
    private final IChemObjectBuilder chemObjectBuilder;

    public RdfileReader(InputStream in, IChemObjectBuilder chemObjectBuilder) {
        this(in, chemObjectBuilder, true);
    }

    public RdfileReader(InputStream in, IChemObjectBuilder chemObjectBuilder, boolean continueOnError) {
        this(new InputStreamReader(in, StandardCharsets.UTF_8), chemObjectBuilder, continueOnError);
    }

    public RdfileReader(Reader reader, IChemObjectBuilder chemObjectBuilder, boolean continueOnError) {
        this.bufferedReader = reader instanceof BufferedReader ? (BufferedReader)reader : new BufferedReader(reader);
        if (chemObjectBuilder == null) {
            throw new NullPointerException("A ChemObjectBuilder must be provided or available on the class path!");
        }
        this.chemObjectBuilder = chemObjectBuilder;
        this.continueOnError = continueOnError;
    }

    @Override
    public boolean hasNext() {
        if (this.nextRecord != null) {
            return true;
        }
        if (this.endOfFile) {
            return false;
        }
        this.nextRecord = this.readNext();
        return this.nextRecord != null;
    }

    @Override
    public RdfileRecord next() {
        if (this.nextRecord == null && !this.endOfFile) {
            this.nextRecord = this.readNext();
        }
        if (this.nextRecord == null) {
            throw new NoSuchElementException("RdfileReader reached end of file.");
        }
        RdfileRecord returnValue = this.nextRecord;
        this.nextRecord = null;
        return returnValue;
    }

    RdfileRecord readNext() {
        try {
            return this.doReadNext();
        }
        catch (IOException exception) {
            LOGGER.error((Object)("I/O error when reading RDfile: " + exception.getMessage()));
            return null;
        }
        catch (CDKException exception) {
            LOGGER.error((Object)("Parsing error when reading RDfile: " + exception.getMessage()));
            if (!this.continueOnError) {
                this.endOfFile = true;
                return null;
            }
            try {
                String line;
                while ((line = this.nextLine()) != null) {
                    if (!this.isStartOfNewRecord(line)) continue;
                    this.pushBack(line);
                    break;
                }
            }
            catch (IOException ioException) {
                LOGGER.error((Object)("I/O error when reading RDfile: " + ioException.getMessage()));
                return null;
            }
            return this.readNext();
        }
    }

    RdfileRecord doReadNext() throws IOException, CDKException {
        String line;
        if (!this.headerRead) {
            String header = this.nextLine();
            if (header == null || !header.startsWith(RDFILE_VERSION_1)) {
                throw new CDKException("Error in line " + this.lineCounter + ": Expected the line to exactly contain '" + RDFILE_VERSION_1 + "', but instead found '" + header + "'.");
            }
            String date = this.nextLine();
            if (date == null || !date.startsWith(DATM)) {
                throw new CDKException("Error in line " + this.lineCounter + ": Expected the line to start with '" + DATM + "', but instead found '" + date + "'.");
            }
            this.headerRead = true;
        }
        if ((line = this.nextLine()) == null) {
            this.endOfFile = true;
            return null;
        }
        RdfileRecord rdFileRecord = this.processMoleculeAndReactionIdentifiers(new CharIter(line), MOLECULE_FMT, MOLECULE_INT_REG, MOLECULE_EXT_REG, false);
        if (rdFileRecord == null) {
            rdFileRecord = this.processMoleculeAndReactionIdentifiers(new CharIter(line), REACTION_FMT, REACTION_INT_REG, REACTION_EXT_REG, true);
        }
        if (rdFileRecord == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected the line to specify the molecule or reaction identifier, but instead found '" + line + "'.");
        }
        line = this.nextLine();
        if (line == null) {
            String expected = rdFileRecord.isMolfile() ? MOLFILE_START : RXNFILE_START;
            throw new CDKException("Error in line " + this.lineCounter + ": Expected this line to start with '" + expected + "', but instead found '" + line + "'.");
        }
        if (rdFileRecord.isMolfile()) {
            rdFileRecord.setContent(this.processMolfile(line, rdFileRecord));
        } else if (rdFileRecord.isRxnFile()) {
            rdFileRecord.setContent(this.processRxn(line, rdFileRecord));
        }
        line = this.nextLine();
        if (line == null) {
            rdFileRecord.setChemObject(this.parseChemicalContent(rdFileRecord));
            return rdFileRecord;
        }
        if (line.startsWith(DTYPE)) {
            Map<Object, Object> dataMap = this.processDataBlock(line);
            rdFileRecord.setData(dataMap);
        } else if (this.isStartOfNewRecord(line)) {
            this.pushBack(line);
        } else {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected start of data block '" + DTYPE + "' or start of next record, but instead found '" + line + "'.");
        }
        rdFileRecord.setChemObject(this.parseChemicalContent(rdFileRecord));
        return rdFileRecord;
    }

    @Override
    public void close() throws IOException {
        this.bufferedReader.close();
    }

    private String nextLine() throws IOException {
        if (this.previousLine != null) {
            String nextLine = this.previousLine;
            this.previousLine = null;
            return nextLine;
        }
        ++this.lineCounter;
        return this.bufferedReader.readLine();
    }

    private void pushBack(String line) {
        this.previousLine = line;
    }

    private RdfileRecord processMoleculeAndReactionIdentifiers(CharIter iter, String fmt, String intReg, String extReg, boolean isTestForRxn) throws CDKException {
        String intRegNo = null;
        String extRegNo = null;
        if (iter.consume(fmt)) {
            iter.skipWhiteSpace();
            if (iter.consume(intReg)) {
                iter.skipWhiteSpace();
                intRegNo = iter.rest();
            } else if (iter.consume(extReg)) {
                iter.skipWhiteSpace();
                extRegNo = iter.rest();
            } else {
                String rest = iter.rest();
                if (!rest.isEmpty()) {
                    throw new CDKException("Error in line " + this.lineCounter + ": Expected either '" + MOLECULE_INT_REG + "' or '" + MOLECULE_EXT_REG + "' after '" + MOLECULE_FMT + "', but instead found '" + rest + "'.");
                }
            }
            return new RdfileRecord(intRegNo, extRegNo, isTestForRxn);
        }
        if (iter.consume(intReg)) {
            iter.skipWhiteSpace();
            intRegNo = iter.rest();
            return new RdfileRecord(intRegNo, extRegNo, isTestForRxn);
        }
        if (iter.consume(extReg)) {
            iter.skipWhiteSpace();
            extRegNo = iter.rest();
            return new RdfileRecord(intRegNo, extRegNo, isTestForRxn);
        }
        return null;
    }

    private String processMolfile(String line, RdfileRecord rdfileRecord) throws IOException, CDKException {
        return this.processMolfile(line, rdfileRecord, null);
    }

    private String processMolfile(String line, RdfileRecord rdfileRecord, RdfileRecord.CTAB_VERSION ctabVersionRecord) throws IOException, CDKException {
        int lastIndexOf;
        if (!line.startsWith(MOLFILE_START)) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected the line to start with '" + MOLFILE_START + "', but instead found '" + line + "'.");
        }
        StringBuilder sb = new StringBuilder();
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with the title of the molecule, but instead found '" + line + "'.");
        }
        sb.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with program, date and time, dimensions etc., but instead found '" + line + "'.");
        }
        sb.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with a comment, but instead found '" + line + "'.");
        }
        sb.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with a comment, but instead found '" + line + "'.");
        }
        Matcher matcher = MDL_CTAB_VERSION.matcher(line);
        if (!matcher.find()) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a counts line that ends with a version of either '" + (Object)((Object)RdfileRecord.CTAB_VERSION.V2000) + "' or '" + (Object)((Object)RdfileRecord.CTAB_VERSION.V3000) + "', but instead found '" + line + "'.");
        }
        RdfileRecord.CTAB_VERSION ctabVersion = RdfileRecord.CTAB_VERSION.valueOf(matcher.group(1));
        if (ctabVersionRecord == null) {
            rdfileRecord.setCtabVersion(ctabVersion);
        } else if (ctabVersionRecord != ctabVersion) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected the CTAB version '" + (Object)((Object)ctabVersion) + "' of this Molfile to be the same as the CTAB version '" + (Object)((Object)ctabVersionRecord) + "' of the record, but instead found '" + line + "'.");
        }
        sb.append(line).append(LINE_SEPARATOR_NEWLINE);
        while ((line = this.nextLine()) != null) {
            sb.append(line);
            sb.append(LINE_SEPARATOR_NEWLINE);
            if (line.startsWith(M_END)) break;
            if (!line.startsWith("$")) continue;
            throw new CDKException("Error in line " + this.lineCounter + ": Unexpected character '$' within molfile at start of line '" + line + "'.");
        }
        if ((lastIndexOf = sb.lastIndexOf(M_END)) == -1 || lastIndexOf != sb.length() - 7) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected molfile to end with '" + M_END + "', but instead found '" + line + "'.");
        }
        return sb.toString();
    }

    private String processRxn(String line, RdfileRecord rdfileRecord) throws IOException, CDKException {
        Matcher matcher = MDL_RXN_VERSION.matcher(line);
        if (!matcher.matches()) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected the line to start with '" + RXNFILE_START + "', but instead found '" + line + "'.");
        }
        if (matcher.group(1) == null) {
            rdfileRecord.setCtabVersion(RdfileRecord.CTAB_VERSION.V2000);
        } else {
            rdfileRecord.setCtabVersion(RdfileRecord.CTAB_VERSION.V3000);
        }
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with the title of the reaction, but instead found '" + line + "'.");
        }
        stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected the line with the header, but instead found '" + line + "'.");
        }
        stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with a remark, but instead found '" + line + "'.");
        }
        stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
        line = this.nextLine();
        if (line == null) {
            throw new CDKException("Error in line " + this.lineCounter + ": Expected a line with the counts for the number of reactants, products and agents, but instead found '" + line + "'.");
        }
        int numReactants = this.readMdlUInt(line, 0);
        int numProducts = this.readMdlUInt(line, 3);
        int numAgents = this.readMdlUInt(line, 6);
        if (numReactants == -1 || numProducts == -1) {
            throw new CDKException("Error in line " + this.lineCounter + ": Incorrect formatting of the line that indicates the number of reaction components '" + line + "'.");
        }
        int numReactionComponents = numReactants + numProducts;
        if (numAgents != -1) {
            numReactionComponents += numAgents;
        }
        stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
        int molfileCounter = 0;
        while ((line = this.nextLine()) != null) {
            if (line.startsWith(DTYPE) || this.isStartOfNewRecord(line)) {
                this.pushBack(line);
                break;
            }
            ++molfileCounter;
            stringBuilder.append(MOLFILE_START).append(LINE_SEPARATOR_NEWLINE);
            stringBuilder.append(this.processMolfile(line, rdfileRecord, rdfileRecord.getCtabVersion()));
        }
        if (molfileCounter != numReactionComponents) {
            throw new CDKException("Error in RXN record: The sum of the number of reactants (" + numReactants + "), products (" + numProducts + ") and agents (" + numAgents + ") is not equal to the number of Molfile entries " + molfileCounter + " in the record.");
        }
        return stringBuilder.toString();
    }

    private Map<Object, Object> processDataBlock(String line) throws CDKException, IOException {
        LinkedHashMap<Object, Object> dataMap = new LinkedHashMap<Object, Object>();
        while (line != null) {
            if (this.isStartOfNewRecord(line)) {
                this.pushBack(line);
                break;
            }
            Matcher dtypeMatcher = DTYPE_KEY.matcher(line);
            if (!dtypeMatcher.matches()) {
                throw new CDKException("Error in data block in line " + this.lineCounter + ". Expected line to start with '" + DTYPE + "' and be followed by a space and a string, but instead found '" + line + "'.");
            }
            String key = dtypeMatcher.group(1);
            line = this.nextLine();
            if (line == null) {
                throw new CDKException("Error in data block in line " + this.lineCounter + ". Expected line to start with '" + DATUM + "', but instead found '" + line + "'.");
            }
            StringBuilder stringBuilder = new StringBuilder();
            Matcher datumFirstLineMatcher = DATUM_VALUE_FIRSTLINE.matcher(line);
            if (!datumFirstLineMatcher.matches()) {
                throw new CDKException("Error in data block in line " + this.lineCounter + ". Expected line to start with '" + DATUM + "' and to either have <= 80 characters or exactly 81 characters and ending with a '+'', but instead found '" + line + "'.");
            }
            stringBuilder.append(datumFirstLineMatcher.group(1));
            if (stringBuilder.charAt(stringBuilder.length() - 1) == '+') {
                stringBuilder.deleteCharAt(stringBuilder.length() - 1);
            } else {
                stringBuilder.append(LINE_SEPARATOR_NEWLINE);
            }
            while ((line = this.nextLine()) != null) {
                if (line.startsWith("$")) {
                    this.pushBack(line);
                    break;
                }
                if (line.length() <= 80) {
                    stringBuilder.append(line).append(LINE_SEPARATOR_NEWLINE);
                    continue;
                }
                if (line.length() == 81 && line.endsWith(PLUS_SIGN)) {
                    stringBuilder.append(line, 0, 80);
                    continue;
                }
                throw new CDKException("Error in data block in line " + this.lineCounter + ". Expected multi-line datum with either less than or equal to 80 characters or with exactly 81 characters and ending with a '+', but instead found '" + line + "'.");
            }
            if (stringBuilder.charAt(stringBuilder.length() - 1) == '\n') {
                stringBuilder.deleteCharAt(stringBuilder.length() - 1);
            }
            dataMap.put(key, stringBuilder.toString());
            line = this.nextLine();
        }
        return dataMap;
    }

    private IChemObject parseChemicalContent(RdfileRecord rdFileRecord) throws CDKException {
        IChemObject chemObject;
        if (rdFileRecord.isMolfile()) {
            DefaultChemObjectReader reader = rdFileRecord.getCtabVersion() == RdfileRecord.CTAB_VERSION.V2000 ? new MDLV2000Reader(new StringReader(rdFileRecord.getContent())) : new MDLV3000Reader(new StringReader(rdFileRecord.getContent()));
            chemObject = reader.read((IChemObject)this.chemObjectBuilder.newAtomContainer());
        } else {
            DefaultChemObjectReader reader = rdFileRecord.getCtabVersion() == RdfileRecord.CTAB_VERSION.V2000 ? new MDLRXNV2000Reader(new StringReader(rdFileRecord.getContent())) : new MDLRXNV3000Reader(new StringReader(rdFileRecord.getContent()));
            chemObject = reader.read((IChemObject)this.chemObjectBuilder.newReaction());
        }
        chemObject.addProperties(rdFileRecord.getData());
        return chemObject;
    }

    private int readMdlUInt(String str, int pos) {
        if (pos + 3 > str.length()) {
            return -1;
        }
        if (!CharIter.isDigit(str.charAt(pos + 2))) {
            return -1;
        }
        int num = str.charAt(pos + 2) - 48;
        if (CharIter.isDigit(str.charAt(pos + 1))) {
            num = 10 * (str.charAt(pos + 1) - 48) + num;
        }
        if (CharIter.isDigit(str.charAt(pos))) {
            num = 100 * (str.charAt(pos) - 48) + num;
        }
        return num;
    }

    private boolean isStartOfNewRecord(String line) {
        return line.startsWith(REACTION_FMT) || line.startsWith(MOLECULE_FMT) || line.startsWith(REACTION_INT_REG) || line.startsWith(REACTION_EXT_REG) || line.startsWith(MOLECULE_INT_REG) || line.startsWith(MOLECULE_EXT_REG);
    }
}

