/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.tools;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.exception.InvalidSmilesException;
import org.openscience.cdk.graph.ConnectivityChecker;
import org.openscience.cdk.graph.GraphUtil;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IAtomContainerSet;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemObjectBuilder;
import org.openscience.cdk.isomorphism.DfPattern;
import org.openscience.cdk.isomorphism.Mappings;
import org.openscience.cdk.isomorphism.UniversalIsomorphismTester;
import org.openscience.cdk.ringsearch.RingSearch;
import org.openscience.cdk.smarts.SmartsPattern;
import org.openscience.cdk.smiles.SmilesParser;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.openscience.cdk.tools.manipulator.AtomContainerComparator;
import org.openscience.cdk.tools.manipulator.AtomContainerManipulator;
import org.openscience.cdk.tools.manipulator.BondManipulator;

public class SugarRemovalUtility {
    public static final String INDEX_PROPERTY_KEY = "SUGAR_REMOVAL_UTILITY_UNIQUE_ATOM_INDEX";
    public static final String IS_SPIRO_ATOM_PROPERTY_KEY = "SUGAR_REMOVAL_UTILITY_IS_SPIRO_ATOM";
    public static final String TOO_SMALL_DISCONNECTED_PART_TO_PRESERVE_PROPERTY_KEY = "SUGAR_REMOVAL_UTILITY_TOO_SMALL_DISCONNECTED_PART_TO_PRESERVE";
    protected static final String[] LINEAR_SUGARS_SMILES = new String[]{"C(C(C(C(C(C(C=O)O)O)O)O)O)O", "C(C(C(C(C(C=O)O)O)O)O)O", "C(C(C(C(C=O)O)O)O)O", "C(C(C(C=O)O)O)O", "C(C(C=O)O)O", "C(C(C(C(C(C(CO)O)O)O)O)=O)O", "C(C(C(C(C(CO)O)O)O)=O)O", "C(C(C(C(CO)O)O)=O)O", "C(C(C(CO)O)=O)O", "C(C(CO)=O)O", "C(C(C(C(C(C(CO)O)O)O)O)O)O", "C(C(C(C(C(CO)O)O)O)O)O", "C(C(C(C(CO)O)O)O)O", "C(C(C(CO)O)O)O", "C(C(CO)O)O", "C(C(C(C(CC=O)O)O)O)O"};
    protected static final String[] LINEAR_ACIDIC_SUGARS_SMILES = new String[]{"C(C(CC(C(CO)O)O)O)(O)=O", "CC(CC(CC(=O)O)O)O", "O=C(O)CC(O)CC(=O)O", "O=C(O)CCC(O)C(=O)O", "C(C(C(CC(=O)O)O)O)O"};
    protected static final String[] CIRCULAR_SUGARS_SMILES = new String[]{"C1CCOC1", "C1CCOCC1", "C1CCCOCC1"};
    public static final boolean DETECT_CIRCULAR_SUGARS_ONLY_WITH_O_GLYCOSIDIC_BOND_DEFAULT = false;
    public static final boolean REMOVE_ONLY_TERMINAL_SUGARS_DEFAULT = true;
    public static final PreservationMode PRESERVATION_MODE_DEFAULT = PreservationMode.HEAVY_ATOM_COUNT;
    public static final boolean DETECT_CIRCULAR_SUGARS_ONLY_WITH_ENOUGH_EXOCYCLIC_OXYGEN_ATOMS_DEFAULT = true;
    public static final double EXOCYCLIC_OXYGEN_ATOMS_TO_ATOMS_IN_RING_RATIO_THRESHOLD_DEFAULT = 0.5;
    public static final boolean DETECT_LINEAR_SUGARS_IN_RINGS_DEFAULT = false;
    public static final int LINEAR_SUGAR_CANDIDATE_MIN_SIZE_DEFAULT = 4;
    public static final int LINEAR_SUGAR_CANDIDATE_MAX_SIZE_DEFAULT = 7;
    public static final boolean DETECT_LINEAR_ACIDIC_SUGARS_DEFAULT = false;
    public static final boolean DETECT_SPIRO_RINGS_AS_CIRCULAR_SUGARS_DEFAULT = false;
    public static final boolean DETECT_CIRCULAR_SUGARS_WITH_KETO_GROUPS_DEFAULT = false;
    protected static final SmartsPattern ESTER_SMARTS_PATTERN = SmartsPattern.create((String)"[C](=O)-[O!R]-[C]");
    protected static final SmartsPattern ETHER_SMARTS_PATTERN = SmartsPattern.create((String)"[C]-[O!R]-[C]");
    protected static final SmartsPattern PEROXIDE_SMARTS_PATTERN = SmartsPattern.create((String)"[C]-[O!R]-[O!R]-[C]");
    private static final ILoggingTool LOGGER = LoggingToolFactory.createLoggingTool(SugarRemovalUtility.class);
    private final IChemObjectBuilder builder;
    private List<IAtomContainer> linearSugarStructuresList;
    private List<IAtomContainer> circularSugarStructuresList;
    private List<DfPattern> linearSugarPatternsList;
    private List<IAtomContainer> linearAcidicSugarStructuresList;
    private boolean detectCircularSugarsOnlyWithOGlycosidicBondSetting;
    private boolean removeOnlyTerminalSugarsSetting;
    private PreservationMode preservationModeSetting;
    private int preservationModeThresholdSetting;
    private boolean detectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting;
    private double exocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting;
    private boolean detectLinearSugarsInRingsSetting;
    private int linearSugarCandidateMinSizeSetting;
    private int linearSugarCandidateMaxSizeSetting;
    private boolean detectLinearAcidicSugarsSetting;
    private boolean detectSpiroRingsAsCircularSugarsSetting;
    private boolean detectCircularSugarsWithKetoGroupsSetting;

    public SugarRemovalUtility(IChemObjectBuilder builder) {
        if (builder == null) {
            throw new NullPointerException("Given chem object builder is null.");
        }
        this.builder = builder;
        this.detectLinearAcidicSugarsSetting = false;
        this.restoreDefaultSettings();
    }

    public List<IAtomContainer> getLinearSugarPatternsList() {
        return this.linearSugarStructuresList;
    }

    public List<IAtomContainer> getCircularSugarPatternsList() {
        return this.circularSugarStructuresList;
    }

    public boolean areOnlyCircularSugarsWithOGlycosidicBondDetected() {
        return this.detectCircularSugarsOnlyWithOGlycosidicBondSetting;
    }

    public boolean areOnlyTerminalSugarsRemoved() {
        return this.removeOnlyTerminalSugarsSetting;
    }

    public PreservationMode getPreservationModeSetting() {
        return this.preservationModeSetting;
    }

    public int getPreservationModeThresholdSetting() {
        return this.preservationModeThresholdSetting;
    }

    public boolean areOnlyCircularSugarsWithEnoughExocyclicOxygenAtomsDetected() {
        return this.detectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting;
    }

    public double getExocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting() {
        return this.exocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting;
    }

    public boolean areLinearSugarsInRingsDetected() {
        return this.detectLinearSugarsInRingsSetting;
    }

    public int getLinearSugarCandidateMinSizeSetting() {
        return this.linearSugarCandidateMinSizeSetting;
    }

    public int getLinearSugarCandidateMaxSizeSetting() {
        return this.linearSugarCandidateMaxSizeSetting;
    }

    public boolean areLinearAcidicSugarsDetected() {
        return this.detectLinearAcidicSugarsSetting;
    }

    public boolean areSpiroRingsDetectedAsCircularSugars() {
        return this.detectSpiroRingsAsCircularSugarsSetting;
    }

    public boolean areCircularSugarsWithKetoGroupsDetected() {
        return this.detectCircularSugarsWithKetoGroupsSetting;
    }

    public boolean addCircularSugarToPatternsList(IAtomContainer circularSugar) {
        boolean additionWasSuccessful;
        if (circularSugar == null) {
            throw new NullPointerException("Given atom container is 'null'");
        }
        if (circularSugar.isEmpty()) {
            return false;
        }
        int[][] adjList = GraphUtil.toAdjList((IAtomContainer)circularSugar);
        RingSearch ringSearch = new RingSearch(circularSugar, adjList);
        List isolatedRingFragments = ringSearch.isolatedRingFragments();
        int size = isolatedRingFragments.size();
        if (size != 1) {
            return false;
        }
        UniversalIsomorphismTester univIsomorphTester = new UniversalIsomorphismTester();
        boolean isolatedRingMatchesEntireInputStructure = false;
        IAtomContainer isolatedRing = (IAtomContainer)isolatedRingFragments.get(0);
        try {
            isolatedRingMatchesEntireInputStructure = univIsomorphTester.isIsomorph(circularSugar, isolatedRing);
        }
        catch (CDKException cdkException) {
            LOGGER.warn((Object)cdkException);
        }
        if (!isolatedRingMatchesEntireInputStructure) {
            return false;
        }
        for (IAtomContainer sugar : this.circularSugarStructuresList) {
            boolean isIsomorphic;
            try {
                isIsomorphic = univIsomorphTester.isIsomorph(sugar, circularSugar);
            }
            catch (CDKException cdkException) {
                LOGGER.warn((Object)cdkException);
                return false;
            }
            if (!isIsomorphic) continue;
            return false;
        }
        try {
            additionWasSuccessful = this.circularSugarStructuresList.add(circularSugar);
        }
        catch (Exception exception) {
            LOGGER.warn((Object)exception);
            additionWasSuccessful = false;
        }
        if (additionWasSuccessful) {
            Comparator comparator = new AtomContainerComparator().reversed();
            this.circularSugarStructuresList.sort(comparator);
        }
        return additionWasSuccessful;
    }

    public boolean addCircularSugarToPatternsList(String smilesCode) {
        IAtomContainer ringSugar;
        if (smilesCode == null) {
            throw new NullPointerException("Given SMILES code is 'null'");
        }
        if (smilesCode.isEmpty()) {
            return false;
        }
        SmilesParser smiPar = new SmilesParser(this.builder);
        try {
            ringSugar = smiPar.parseSmiles(smilesCode);
        }
        catch (InvalidSmilesException exception) {
            LOGGER.warn((Object)exception);
            return false;
        }
        return this.addCircularSugarToPatternsList(ringSugar);
    }

    public boolean addLinearSugarToPatternsList(IAtomContainer linearSugar) {
        boolean additionWasSuccessful;
        boolean isIsomorphic;
        if (linearSugar == null) {
            throw new NullPointerException("Given atom container is 'null'");
        }
        if (linearSugar.isEmpty()) {
            return false;
        }
        UniversalIsomorphismTester univIsomorphTester = new UniversalIsomorphismTester();
        for (IAtomContainer sugar : this.linearSugarStructuresList) {
            try {
                isIsomorphic = univIsomorphTester.isIsomorph(sugar, linearSugar);
            }
            catch (CDKException cdkException) {
                LOGGER.warn((Object)cdkException);
                return false;
            }
            if (!isIsomorphic) continue;
            return false;
        }
        for (IAtomContainer sugar : this.circularSugarStructuresList) {
            try {
                isIsomorphic = univIsomorphTester.isIsomorph(sugar, linearSugar);
            }
            catch (CDKException cdkException) {
                LOGGER.warn((Object)cdkException);
                return false;
            }
            if (!isIsomorphic) continue;
            return false;
        }
        try {
            additionWasSuccessful = this.linearSugarStructuresList.add(linearSugar);
        }
        catch (Exception exception) {
            LOGGER.warn((Object)exception);
            additionWasSuccessful = false;
        }
        if (additionWasSuccessful) {
            this.updateLinearSugarPatterns();
        }
        return additionWasSuccessful;
    }

    public boolean addLinearSugarToPatternsList(String smilesCode) {
        IAtomContainer linearSugar;
        if (smilesCode == null) {
            throw new NullPointerException("Given SMILES code is 'null'");
        }
        if (smilesCode.isEmpty()) {
            return false;
        }
        SmilesParser smiPar = new SmilesParser(this.builder);
        try {
            linearSugar = smiPar.parseSmiles(smilesCode);
        }
        catch (InvalidSmilesException exception) {
            LOGGER.warn((Object)exception);
            return false;
        }
        return this.addLinearSugarToPatternsList(linearSugar);
    }

    public boolean removeCircularSugarFromPatternsList(String smilesCode) {
        IAtomContainer circularSugar;
        if (smilesCode == null) {
            throw new NullPointerException("Given SMILES code is 'null'");
        }
        SmilesParser smiPar = new SmilesParser(this.builder);
        try {
            circularSugar = smiPar.parseSmiles(smilesCode);
        }
        catch (InvalidSmilesException exception) {
            LOGGER.warn((Object)exception);
            return false;
        }
        return this.removeCircularSugarFromPatternsList(circularSugar);
    }

    public boolean removeCircularSugarFromPatternsList(IAtomContainer circularSugar) {
        if (circularSugar == null) {
            throw new NullPointerException("Given atom container is 'null'");
        }
        if (circularSugar.isEmpty()) {
            return false;
        }
        UniversalIsomorphismTester univIsomorphTester = new UniversalIsomorphismTester();
        boolean isIsomorphic = false;
        boolean wasRemovalSuccessful = false;
        for (IAtomContainer sugar : this.circularSugarStructuresList) {
            try {
                isIsomorphic = univIsomorphTester.isIsomorph(sugar, circularSugar);
            }
            catch (CDKException cdkException) {
                LOGGER.warn((Object)cdkException);
                return false;
            }
            if (!isIsomorphic) continue;
            try {
                wasRemovalSuccessful = this.circularSugarStructuresList.remove(sugar);
                break;
            }
            catch (Exception exception) {
                LOGGER.warn((Object)exception);
                return false;
            }
        }
        if (!isIsomorphic) {
            return false;
        }
        if (wasRemovalSuccessful) {
            Comparator comparator = new AtomContainerComparator().reversed();
            this.circularSugarStructuresList.sort(comparator);
        }
        return wasRemovalSuccessful;
    }

    public boolean removeLinearSugarFromPatternsList(String smilesCode) {
        IAtomContainer linearSugar;
        if (smilesCode == null) {
            throw new NullPointerException("Given SMILES code is 'null'");
        }
        if (smilesCode.isEmpty()) {
            return false;
        }
        SmilesParser smiPar = new SmilesParser(this.builder);
        try {
            linearSugar = smiPar.parseSmiles(smilesCode);
        }
        catch (InvalidSmilesException exception) {
            LOGGER.warn((Object)exception);
            return false;
        }
        return this.removeLinearSugarFromPatternsList(linearSugar);
    }

    public boolean removeLinearSugarFromPatternsList(IAtomContainer linearSugar) {
        if (linearSugar == null) {
            throw new NullPointerException("Given atom container is 'null'");
        }
        if (linearSugar.isEmpty()) {
            return false;
        }
        UniversalIsomorphismTester univIsomorphTester = new UniversalIsomorphismTester();
        boolean isIsomorphic = false;
        boolean wasRemovalSuccessful = false;
        for (IAtomContainer sugar : this.linearSugarStructuresList) {
            try {
                isIsomorphic = univIsomorphTester.isIsomorph(sugar, linearSugar);
            }
            catch (CDKException cdkException) {
                LOGGER.warn((Object)cdkException);
                return false;
            }
            if (!isIsomorphic) continue;
            try {
                wasRemovalSuccessful = this.linearSugarStructuresList.remove(sugar);
                break;
            }
            catch (Exception exception) {
                LOGGER.warn((Object)exception);
                return false;
            }
        }
        if (!isIsomorphic) {
            return false;
        }
        if (wasRemovalSuccessful) {
            this.updateLinearSugarPatterns();
        }
        return wasRemovalSuccessful;
    }

    public void clearCircularSugarPatternsList() {
        try {
            this.circularSugarStructuresList.clear();
        }
        catch (UnsupportedOperationException exception) {
            LOGGER.warn((Object)exception);
            this.circularSugarStructuresList = new ArrayList<IAtomContainer>(CIRCULAR_SUGARS_SMILES.length);
        }
    }

    public void clearLinearSugarPatternsList() {
        try {
            if (this.detectLinearAcidicSugarsSetting) {
                this.setDetectLinearAcidicSugarsSetting(false);
            }
            this.linearSugarStructuresList.clear();
            this.linearSugarPatternsList.clear();
        }
        catch (UnsupportedOperationException exception) {
            LOGGER.warn((Object)exception);
            this.detectLinearAcidicSugarsSetting = false;
            this.linearSugarStructuresList = new ArrayList<IAtomContainer>(LINEAR_SUGARS_SMILES.length);
            this.linearSugarPatternsList = new ArrayList<DfPattern>(LINEAR_SUGARS_SMILES.length);
        }
    }

    public void setDetectCircularSugarsOnlyWithOGlycosidicBondSetting(boolean bool) {
        this.detectCircularSugarsOnlyWithOGlycosidicBondSetting = bool;
    }

    public void setRemoveOnlyTerminalSugarsSetting(boolean bool) {
        this.removeOnlyTerminalSugarsSetting = bool;
    }

    public void setPreservationModeSetting(PreservationMode option) {
        if (option == null) {
            throw new NullPointerException("Given mode is 'null'.");
        }
        this.preservationModeSetting = option;
        this.preservationModeThresholdSetting = this.preservationModeSetting.getDefaultThreshold();
    }

    public void setPreservationModeThresholdSetting(int threshold) {
        this.preservationModeThresholdSetting = threshold;
    }

    public void setDetectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting(boolean bool) {
        this.detectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting = bool;
    }

    public boolean setExocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting(double threshold) {
        boolean isNegative;
        boolean isFinite = Double.isFinite(threshold);
        boolean bl = isNegative = threshold < 0.0;
        if (!isFinite || isNegative) {
            return false;
        }
        this.exocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting = threshold;
        return true;
    }

    public void setDetectLinearSugarsInRingsSetting(boolean bool) {
        this.detectLinearSugarsInRingsSetting = bool;
    }

    public void setLinearSugarCandidateMinSizeSetting(int minSize) {
        this.linearSugarCandidateMinSizeSetting = minSize;
    }

    public void setLinearSugarCandidateMaxSizeSetting(int maxSize) {
        this.linearSugarCandidateMaxSizeSetting = maxSize;
    }

    public void setDetectLinearAcidicSugarsSetting(boolean bool) {
        block4: {
            boolean hasSettingChanged = this.detectLinearAcidicSugarsSetting != bool;
            this.detectLinearAcidicSugarsSetting = bool;
            if (!hasSettingChanged) break block4;
            if (this.detectLinearAcidicSugarsSetting) {
                for (IAtomContainer linearAcidicSugar : this.linearAcidicSugarStructuresList) {
                    this.addLinearSugarToPatternsList(linearAcidicSugar);
                }
            } else {
                for (IAtomContainer linearAcidicSugar : this.linearAcidicSugarStructuresList) {
                    this.removeLinearSugarFromPatternsList(linearAcidicSugar);
                }
            }
        }
    }

    public void setDetectSpiroRingsAsCircularSugarsSetting(boolean bool) {
        this.detectSpiroRingsAsCircularSugarsSetting = bool;
    }

    public void setDetectCircularSugarsWithKetoGroupsSetting(boolean bool) {
        this.detectCircularSugarsWithKetoGroupsSetting = bool;
    }

    public void restoreDefaultSettings() {
        this.linearSugarStructuresList = new ArrayList<IAtomContainer>(LINEAR_SUGARS_SMILES.length);
        this.circularSugarStructuresList = new ArrayList<IAtomContainer>(CIRCULAR_SUGARS_SMILES.length);
        this.linearAcidicSugarStructuresList = new ArrayList<IAtomContainer>(LINEAR_ACIDIC_SUGARS_SMILES.length);
        this.linearSugarPatternsList = new ArrayList<DfPattern>(LINEAR_SUGARS_SMILES.length);
        SmilesParser smilesParser = new SmilesParser(this.builder);
        for (String smiles : LINEAR_SUGARS_SMILES) {
            try {
                this.linearSugarStructuresList.add(smilesParser.parseSmiles(smiles));
            }
            catch (InvalidSmilesException exception) {
                LOGGER.warn((Object)("Unable to parse linear sugar pattern SMILES code: " + smiles));
            }
        }
        this.updateLinearSugarPatterns();
        for (String smiles : LINEAR_ACIDIC_SUGARS_SMILES) {
            try {
                this.linearAcidicSugarStructuresList.add(smilesParser.parseSmiles(smiles));
            }
            catch (InvalidSmilesException exception) {
                LOGGER.warn((Object)("Unable to parse linear acidic sugar pattern SMILES code: " + smiles));
            }
        }
        Comparator comparator = new AtomContainerComparator().reversed();
        this.linearAcidicSugarStructuresList.sort(comparator);
        for (String smiles : CIRCULAR_SUGARS_SMILES) {
            try {
                this.circularSugarStructuresList.add(smilesParser.parseSmiles(smiles));
            }
            catch (InvalidSmilesException exception) {
                LOGGER.warn((Object)("Unable to parse circular sugar pattern SMILES code: " + smiles));
            }
        }
        this.circularSugarStructuresList.sort(comparator);
        this.setDetectCircularSugarsOnlyWithOGlycosidicBondSetting(false);
        this.setRemoveOnlyTerminalSugarsSetting(true);
        this.setPreservationModeSetting(PRESERVATION_MODE_DEFAULT);
        this.setPreservationModeThresholdSetting(this.preservationModeSetting.getDefaultThreshold());
        this.setDetectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting(true);
        this.setExocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting(0.5);
        this.setDetectLinearSugarsInRingsSetting(false);
        this.setLinearSugarCandidateMinSizeSetting(4);
        this.setLinearSugarCandidateMaxSizeSetting(7);
        this.setDetectLinearAcidicSugarsSetting(false);
        this.setDetectSpiroRingsAsCircularSugarsSetting(false);
        this.setDetectCircularSugarsWithKetoGroupsSetting(false);
    }

    public boolean hasLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        List<IAtomContainer> sugarCandidates = this.getLinearSugarCandidates(moleculeParam);
        return !sugarCandidates.isEmpty();
    }

    public boolean hasCircularSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        List<IAtomContainer> sugarCandidates = this.getCircularSugarCandidates(moleculeParam);
        return !sugarCandidates.isEmpty();
    }

    public boolean hasCircularOrLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        List<IAtomContainer> circularSugarCandidates = this.getCircularSugarCandidates(moleculeParam);
        boolean containsCircularSugar = !circularSugarCandidates.isEmpty();
        List<IAtomContainer> linearSugarCandidates = this.getLinearSugarCandidates(moleculeParam);
        boolean containsLinearSugar = !linearSugarCandidates.isEmpty();
        return containsCircularSugar || containsLinearSugar;
    }

    public boolean isQualifiedForGlycosidicBondExemption(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        List<IAtomContainer> potentialSugarRings = this.detectPotentialSugarCycles(moleculeParam, this.detectSpiroRingsAsCircularSugarsSetting, this.detectCircularSugarsWithKetoGroupsSetting);
        if (potentialSugarRings.size() != 1) {
            return false;
        }
        IAtomContainer potentialSugarRing = potentialSugarRings.get(0);
        boolean hasGlycosidicBond = this.hasGlycosidicBond(potentialSugarRing, moleculeParam);
        if (hasGlycosidicBond) {
            return false;
        }
        boolean moleculeIsOnlyOneSugarRing = this.isMoleculeEmptyAfterRemovalOfThisRing(potentialSugarRing, moleculeParam);
        return moleculeIsOnlyOneSugarRing;
    }

    public int getNumberOfCircularSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return 0;
        }
        List<IAtomContainer> circularSugarCandidates = this.getCircularSugarCandidates(moleculeParam);
        return circularSugarCandidates.size();
    }

    public int getNumberOfLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return 0;
        }
        List<IAtomContainer> linearSugarCandidates = this.getLinearSugarCandidates(moleculeParam);
        return linearSugarCandidates.size();
    }

    public int getNumberOfCircularAndLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return 0;
        }
        List<IAtomContainer> circularSugarCandidates = this.getCircularSugarCandidates(moleculeParam);
        List<IAtomContainer> linearSugarCandidates = this.getLinearSugarCandidates(moleculeParam);
        return circularSugarCandidates.size() + linearSugarCandidates.size();
    }

    public boolean removeCircularSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        List<IAtomContainer> deglycosylatedMoleculeAndSugarMoietiesList = this.removeAndReturnCircularSugars(moleculeParam);
        return deglycosylatedMoleculeAndSugarMoietiesList.size() > 1;
    }

    public List<IAtomContainer> removeAndReturnCircularSugars(IAtomContainer moleculeParam) {
        List<IAtomContainer> sugarCandidates;
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            ArrayList<IAtomContainer> returnList = new ArrayList<IAtomContainer>(1);
            returnList.add(0, moleculeParam);
            return returnList;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        if (this.preservationModeSetting != PreservationMode.ALL && !ConnectivityChecker.isConnected((IAtomContainer)moleculeParam)) {
            this.flagTooSmallDisconnectedPartsToPreserve(moleculeParam);
        }
        boolean containsSugar = !(sugarCandidates = this.getCircularSugarCandidates(moleculeParam)).isEmpty();
        ArrayList<IAtomContainer> resultList = new ArrayList<IAtomContainer>(sugarCandidates.size() + 1);
        resultList.add(0, moleculeParam);
        if (containsSugar) {
            resultList.addAll(1, this.removeSugarCandidates(moleculeParam, sugarCandidates));
        }
        return resultList;
    }

    public boolean removeLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        List<IAtomContainer> deglycosylatedMoleculeAndSugarMoietiesList = this.removeAndReturnLinearSugars(moleculeParam);
        return deglycosylatedMoleculeAndSugarMoietiesList.size() > 1;
    }

    public List<IAtomContainer> removeAndReturnLinearSugars(IAtomContainer moleculeParam) {
        List<IAtomContainer> sugarCandidates;
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            ArrayList<IAtomContainer> returnList = new ArrayList<IAtomContainer>(1);
            returnList.add(0, moleculeParam);
            return returnList;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        if (this.preservationModeSetting != PreservationMode.ALL && !ConnectivityChecker.isConnected((IAtomContainer)moleculeParam)) {
            this.flagTooSmallDisconnectedPartsToPreserve(moleculeParam);
        }
        boolean containsSugar = !(sugarCandidates = this.getLinearSugarCandidates(moleculeParam)).isEmpty();
        ArrayList<IAtomContainer> resultList = new ArrayList<IAtomContainer>(sugarCandidates.size() + 1);
        resultList.add(0, moleculeParam);
        if (containsSugar) {
            resultList.addAll(1, this.removeSugarCandidates(moleculeParam, sugarCandidates));
        }
        return resultList;
    }

    public boolean removeCircularAndLinearSugars(IAtomContainer moleculeParam) {
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return false;
        }
        List<IAtomContainer> deglycosylatedMoleculeAndSugarMoietiesList = this.removeAndReturnCircularAndLinearSugars(moleculeParam);
        return deglycosylatedMoleculeAndSugarMoietiesList.size() > 1;
    }

    public List<IAtomContainer> removeAndReturnCircularAndLinearSugars(IAtomContainer moleculeParam) {
        List<Object> removedLinearSugarMoieties;
        List<Object> removedCircularSugarMoieties;
        boolean wasSthRemoved;
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            ArrayList<IAtomContainer> returnList = new ArrayList<IAtomContainer>(1);
            returnList.add(0, moleculeParam);
            return returnList;
        }
        this.addUniqueIndicesToAtoms(moleculeParam);
        if (this.preservationModeSetting != PreservationMode.ALL && !ConnectivityChecker.isConnected((IAtomContainer)moleculeParam)) {
            this.flagTooSmallDisconnectedPartsToPreserve(moleculeParam);
        }
        ArrayList<IAtomContainer> resultList = new ArrayList<IAtomContainer>(moleculeParam.getAtomCount() / 6);
        resultList.add(0, moleculeParam);
        do {
            List<IAtomContainer> circularSugarCandidates;
            boolean isCandidateListNotEmpty = !(circularSugarCandidates = this.getCircularSugarCandidates(moleculeParam)).isEmpty();
            removedCircularSugarMoieties = new ArrayList(0);
            if (isCandidateListNotEmpty) {
                removedCircularSugarMoieties = this.removeSugarCandidates(moleculeParam, circularSugarCandidates);
                resultList.addAll(removedCircularSugarMoieties);
            }
            if (moleculeParam.isEmpty()) break;
            List<IAtomContainer> linearSugarCandidates = this.getLinearSugarCandidates(moleculeParam);
            isCandidateListNotEmpty = !linearSugarCandidates.isEmpty();
            removedLinearSugarMoieties = new ArrayList(0);
            if (!isCandidateListNotEmpty) continue;
            removedLinearSugarMoieties = this.removeSugarCandidates(moleculeParam, linearSugarCandidates);
            resultList.addAll(removedLinearSugarMoieties);
        } while (!moleculeParam.isEmpty() && this.removeOnlyTerminalSugarsSetting && (wasSthRemoved = !removedCircularSugarMoieties.isEmpty() || !removedLinearSugarMoieties.isEmpty()));
        return resultList;
    }

    public List<IAtomContainer> getCircularSugarCandidates(IAtomContainer moleculeParam) {
        List<IAtomContainer> potentialSugarRings;
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        boolean areIndicesSet = this.checkUniqueIndicesOfAtoms(moleculeParam);
        if (!areIndicesSet) {
            this.addUniqueIndicesToAtoms(moleculeParam);
        }
        if ((potentialSugarRings = this.detectPotentialSugarCycles(moleculeParam, this.detectSpiroRingsAsCircularSugarsSetting, this.detectCircularSugarsWithKetoGroupsSetting)).isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        ArrayList<IAtomContainer> sugarCandidates = new ArrayList<IAtomContainer>(potentialSugarRings.size());
        for (IAtomContainer potentialSugarRing : potentialSugarRings) {
            boolean moleculeIsOnlyOneSugarRing;
            boolean hasGlycosidicBond;
            if (potentialSugarRing == null || potentialSugarRing.isEmpty() || this.detectCircularSugarsOnlyWithOGlycosidicBondSetting && !(hasGlycosidicBond = this.hasGlycosidicBond(potentialSugarRing, moleculeParam)) && (potentialSugarRings.size() != 1 || !(moleculeIsOnlyOneSugarRing = this.isMoleculeEmptyAfterRemovalOfThisRing(potentialSugarRing, moleculeParam)))) continue;
            if (this.detectCircularSugarsOnlyWithEnoughExocyclicOxygenAtomsSetting) {
                int exocyclicOxygenCount = this.getExocyclicOxygenAtomCount(potentialSugarRing, moleculeParam);
                int atomsInRingCount = potentialSugarRing.getAtomCount();
                boolean areEnoughOxygensAttached = this.doesRingHaveEnoughExocyclicOxygenAtoms(atomsInRingCount, exocyclicOxygenCount);
                if (!areEnoughOxygensAttached) continue;
            }
            sugarCandidates.add(potentialSugarRing);
        }
        return sugarCandidates;
    }

    public List<IAtomContainer> getLinearSugarCandidates(IAtomContainer moleculeParam) {
        List<IAtomContainer> sugarCandidates;
        if (moleculeParam == null) {
            throw new NullPointerException("Given molecule is 'null'.");
        }
        if (moleculeParam.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        boolean areIndicesSet = this.checkUniqueIndicesOfAtoms(moleculeParam);
        if (!areIndicesSet) {
            this.addUniqueIndicesToAtoms(moleculeParam);
        }
        if (!(sugarCandidates = this.detectLinearSugarCandidatesByPatternMatching(moleculeParam)).isEmpty()) {
            sugarCandidates = this.combineOverlappingCandidates(sugarCandidates);
            sugarCandidates = this.splitEtherEsterAndPeroxideBonds(sugarCandidates);
            this.removeAtomsOfCircularSugarsFromCandidates(sugarCandidates, moleculeParam);
        }
        if (!this.detectLinearSugarsInRingsSetting && !sugarCandidates.isEmpty()) {
            this.removeCyclicAtomsFromSugarCandidates(sugarCandidates, moleculeParam);
        }
        if (!sugarCandidates.isEmpty()) {
            sugarCandidates = this.removeTooSmallAndTooLargeCandidates(sugarCandidates);
        }
        return sugarCandidates;
    }

    protected void flagTooSmallDisconnectedPartsToPreserve(IAtomContainer moleculeParam) {
        boolean areIndicesSet = this.checkUniqueIndicesOfAtoms(moleculeParam);
        if (!areIndicesSet) {
            this.addUniqueIndicesToAtoms(moleculeParam);
        }
        float loadFactor = 0.75f;
        int indexToAtomMapInitCapacity = (int)((float)moleculeParam.getAtomCount() * (1.0f / loadFactor) + 2.0f);
        HashMap<Integer, IAtom> indexToAtomMap = new HashMap<Integer, IAtom>(indexToAtomMapInitCapacity, loadFactor);
        for (IAtom atom : moleculeParam.atoms()) {
            indexToAtomMap.put((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY), atom);
        }
        IAtomContainerSet unconnectedParts = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)moleculeParam);
        for (IAtomContainer part : unconnectedParts) {
            if (!this.isTooSmallToPreserve(part)) continue;
            for (IAtom atom : part.atoms()) {
                ((IAtom)indexToAtomMap.get((int)((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY)))).setProperty((Object)TOO_SMALL_DISCONNECTED_PART_TO_PRESERVE_PROPERTY_KEY, (Object)true);
            }
        }
    }

    protected void removeTooSmallDisconnectedStructures(IAtomContainer moleculeParam) {
        if (moleculeParam.isEmpty()) {
            return;
        }
        if (this.preservationModeSetting == PreservationMode.ALL) {
            return;
        }
        IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)moleculeParam);
        for (int i = 0; i < components.getAtomContainerCount(); ++i) {
            IAtomContainer component = components.getAtomContainer(i);
            boolean isTooSmall = this.isTooSmallToPreserve(component);
            if (!isTooSmall) continue;
            for (IAtom atom : component.atoms()) {
                if (!moleculeParam.contains(atom)) continue;
                moleculeParam.removeAtom(atom);
            }
        }
    }

    protected boolean isTooSmallToPreserve(IAtomContainer moleculeParam) {
        boolean isTooSmall;
        if (moleculeParam.isEmpty()) {
            return true;
        }
        for (IAtom atom : moleculeParam.atoms()) {
            if (atom.getProperty((Object)TOO_SMALL_DISCONNECTED_PART_TO_PRESERVE_PROPERTY_KEY) == null || !((Boolean)atom.getProperty((Object)TOO_SMALL_DISCONNECTED_PART_TO_PRESERVE_PROPERTY_KEY)).booleanValue()) continue;
            return false;
        }
        if (this.preservationModeSetting == PreservationMode.ALL) {
            isTooSmall = false;
        } else if (this.preservationModeSetting == PreservationMode.HEAVY_ATOM_COUNT) {
            int heavyAtomCount = AtomContainerManipulator.getHeavyAtoms((IAtomContainer)moleculeParam).size();
            isTooSmall = heavyAtomCount < this.preservationModeThresholdSetting;
        } else if (this.preservationModeSetting == PreservationMode.MOLECULAR_WEIGHT) {
            double molWeight = AtomContainerManipulator.getMass((IAtomContainer)moleculeParam, (int)1);
            isTooSmall = molWeight < (double)this.preservationModeThresholdSetting;
        } else {
            throw new UnsupportedOperationException("Undefined PreservationMode setting!");
        }
        return isTooSmall;
    }

    protected boolean isTerminal(IAtomContainer substructure, IAtomContainer parentMolecule, List<IAtomContainer> candidateList) {
        boolean isTerminal;
        if (!this.checkUniqueIndicesOfAtoms(parentMolecule)) {
            this.addUniqueIndicesToAtoms(parentMolecule);
        }
        if (!this.checkUniqueIndicesOfAtoms(substructure)) {
            this.addUniqueIndicesToAtoms(substructure);
        }
        for (IAtomContainer candidate : candidateList) {
            boolean areIndicesSet = this.checkUniqueIndicesOfAtoms(candidate);
            if (areIndicesSet) continue;
            this.addUniqueIndicesToAtoms(candidate);
        }
        Object moleculeCopy = this.copy(parentMolecule);
        if (!ConnectivityChecker.isConnected((IAtomContainer)moleculeCopy)) {
            IAtomContainerSet unconnectedParts = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)moleculeCopy);
            boolean breakOuterLoop = false;
            for (Object part : unconnectedParts) {
                HashSet<Integer> partAtomIndices = new HashSet<Integer>((int)((float)part.getAtomCount() * 1.3333334f + 2.0f), 0.75f);
                for (IAtom partAtom : part.atoms()) {
                    partAtomIndices.add((Integer)partAtom.getProperty((Object)INDEX_PROPERTY_KEY));
                }
                for (IAtom atom : substructure.atoms()) {
                    if (!partAtomIndices.contains((int)((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY)))) continue;
                    moleculeCopy = part;
                    breakOuterLoop = true;
                    break;
                }
                if (!breakOuterLoop) continue;
                break;
            }
        }
        float loadFactor = 0.75f;
        int indexToAtomMapInitCapacity = (int)((float)moleculeCopy.getAtomCount() * (1.0f / loadFactor) + 2.0f);
        HashMap<Integer, IAtom> indexToAtomMap = new HashMap<Integer, IAtom>(indexToAtomMapInitCapacity, loadFactor);
        for (IAtom atom : moleculeCopy.atoms()) {
            indexToAtomMap.put((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY), atom);
        }
        for (IAtom atom : substructure.atoms()) {
            moleculeCopy.removeAtom((IAtom)indexToAtomMap.get((int)((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY))));
        }
        boolean isConnected = ConnectivityChecker.isConnected((IAtomContainer)moleculeCopy);
        if (this.preservationModeSetting == PreservationMode.ALL) {
            isTerminal = isConnected;
        } else if (isConnected) {
            isTerminal = true;
        } else {
            IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)moleculeCopy);
            int atomIndicesThatArePartOfSugarCandidatesSetInitCapacity = (int)((float)parentMolecule.getAtomCount() * (1.0f / loadFactor) + 2.0f);
            HashSet<Integer> atomIndicesThatArePartOfSugarCandidatesSet = new HashSet<Integer>(atomIndicesThatArePartOfSugarCandidatesSetInitCapacity, loadFactor);
            for (IAtomContainer candidate : candidateList) {
                for (IAtom atom : candidate.atoms()) {
                    int index = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
                    atomIndicesThatArePartOfSugarCandidatesSet.add(index);
                }
            }
            for (IAtomContainer component : components.atomContainers()) {
                if (component == null || component.isEmpty()) continue;
                boolean isTooSmall = this.isTooSmallToPreserve(component);
                boolean isPartOfSugarCandidate = false;
                for (IAtom atom : component.atoms()) {
                    int index = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
                    if (!atomIndicesThatArePartOfSugarCandidatesSet.contains(index)) continue;
                    isPartOfSugarCandidate = true;
                    break;
                }
                if (!isTooSmall || isPartOfSugarCandidate) continue;
                moleculeCopy.remove(component);
            }
            isTerminal = ConnectivityChecker.isConnected((IAtomContainer)moleculeCopy);
        }
        return isTerminal;
    }

    protected List<IAtomContainer> removeSugarCandidates(IAtomContainer moleculeParam, List<IAtomContainer> candidateList) {
        if (candidateList.isEmpty() || moleculeParam.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        ArrayList<IAtomContainer> sugarCandidates = new ArrayList<IAtomContainer>(candidateList);
        ArrayList<IAtomContainer> removedSugarMoieties = new ArrayList<IAtomContainer>(candidateList.size());
        if (this.removeOnlyTerminalSugarsSetting) {
            boolean containsNoTerminalSugar = false;
            while (!containsNoTerminalSugar) {
                boolean wasSthRemoved = false;
                for (int i = 0; i < sugarCandidates.size(); ++i) {
                    IAtomContainer candidate = (IAtomContainer)sugarCandidates.get(i);
                    if (candidate == null || candidate.isEmpty() || !this.isTerminal(candidate, moleculeParam, sugarCandidates)) continue;
                    for (IAtom atom : candidate.atoms()) {
                        if (!moleculeParam.contains(atom)) continue;
                        Boolean atomIsSpiroAtom = (Boolean)atom.getProperty((Object)IS_SPIRO_ATOM_PROPERTY_KEY);
                        if (atomIsSpiroAtom != null && atomIsSpiroAtom.booleanValue()) {
                            atom.setProperty((Object)IS_SPIRO_ATOM_PROPERTY_KEY, (Object)false);
                            continue;
                        }
                        moleculeParam.removeAtom(atom);
                    }
                    removedSugarMoieties.add(candidate);
                    sugarCandidates.remove(i);
                    --i;
                    if (!moleculeParam.isEmpty() && this.preservationModeSetting != PreservationMode.ALL) {
                        this.removeTooSmallDisconnectedStructures(moleculeParam);
                    }
                    if (moleculeParam.isEmpty()) {
                        containsNoTerminalSugar = true;
                        break;
                    }
                    wasSthRemoved = true;
                }
                if (wasSthRemoved) continue;
                containsNoTerminalSugar = true;
            }
        } else {
            for (IAtomContainer sugarCandidate : sugarCandidates) {
                for (IAtom atom : sugarCandidate.atoms()) {
                    if (!moleculeParam.contains(atom)) continue;
                    Boolean atomIsSpiroAtom = (Boolean)atom.getProperty((Object)IS_SPIRO_ATOM_PROPERTY_KEY);
                    if (atomIsSpiroAtom != null && atomIsSpiroAtom.booleanValue()) {
                        atom.setProperty((Object)IS_SPIRO_ATOM_PROPERTY_KEY, (Object)false);
                        continue;
                    }
                    moleculeParam.removeAtom(atom);
                }
                removedSugarMoieties.add(sugarCandidate);
            }
        }
        if (!moleculeParam.isEmpty() && this.preservationModeSetting != PreservationMode.ALL) {
            this.removeTooSmallDisconnectedStructures(moleculeParam);
        }
        return removedSugarMoieties;
    }

    protected IAtomContainer copy(IAtomContainer molecule) {
        IAtomContainer copy = (IAtomContainer)molecule.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        if (molecule.isEmpty()) {
            return copy;
        }
        float loadFactor = 0.75f;
        int mapInitCapacity = (int)((float)molecule.getAtomCount() * (1.0f / loadFactor) + 2.0f);
        HashMap<Integer, IAtom> indexToCopyAtomMap = new HashMap<Integer, IAtom>(mapInitCapacity, loadFactor);
        for (IAtom atom : molecule.atoms()) {
            IAtom cpyAtom = copy.newAtom(atom.getAtomicNumber().intValue(), atom.getImplicitHydrogenCount().intValue());
            cpyAtom.setProperty((Object)INDEX_PROPERTY_KEY, atom.getProperty((Object)INDEX_PROPERTY_KEY));
            indexToCopyAtomMap.put((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY), cpyAtom);
        }
        for (IBond bond : molecule.bonds()) {
            IAtom beg = (IAtom)indexToCopyAtomMap.get((int)((Integer)bond.getBegin().getProperty((Object)INDEX_PROPERTY_KEY)));
            IAtom end = (IAtom)indexToCopyAtomMap.get((int)((Integer)bond.getEnd().getProperty((Object)INDEX_PROPERTY_KEY)));
            if (beg == null || end == null) continue;
            beg.getContainer().newBond(beg, end, bond.getOrder());
        }
        return copy;
    }

    protected void addUniqueIndicesToAtoms(IAtomContainer moleculeParam) {
        if (moleculeParam.isEmpty()) {
            return;
        }
        for (int i = 0; i < moleculeParam.getAtomCount(); ++i) {
            IAtom atom = moleculeParam.getAtom(i);
            atom.setProperty((Object)INDEX_PROPERTY_KEY, (Object)i);
        }
    }

    protected String generateSubstructureIdentifier(IAtomContainer substructure) {
        int atomIndex;
        if (substructure.isEmpty()) {
            return "";
        }
        ArrayList<Integer> indicesList = new ArrayList<Integer>(substructure.getAtomCount());
        for (IAtom atom : substructure.atoms()) {
            atomIndex = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
            indicesList.add(atomIndex);
        }
        Collections.sort(indicesList);
        String substructureIdentifier = "";
        Iterator iterator = indicesList.iterator();
        while (iterator.hasNext()) {
            atomIndex = (Integer)iterator.next();
            substructureIdentifier = substructureIdentifier.concat(Integer.toString(atomIndex)).concat(":");
        }
        return substructureIdentifier;
    }

    protected boolean checkUniqueIndicesOfAtoms(IAtomContainer moleculeParam) {
        if (moleculeParam.isEmpty()) {
            return true;
        }
        float loadFactor = 0.75f;
        int atomIndicesSetInitCapacity = (int)((float)moleculeParam.getAtomCount() * (1.0f / loadFactor) + 2.0f);
        HashSet<Integer> atomIndicesSet = new HashSet<Integer>(atomIndicesSetInitCapacity, loadFactor);
        for (IAtom atom : moleculeParam.atoms()) {
            if (atom.getProperty((Object)INDEX_PROPERTY_KEY) == null) {
                return false;
            }
            int index = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
            if (atomIndicesSet.contains(index)) {
                return false;
            }
            atomIndicesSet.add(index);
        }
        return true;
    }

    protected List<IAtomContainer> detectPotentialSugarCycles(IAtomContainer moleculeParam, boolean includeSpiroRings, boolean ignoreKetoGroups) {
        int[][] adjList;
        RingSearch ringSearch;
        List isolatedRings;
        if (moleculeParam.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        boolean areIndicesSet = this.checkUniqueIndicesOfAtoms(moleculeParam);
        if (!areIndicesSet) {
            this.addUniqueIndicesToAtoms(moleculeParam);
        }
        if ((isolatedRings = (ringSearch = new RingSearch(moleculeParam, adjList = GraphUtil.toAdjList((IAtomContainer)moleculeParam))).isolatedRingFragments()).isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        List ringFragments = ringSearch.isolatedRingFragments();
        ringFragments.addAll(ringSearch.fusedRingFragments());
        float loadFactor = 0.75f;
        int ringIdentifierToIsFusedOrSpiroMapInitCapacity = (int)((float)ringFragments.size() * (1.0f / loadFactor) + 2.0f);
        HashMap<String, Boolean> ringIdentifierToIsFusedOrSpiroMap = new HashMap<String, Boolean>(ringIdentifierToIsFusedOrSpiroMapInitCapacity, loadFactor);
        int atomIDToRingIDMapInitCapacity = 6 * (int)((float)ringFragments.size() * (1.0f / loadFactor) + 2.0f);
        HashMap<Integer, Object> atomIDToRingIDMap = new HashMap<Integer, Object>(atomIDToRingIDMapInitCapacity, loadFactor);
        for (IAtomContainer ring : ringFragments) {
            String ringID = this.generateSubstructureIdentifier(ring);
            ringIdentifierToIsFusedOrSpiroMap.put(ringID, false);
            for (IAtom atom : ring.atoms()) {
                int atomID = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
                if (!atomIDToRingIDMap.containsKey(atomID)) {
                    int ringIDSetInitCapacity = (int)(5.0f * (1.0f / loadFactor) + 2.0f);
                    HashSet ringIDSet = new HashSet(ringIDSetInitCapacity, loadFactor);
                    ringIDSet.add(ringID);
                    atomIDToRingIDMap.put(atomID, ringIDSet);
                    continue;
                }
                ringIdentifierToIsFusedOrSpiroMap.put(ringID, true);
                Set ringIDSet = (Set)atomIDToRingIDMap.get(atomID);
                for (String alreadyVisitedRingID : ringIDSet) {
                    ringIdentifierToIsFusedOrSpiroMap.put(alreadyVisitedRingID, true);
                }
                ringIDSet.add(ringID);
            }
        }
        ArrayList<IAtomContainer> sugarCandidates = new ArrayList<IAtomContainer>(isolatedRings.size());
        block5: for (IAtomContainer isolatedRing : isolatedRings) {
            String ringID;
            if (isolatedRing == null || isolatedRing.isEmpty() || !includeSpiroRings && Boolean.TRUE.equals(ringIdentifierToIsFusedOrSpiroMap.get(ringID = this.generateSubstructureIdentifier(isolatedRing)))) continue;
            for (IAtomContainer referenceRing : this.circularSugarStructuresList) {
                boolean isIsomorphic;
                UniversalIsomorphismTester univIsoTester = new UniversalIsomorphismTester();
                try {
                    isIsomorphic = univIsoTester.isIsomorph(referenceRing, isolatedRing);
                }
                catch (CDKException cdkException) {
                    LOGGER.warn((Object)cdkException);
                    continue;
                }
                if (!isIsomorphic) continue;
                boolean areAllExocyclicBondsSingle = this.areAllExocyclicBondsSingle(isolatedRing, moleculeParam, ignoreKetoGroups);
                if (!areAllExocyclicBondsSingle) continue block5;
                for (IAtom atom : isolatedRing.atoms()) {
                    int atomID = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
                    Set ringIDSet = (Set)atomIDToRingIDMap.get(atomID);
                    int size = ringIDSet.size();
                    if (size <= 1) continue;
                    atom.setProperty((Object)IS_SPIRO_ATOM_PROPERTY_KEY, (Object)true);
                }
                sugarCandidates.add(isolatedRing);
                continue block5;
            }
        }
        return sugarCandidates;
    }

    protected boolean areAllExocyclicBondsSingle(IAtomContainer ringToTest, IAtomContainer originalMolecule, boolean ignoreKetoGroups) {
        if (ringToTest.isEmpty() || originalMolecule.isEmpty()) {
            return true;
        }
        int atomCountInRing = ringToTest.getAtomCount();
        int arrayListInitCapacity = atomCountInRing * 2;
        ArrayList<IBond> exocyclicBondsList = new ArrayList<IBond>(arrayListInitCapacity);
        Iterable ringAtoms = ringToTest.atoms();
        for (IAtom ringAtom : ringAtoms) {
            if (!originalMolecule.contains(ringAtom)) continue;
            List connectedBondsList = originalMolecule.getConnectedBondsList(ringAtom);
            for (IBond bond : connectedBondsList) {
                boolean isInRing = ringToTest.contains(bond);
                if (isInRing) continue;
                exocyclicBondsList.add(bond);
            }
        }
        if (ignoreKetoGroups) {
            for (IBond bond : exocyclicBondsList) {
                IBond.Order order = bond.getOrder();
                if (order == IBond.Order.SINGLE) continue;
                if (order == IBond.Order.DOUBLE) {
                    boolean containsOxygen = false;
                    for (IAtom atom : bond.atoms()) {
                        if (atom.getAtomicNumber() != 8) continue;
                        containsOxygen = true;
                    }
                    if (containsOxygen) continue;
                    return false;
                }
                return false;
            }
            return true;
        }
        return BondManipulator.getMaximumBondOrder(exocyclicBondsList) == IBond.Order.SINGLE;
    }

    protected boolean hasGlycosidicBond(IAtomContainer ringToTest, IAtomContainer originalMolecule) {
        if (ringToTest.isEmpty() || originalMolecule.isEmpty()) {
            return false;
        }
        Iterable ringAtoms = ringToTest.atoms();
        boolean containsGlycosidicBond = false;
        for (IAtom ringAtom : ringAtoms) {
            boolean breakOuterLoop = false;
            if (!originalMolecule.contains(ringAtom)) continue;
            List connectedAtomsList = originalMolecule.getConnectedAtomsList(ringAtom);
            for (IAtom atom : connectedAtomsList) {
                boolean isOxygen;
                boolean isInRing = ringToTest.contains(atom);
                if (isInRing || !(isOxygen = atom.getAtomicNumber() == 8)) continue;
                List connectedBondsList = originalMolecule.getConnectedBondsList(atom);
                boolean hasOnlyTwoBonds = connectedBondsList.size() == 2;
                boolean areAllBondsSingle = BondManipulator.getMaximumBondOrder((List)connectedBondsList) == IBond.Order.SINGLE;
                boolean isOneBondAtomHydrogen = false;
                for (IBond bond : connectedBondsList) {
                    for (IAtom bondAtom : bond.atoms()) {
                        if (bondAtom.getAtomicNumber() != 1) continue;
                        isOneBondAtomHydrogen = true;
                    }
                }
                if (!hasOnlyTwoBonds || !areAllBondsSingle || isOneBondAtomHydrogen) continue;
                containsGlycosidicBond = true;
                breakOuterLoop = true;
                break;
            }
            if (!breakOuterLoop) continue;
            break;
        }
        return containsGlycosidicBond;
    }

    protected boolean isMoleculeEmptyAfterRemovalOfThisRing(IAtomContainer ringParam, IAtomContainer moleculeParam) {
        boolean isMoleculeEmptyAfterRemoval;
        if (!this.checkUniqueIndicesOfAtoms(moleculeParam)) {
            this.addUniqueIndicesToAtoms(moleculeParam);
        }
        IAtomContainer moleculeCopy = this.copy(moleculeParam);
        float loadFactor = 0.75f;
        int indexToAtomMapInitCapacity = (int)((float)moleculeCopy.getAtomCount() * (1.0f / loadFactor) + 2.0f);
        HashMap<Integer, IAtom> indexToAtomMap = new HashMap<Integer, IAtom>(indexToAtomMapInitCapacity, loadFactor);
        for (IAtom atom : moleculeCopy.atoms()) {
            indexToAtomMap.put((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY), atom);
        }
        for (IAtom atom : ringParam.atoms()) {
            moleculeCopy.removeAtom((IAtom)indexToAtomMap.get((int)((Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY))));
        }
        if (moleculeCopy.isEmpty()) {
            isMoleculeEmptyAfterRemoval = true;
        } else {
            this.removeTooSmallDisconnectedStructures(moleculeCopy);
            isMoleculeEmptyAfterRemoval = moleculeCopy.isEmpty();
        }
        return isMoleculeEmptyAfterRemoval;
    }

    protected int getExocyclicOxygenAtomCount(IAtomContainer ringToTest, IAtomContainer originalMolecule) {
        int exocyclicOxygenCounter = 0;
        Iterable ringAtoms = ringToTest.atoms();
        for (IAtom ringAtom : ringAtoms) {
            if (!originalMolecule.contains(ringAtom)) continue;
            List connectedAtomsList = originalMolecule.getConnectedAtomsList(ringAtom);
            for (IAtom connectedAtom : connectedAtomsList) {
                boolean isOxygen = connectedAtom.getAtomicNumber() == 8;
                boolean isInRing = ringToTest.contains(connectedAtom);
                if (!isOxygen || isInRing) continue;
                ++exocyclicOxygenCounter;
            }
        }
        return exocyclicOxygenCounter;
    }

    protected boolean doesRingHaveEnoughExocyclicOxygenAtoms(int numberOfAtomsInRing, int numberOfAttachedExocyclicOxygenAtoms) {
        if (numberOfAtomsInRing == 0) {
            return false;
        }
        double attachedOxygensToAtomsInRingRatio = (double)numberOfAttachedExocyclicOxygenAtoms / (double)numberOfAtomsInRing;
        return attachedOxygensToAtomsInRingRatio >= this.exocyclicOxygenAtomsToAtomsInRingRatioThresholdSetting;
    }

    protected void updateLinearSugarPatterns() {
        Comparator comparator = new AtomContainerComparator().reversed();
        this.linearSugarStructuresList.sort(comparator);
        for (IAtomContainer sugarAC : this.linearSugarStructuresList) {
            try {
                this.linearSugarPatternsList.add(DfPattern.findSubstructure((IAtomContainer)sugarAC));
            }
            catch (Exception exception) {
                LOGGER.warn((Object)exception);
            }
        }
    }

    protected List<IAtomContainer> detectLinearSugarCandidatesByPatternMatching(IAtomContainer moleculeParam) {
        if (moleculeParam.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        ArrayList<IAtomContainer> sugarCandidates = new ArrayList<IAtomContainer>(moleculeParam.getAtomCount() / 2);
        int listSize = this.linearSugarPatternsList.size();
        ArrayList<DfPattern> listToIterate = new ArrayList<DfPattern>(listSize);
        listToIterate.addAll(0, this.linearSugarPatternsList);
        for (DfPattern linearSugarPattern : listToIterate) {
            if (linearSugarPattern == null) continue;
            Mappings mappings = linearSugarPattern.matchAll(moleculeParam);
            Mappings uniqueMappings = mappings.uniqueAtoms();
            Iterable uniqueSubstructureMappings = uniqueMappings.toSubstructures();
            for (IAtomContainer matchedStructure : uniqueSubstructureMappings) {
                if (matchedStructure == null) continue;
                sugarCandidates.add(matchedStructure);
            }
        }
        return sugarCandidates;
    }

    protected List<IAtomContainer> combineOverlappingCandidates(List<IAtomContainer> candidateList) {
        if (candidateList == null || candidateList.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        int listSize = candidateList.size();
        ArrayList<IAtomContainer> nonOverlappingSugarCandidates = new ArrayList<IAtomContainer>(listSize);
        IAtomContainer matchesContainer = (IAtomContainer)candidateList.get(0).getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        for (IAtomContainer candidate : candidateList) {
            if (candidate == null) continue;
            matchesContainer.add(candidate);
        }
        if (ConnectivityChecker.isConnected((IAtomContainer)matchesContainer)) {
            nonOverlappingSugarCandidates.add(matchesContainer);
        } else {
            IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)matchesContainer);
            Iterable molecules = components.atomContainers();
            for (IAtomContainer component : molecules) {
                nonOverlappingSugarCandidates.add(component);
            }
        }
        return nonOverlappingSugarCandidates;
    }

    protected List<IAtomContainer> splitEtherEsterAndPeroxideBonds(List<IAtomContainer> candidateList) {
        if (candidateList == null || candidateList.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        ArrayList<IAtomContainer> processedCandidates = new ArrayList<IAtomContainer>(candidateList.size() * 2);
        for (IAtomContainer candidate : candidateList) {
            boolean isConnected;
            Mappings peroxideMappings;
            Mappings etherMappings;
            if (candidate == null) continue;
            SmartsPattern.prepare((IAtomContainer)candidate);
            Mappings esterMappings = ESTER_SMARTS_PATTERN.matchAll(candidate).uniqueAtoms();
            if (esterMappings.atLeast(1)) {
                for (Object esterGroup : esterMappings.toSubstructures()) {
                    IAtom doubleBondedOxygen = null;
                    IAtom connectingOxygen = null;
                    for (IAtom atom : esterGroup.atoms()) {
                        if (atom.getAtomicNumber() != 8) continue;
                        int bondCount = atom.getBondCount();
                        if (bondCount == 1) {
                            doubleBondedOxygen = atom;
                            continue;
                        }
                        connectingOxygen = atom;
                    }
                    IAtom carbonBoundToDoubleBondedOxygen = (IAtom)esterGroup.getConnectedAtomsList(doubleBondedOxygen).get(0);
                    candidate.removeBond(carbonBoundToDoubleBondedOxygen, connectingOxygen);
                }
            }
            if ((etherMappings = ETHER_SMARTS_PATTERN.matchAll(candidate).uniqueAtoms()).atLeast(1)) {
                Object esterGroup;
                esterGroup = etherMappings.toSubstructures().iterator();
                while (esterGroup.hasNext()) {
                    IAtomContainer etherGroup = (IAtomContainer)esterGroup.next();
                    IAtom carbon1 = null;
                    IAtom carbon2 = null;
                    IAtom oxygen = null;
                    for (IAtom atom : etherGroup.atoms()) {
                        if (atom.getAtomicNumber() == 8) {
                            oxygen = atom;
                            continue;
                        }
                        if (atom.getAtomicNumber() == 6 && carbon1 == null) {
                            carbon1 = atom;
                            continue;
                        }
                        carbon2 = atom;
                    }
                    candidate.removeBond(oxygen, carbon2);
                }
            }
            if ((peroxideMappings = PEROXIDE_SMARTS_PATTERN.matchAll(candidate).uniqueAtoms()).atLeast(1)) {
                for (IAtomContainer peroxideGroup : peroxideMappings.toSubstructures()) {
                    IAtom oxygen1 = null;
                    IAtom oxygen2 = null;
                    for (IAtom atom : peroxideGroup.atoms()) {
                        if (atom.getAtomicNumber() != 8) continue;
                        if (oxygen1 == null) {
                            oxygen1 = atom;
                            continue;
                        }
                        oxygen2 = atom;
                    }
                    candidate.removeBond(oxygen1, oxygen2);
                }
            }
            if (isConnected = ConnectivityChecker.isConnected((IAtomContainer)candidate)) {
                processedCandidates.add(candidate);
                continue;
            }
            IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)candidate);
            for (IAtomContainer component : components.atomContainers()) {
                processedCandidates.add(component);
            }
        }
        return processedCandidates;
    }

    protected void removeAtomsOfCircularSugarsFromCandidates(List<IAtomContainer> candidateList, IAtomContainer parentMolecule) {
        List<IAtomContainer> potentialSugarRingsParent;
        if (candidateList == null || candidateList.isEmpty()) {
            return;
        }
        if (!this.checkUniqueIndicesOfAtoms(parentMolecule)) {
            this.addUniqueIndicesToAtoms(parentMolecule);
        }
        if ((potentialSugarRingsParent = this.detectPotentialSugarCycles(parentMolecule, true, true)).isEmpty()) {
            return;
        }
        float loadFactor = 0.75f;
        int circularSugarAtomIDSetInitCapacity = (int)(7.0f * (float)potentialSugarRingsParent.size() * (1.0f / loadFactor) + 2.0f);
        HashSet<Integer> circularSugarAtomIDSet = new HashSet<Integer>(circularSugarAtomIDSetInitCapacity, loadFactor);
        for (IAtomContainer circularSugarCandidate : potentialSugarRingsParent) {
            for (IAtom atom : circularSugarCandidate.atoms()) {
                int atomIndex = (Integer)atom.getProperty((Object)INDEX_PROPERTY_KEY);
                circularSugarAtomIDSet.add(atomIndex);
            }
        }
        for (int i = 0; i < candidateList.size(); ++i) {
            IAtomContainer candidate = candidateList.get(i);
            if (candidate == null) {
                candidateList.remove(i);
                --i;
                continue;
            }
            if (!this.checkUniqueIndicesOfAtoms(candidate)) {
                this.addUniqueIndicesToAtoms(candidate);
            }
            for (IAtom candidateAtom : candidate.atoms()) {
                int atomIndex = (Integer)candidateAtom.getProperty((Object)INDEX_PROPERTY_KEY);
                if (!circularSugarAtomIDSet.contains(atomIndex) || !candidate.contains(candidateAtom)) continue;
                candidate.removeAtom(candidateAtom);
            }
            if (candidate.isEmpty()) {
                candidateList.remove(i);
                --i;
                continue;
            }
            boolean isConnected = ConnectivityChecker.isConnected((IAtomContainer)candidate);
            if (isConnected) continue;
            IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)candidate);
            for (IAtomContainer component : components.atomContainers()) {
                candidateList.add(component);
            }
            candidateList.remove(i);
            --i;
        }
    }

    protected void removeCyclicAtomsFromSugarCandidates(List<IAtomContainer> candidateList, IAtomContainer moleculeParam) {
        boolean moleculeHasRings;
        if (candidateList == null || candidateList.isEmpty()) {
            return;
        }
        int[][] adjList = GraphUtil.toAdjList((IAtomContainer)moleculeParam);
        RingSearch ringSearch = new RingSearch(moleculeParam, adjList);
        boolean bl = moleculeHasRings = ringSearch.numRings() > 0;
        if (!moleculeHasRings) {
            return;
        }
        for (int i = 0; i < candidateList.size(); ++i) {
            boolean isConnected;
            IAtomContainer candidate = candidateList.get(i);
            if (candidate == null) {
                candidateList.remove(i);
                --i;
                continue;
            }
            for (int j = 0; j < candidate.getAtomCount(); ++j) {
                IAtom atom = candidate.getAtom(j);
                if (!ringSearch.cyclic(atom) || !candidate.contains(atom)) continue;
                candidate.removeAtom(atom);
                --j;
            }
            if (candidate.isEmpty()) {
                candidateList.remove(i);
                --i;
            }
            if (isConnected = ConnectivityChecker.isConnected((IAtomContainer)candidate)) continue;
            IAtomContainerSet components = ConnectivityChecker.partitionIntoMolecules((IAtomContainer)candidate);
            for (IAtomContainer component : components.atomContainers()) {
                candidateList.add(component);
            }
            candidateList.remove(i);
            --i;
        }
    }

    protected List<IAtomContainer> removeTooSmallAndTooLargeCandidates(List<IAtomContainer> candidateList) {
        if (candidateList == null || candidateList.isEmpty()) {
            return new ArrayList<IAtomContainer>(0);
        }
        ArrayList<IAtomContainer> processedCandidates = new ArrayList<IAtomContainer>(candidateList.size());
        for (IAtomContainer candidate : candidateList) {
            int carbonCount = 0;
            for (IAtom atom : candidate.atoms()) {
                if (atom.getAtomicNumber() != 6) continue;
                ++carbonCount;
            }
            if (carbonCount < this.linearSugarCandidateMinSizeSetting || carbonCount > this.linearSugarCandidateMaxSizeSetting) continue;
            processedCandidates.add(candidate);
        }
        return processedCandidates;
    }

    public static enum PreservationMode {
        ALL(0),
        HEAVY_ATOM_COUNT(5),
        MOLECULAR_WEIGHT(60);

        private final int defaultThreshold;

        private PreservationMode(int aDefaultValue) {
            this.defaultThreshold = aDefaultValue;
        }

        public int getDefaultThreshold() {
            return this.defaultThreshold;
        }
    }
}

