/*
 * Decompiled with CFR 0.152.
 */
package org.h2gis.drivers.geojson;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryCollection;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.LineString;
import com.vividsolutions.jts.geom.LinearRing;
import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;
import com.vividsolutions.jts.geom.PrecisionModel;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import org.h2gis.drivers.geojson.GeoJsonField;
import org.h2gis.drivers.utility.FileUtil;
import org.h2gis.h2spatialapi.EmptyProgressVisitor;
import org.h2gis.h2spatialapi.ProgressVisitor;
import org.h2gis.utilities.JDBCUtilities;
import org.h2gis.utilities.SFSUtilities;
import org.h2gis.utilities.TableLocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GeoJsonReaderDriver {
    private static final ArrayList<String> geomTypes;
    private final File fileName;
    private final Connection connection;
    private static GeometryFactory GF;
    private PreparedStatement preparedStatement = null;
    private JsonFactory jsFactory;
    private boolean hasProperties = false;
    private int featureCounter = 1;
    private ProgressVisitor progress = new EmptyProgressVisitor();
    private FileChannel fc;
    private long fileSize = 0L;
    private long readFileSizeEachNode = 1L;
    private long nodeCountProgress = 0L;
    private static final int AVERAGE_NODE_SIZE = 500;
    boolean hasGeometryField = false;
    int fieldIndex = 0;
    private static final Logger log;
    private int parsedSRID = 0;
    private boolean isH2;
    private TableLocation tableLocation;
    private String firstGeometryType;
    private boolean mixedGeometries = false;

    public GeoJsonReaderDriver(Connection connection, File fileName) {
        this.connection = connection;
        this.fileName = fileName;
    }

    public void read(ProgressVisitor progress, String tableReference) throws SQLException, IOException {
        if (FileUtil.isFileImportable(this.fileName, "geojson")) {
            this.isH2 = JDBCUtilities.isH2DataBase((DatabaseMetaData)this.connection.getMetaData());
            this.tableLocation = TableLocation.parse((String)tableReference, (Boolean)this.isH2);
            this.parseGeoJson(progress);
        }
    }

    private void parseGeoJson(ProgressVisitor progress) throws SQLException, IOException {
        this.progress = progress.subProcess(100);
        this.init();
        if (!this.parseMetadata()) {
            throw new SQLException("Cannot create the table " + this.tableLocation + " to import the GeoJSON data");
        }
        GF = new GeometryFactory(new PrecisionModel(), this.parsedSRID);
        this.parseData();
        this.setGeometryTypeConstraints();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean parseMetadata() throws SQLException, IOException {
        FileInputStream fis = null;
        StringBuilder metadataBuilder = new StringBuilder();
        try {
            fis = new FileInputStream(this.fileName);
            this.fc = fis.getChannel();
            this.fileSize = this.fc.size();
            this.readFileSizeEachNode = Math.max(1L, this.fileSize / 500L / 100L);
            this.nodeCountProgress = 0L;
            JsonParser jp = this.jsFactory.createParser((InputStream)fis);
            metadataBuilder.append("CREATE TABLE ");
            metadataBuilder.append(this.tableLocation);
            metadataBuilder.append(" (");
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            String geomType = jp.getText();
            if (!geomType.equalsIgnoreCase(GeoJsonField.FEATURECOLLECTION)) throw new SQLException("Malformed GeoJSON file. Expected 'FeatureCollection', found '" + geomType + "'");
            jp.nextToken();
            String firstParam = jp.getText();
            if (firstParam.equalsIgnoreCase(GeoJsonField.CRS)) {
                this.parsedSRID = this.readCRS(jp);
                this.readFeatures(jp, geomType, metadataBuilder);
            } else {
                if (!firstParam.equalsIgnoreCase(GeoJsonField.FEATURES)) throw new SQLException("Malformed GeoJSON file. Expected 'features', found '" + firstParam + "'");
                this.readFeatures(jp, geomType, metadataBuilder);
            }
            jp.close();
        }
        catch (FileNotFoundException ex) {
            throw new SQLException(ex);
        }
        finally {
            try {
                if (fis != null) {
                    fis.close();
                }
            }
            catch (IOException ex) {
                throw new IOException(ex);
            }
        }
        if (!this.hasGeometryField) throw new SQLException("The first feature must contains a geometry field.");
        Statement stmt = this.connection.createStatement();
        stmt.execute(metadataBuilder.toString());
        stmt.close();
        if (this.fieldIndex <= 0) return false;
        StringBuilder insert = new StringBuilder("INSERT INTO ").append(this.tableLocation).append(" VALUES ( ?");
        for (int i = 1; i < this.fieldIndex; ++i) {
            insert.append(",?");
        }
        insert.append(");");
        this.preparedStatement = this.connection.prepareStatement(insert.toString());
        return true;
    }

    private void readFeatures(JsonParser jp, String geomType, StringBuilder metadataBuilder) throws IOException, SQLException {
        jp.nextToken();
        JsonToken token = jp.nextToken();
        if (token != JsonToken.END_ARRAY) {
            jp.nextToken();
            jp.nextToken();
            geomType = jp.getText();
            if (geomType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                jp.nextToken();
                String firstField = jp.getText();
                if (firstField.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
                    this.parseGeometryMetadata(jp, metadataBuilder);
                    this.hasGeometryField = true;
                    ++this.fieldIndex;
                    jp.nextToken();
                } else if (firstField.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
                    this.fieldIndex = this.parseMetadataProperties(jp, metadataBuilder, this.fieldIndex);
                    this.hasProperties = true;
                }
                jp.nextToken();
                if (jp.getCurrentToken() != JsonToken.END_OBJECT) {
                    String secondParam = jp.getText();
                    if (secondParam.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
                        this.parseGeometryMetadata(jp, metadataBuilder);
                        this.hasGeometryField = true;
                        ++this.fieldIndex;
                        jp.nextToken();
                    } else if (secondParam.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
                        this.fieldIndex = this.parseMetadataProperties(jp, metadataBuilder, this.fieldIndex);
                        this.hasProperties = true;
                    }
                    jp.nextToken();
                }
                if (!this.hasProperties) {
                    metadataBuilder.append(",ID INT, PRIMARY KEY (ID)");
                    ++this.fieldIndex;
                }
                metadataBuilder.append(")");
            } else {
                throw new SQLException("Malformed GeoJSON file. Expected 'Feature', found '" + geomType + "'");
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void parseGeometryMetadata(JsonParser jp, StringBuilder metadataBuilder) throws IOException, SQLException {
        jp.nextToken();
        jp.nextToken();
        jp.nextToken();
        this.firstGeometryType = jp.getText().toLowerCase();
        if (geomTypes.contains(this.firstGeometryType)) {
            this.checkCoordinates(jp, metadataBuilder);
            return;
        } else {
            if (!this.firstGeometryType.equals(GeoJsonField.GEOMETRYCOLLECTION)) throw new SQLException("Unsupported geometry : " + this.firstGeometryType);
            jp.nextToken();
            if (!jp.getText().equalsIgnoreCase(GeoJsonField.GEOMETRIES)) throw new SQLException("Malformed GeoJSON file. Expected 'geometries', found '" + jp.getText() + "'");
            jp.skipChildren();
            if (this.isH2) {
                metadataBuilder.append("THE_GEOM GEOMETRY");
            } else {
                metadataBuilder.append("THE_GEOM GEOMETRY(geometry,").append(this.parsedSRID).append(")");
            }
            this.firstGeometryType = GeoJsonField.GEOMETRY;
            this.mixedGeometries = true;
        }
    }

    private void checkCoordinates(JsonParser jp, StringBuilder metadataBuilder) throws SQLException, IOException {
        jp.nextToken();
        if (jp.getText().equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.skipChildren();
            if (this.isH2) {
                metadataBuilder.append("THE_GEOM GEOMETRY");
            } else {
                metadataBuilder.append("THE_GEOM GEOMETRY(geometry,").append(this.parsedSRID).append(")");
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + jp.getText() + "'");
        }
    }

    private int parseMetadataProperties(JsonParser jp, StringBuilder metadataBuilder, int fieldIndex) throws IOException {
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String fieldName = jp.getText().toUpperCase();
            JsonToken value = jp.nextToken();
            if (fieldIndex > 0) {
                metadataBuilder.append(",");
            }
            if (value == JsonToken.VALUE_STRING) {
                metadataBuilder.append(fieldName).append(" VARCHAR");
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_TRUE) {
                metadataBuilder.append(fieldName).append(" BOOLEAN");
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_FALSE) {
                metadataBuilder.append(fieldName).append(" BOOLEAN");
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_NUMBER_FLOAT) {
                metadataBuilder.append(fieldName).append(" FLOAT8");
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_NUMBER_INT) {
                metadataBuilder.append(fieldName).append(" INT");
                ++fieldIndex;
                continue;
            }
            if (value != JsonToken.VALUE_NULL) continue;
            metadataBuilder.append(fieldName).append(" VARCHAR");
            ++fieldIndex;
        }
        return fieldIndex;
    }

    private void init() {
        this.jsFactory = new JsonFactory();
        this.jsFactory.configure(JsonParser.Feature.ALLOW_COMMENTS, true);
        this.jsFactory.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        this.jsFactory.configure(JsonParser.Feature.ALLOW_NON_NUMERIC_NUMBERS, true);
    }

    private PreparedStatement getPreparedStatement() {
        return this.preparedStatement;
    }

    private void parseFeature(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String firstField = jp.getText();
        this.fieldIndex = 1;
        if (firstField.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
            this.setGeometry(jp);
        } else if (firstField.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
            this.parseProperties(jp, this.fieldIndex);
        }
        jp.nextToken();
        if (jp.getCurrentToken() != JsonToken.END_OBJECT) {
            String secondParam = jp.getText();
            if (secondParam.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
                this.setGeometry(jp);
            } else if (secondParam.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
                this.parseProperties(jp, this.fieldIndex);
            }
            jp.nextToken();
        }
        if (!this.hasProperties) {
            this.getPreparedStatement().setObject(this.fieldIndex, this.featureCounter);
        }
        this.getPreparedStatement().execute();
    }

    private void setGeometry(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        jp.nextToken();
        jp.nextToken();
        String geometryType = jp.getText();
        if (!this.mixedGeometries && !geometryType.equalsIgnoreCase(this.firstGeometryType)) {
            this.mixedGeometries = true;
            this.firstGeometryType = "geometry";
        }
        this.getPreparedStatement().setObject(this.fieldIndex, this.parseGeometry(jp, geometryType));
        ++this.fieldIndex;
    }

    private Geometry parseGeometry(JsonParser jsParser, String geometryType) throws IOException, SQLException {
        if (geometryType.equalsIgnoreCase(GeoJsonField.POINT)) {
            return this.parsePoint(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOINT)) {
            return this.parseMultiPoint(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.LINESTRING)) {
            return this.parseLinestring(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTILINESTRING)) {
            return this.parseMultiLinestring(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.POLYGON)) {
            return this.parsePolygon(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOLYGON)) {
            return this.parseMultiPolygon(jsParser);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.GEOMETRYCOLLECTION)) {
            return this.parseGeometryCollection(jsParser);
        }
        throw new SQLException("Unsupported geometry : " + geometryType);
    }

    private void parseProperties(JsonParser jp, int fieldIndex) throws IOException, SQLException {
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            JsonToken value = jp.nextToken();
            if (value == JsonToken.VALUE_STRING) {
                this.getPreparedStatement().setObject(fieldIndex, jp.getText());
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_TRUE) {
                this.getPreparedStatement().setObject(fieldIndex, jp.getValueAsBoolean());
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_FALSE) {
                this.getPreparedStatement().setObject(fieldIndex, jp.getValueAsBoolean());
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_NUMBER_FLOAT) {
                this.getPreparedStatement().setObject(fieldIndex, jp.getValueAsDouble());
                ++fieldIndex;
                continue;
            }
            if (value == JsonToken.VALUE_NUMBER_INT) {
                this.getPreparedStatement().setObject(fieldIndex, jp.getValueAsInt());
                ++fieldIndex;
                continue;
            }
            if (value != JsonToken.VALUE_NULL) continue;
            this.getPreparedStatement().setObject(fieldIndex, null);
            ++fieldIndex;
        }
    }

    private void parseFeatures(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String firstParam = jp.getText();
        if (firstParam.equalsIgnoreCase(GeoJsonField.CRS)) {
            firstParam = this.skipCRS(jp);
        }
        if (firstParam.equalsIgnoreCase(GeoJsonField.FEATURES)) {
            jp.nextToken();
            JsonToken token = jp.nextToken();
            while (token != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geomType = jp.getText();
                if (geomType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                    if (this.progress.isCanceled()) {
                        throw new SQLException("Canceled by user");
                    }
                    this.parseFeature(jp);
                    token = jp.nextToken();
                    ++this.featureCounter;
                    if (this.nodeCountProgress++ % this.readFileSizeEachNode != 0L) continue;
                    try {
                        this.progress.setStep((int)((double)this.fc.position() / (double)this.fileSize * 100.0));
                    }
                    catch (IOException iOException) {}
                    continue;
                }
                throw new SQLException("Malformed GeoJSON file. Expected 'Feature', found '" + geomType + "'");
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'features', found '" + firstParam + "'");
        }
    }

    private Point parsePoint(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            Point point = GF.createPoint(this.parseCoordinate(jp));
            return point;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiPoint parseMultiPoint(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            MultiPoint mPoint = GF.createMultiPoint(this.parseCoordinates(jp));
            jp.nextToken();
            return mPoint;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private LineString parseLinestring(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            LineString line = GF.createLineString(this.parseCoordinates(jp));
            jp.nextToken();
            return line;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiLineString parseMultiLinestring(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            ArrayList<LineString> lineStrings = new ArrayList<LineString>();
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                lineStrings.add(GF.createLineString(this.parseCoordinates(jp)));
                jp.nextToken();
            }
            MultiLineString line = GF.createMultiLineString(lineStrings.toArray(new LineString[lineStrings.size()]));
            jp.nextToken();
            return line;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private Polygon parsePolygon(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.nextToken();
            int linesIndex = 0;
            LinearRing linearRing = null;
            ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                if (linesIndex == 0) {
                    linearRing = GF.createLinearRing(this.parseCoordinates(jp));
                } else {
                    holes.add(GF.createLinearRing(this.parseCoordinates(jp)));
                }
                jp.nextToken();
                ++linesIndex;
            }
            if (linesIndex > 1) {
                jp.nextToken();
                return GF.createPolygon(linearRing, holes.toArray(new LinearRing[holes.size()]));
            }
            jp.nextToken();
            return GF.createPolygon(linearRing, null);
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiPolygon parseMultiPolygon(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            ArrayList<Polygon> polygons = new ArrayList<Polygon>();
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                int linesIndex = 0;
                LinearRing linearRing = null;
                ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
                while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                    if (linesIndex == 0) {
                        linearRing = GF.createLinearRing(this.parseCoordinates(jp));
                    } else {
                        holes.add(GF.createLinearRing(this.parseCoordinates(jp)));
                    }
                    jp.nextToken();
                    ++linesIndex;
                }
                if (linesIndex > 1) {
                    jp.nextToken();
                    polygons.add(GF.createPolygon(linearRing, holes.toArray(new LinearRing[holes.size()])));
                    continue;
                }
                jp.nextToken();
                polygons.add(GF.createPolygon(linearRing, null));
            }
            jp.nextToken();
            return GF.createMultiPolygon(polygons.toArray(new Polygon[polygons.size()]));
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private GeometryCollection parseGeometryCollection(JsonParser jp) throws IOException, SQLException {
        this.firstGeometryType = GeoJsonField.GEOMETRY;
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.GEOMETRIES)) {
            jp.nextToken();
            jp.nextToken();
            ArrayList<Geometry> geometries = new ArrayList<Geometry>();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geometryType = jp.getText();
                geometries.add(this.parseGeometry(jp, geometryType));
                jp.nextToken();
            }
            jp.nextToken();
            return GF.createGeometryCollection(geometries.toArray(new Geometry[geometries.size()]));
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'geometries', found '" + coordinatesField + "'");
    }

    private Coordinate[] parseCoordinates(JsonParser jp) throws IOException {
        jp.nextToken();
        ArrayList<Coordinate> coords = new ArrayList<Coordinate>();
        while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            coords.add(this.parseCoordinate(jp));
        }
        return coords.toArray(new Coordinate[coords.size()]);
    }

    private Coordinate parseCoordinate(JsonParser jp) throws IOException {
        Coordinate coord;
        jp.nextToken();
        double x = jp.getDoubleValue();
        jp.nextToken();
        double y = jp.getDoubleValue();
        jp.nextToken();
        if (jp.getCurrentToken() == JsonToken.END_ARRAY) {
            coord = new Coordinate(x, y);
        } else {
            double z = jp.getDoubleValue();
            jp.nextToken();
            coord = new Coordinate(x, y, z);
        }
        jp.nextToken();
        return coord;
    }

    private void parseData() throws IOException, SQLException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(this.fileName);
            JsonParser jp = this.jsFactory.createParser((InputStream)fis);
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            String geomType = jp.getText();
            if (!geomType.equalsIgnoreCase(GeoJsonField.FEATURECOLLECTION)) {
                throw new SQLException("Malformed GeoJSON file. Expected 'FeatureCollection', found '" + geomType + "'");
            }
            this.parseFeatures(jp);
            jp.close();
        }
        catch (FileNotFoundException ex) {
            throw new SQLException(ex);
        }
        finally {
            try {
                if (fis != null) {
                    fis.close();
                }
            }
            catch (IOException ex) {
                throw new SQLException(ex);
            }
        }
    }

    private int readCRS(JsonParser jp) throws IOException, SQLException {
        int srid = 0;
        jp.nextToken();
        jp.nextToken();
        jp.nextToken();
        String firstField = jp.getText();
        if (firstField.equalsIgnoreCase(GeoJsonField.NAME)) {
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            String crsURI = jp.getText();
            String[] split = crsURI.toLowerCase().split(GeoJsonField.CRS_URN_EPSG);
            if (split != null) {
                srid = Integer.valueOf(split[1]);
            } else {
                log.warn("The CRS URN " + crsURI + " is not supported.");
            }
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
        } else if (firstField.equalsIgnoreCase(GeoJsonField.LINK)) {
            log.warn("Linked CRS is not supported.");
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
        } else {
            throw new SQLException("Malformed GeoJSON CRS element.");
        }
        return srid;
    }

    private String skipCRS(JsonParser jp) throws IOException {
        jp.nextToken();
        jp.skipChildren();
        jp.nextToken();
        return jp.getText();
    }

    private void setGeometryTypeConstraints() throws SQLException {
        if (this.isH2) {
            this.connection.createStatement().execute(String.format("ALTER TABLE %s ALTER COLUMN the_geom %s", this.tableLocation.toString(), this.firstGeometryType));
            SFSUtilities.addTableSRIDConstraint((Connection)this.connection, (TableLocation)this.tableLocation, (int)this.parsedSRID);
        } else {
            this.connection.createStatement().execute(String.format("ALTER TABLE %s ALTER COLUMN the_geom SET DATA TYPE geometry(%s,%d)", this.tableLocation.toString(), this.firstGeometryType, this.parsedSRID));
        }
    }

    static {
        log = LoggerFactory.getLogger(GeoJsonReaderDriver.class);
        geomTypes = new ArrayList();
        geomTypes.add(GeoJsonField.POINT);
        geomTypes.add(GeoJsonField.MULTIPOINT);
        geomTypes.add(GeoJsonField.LINESTRING);
        geomTypes.add(GeoJsonField.MULTILINESTRING);
        geomTypes.add(GeoJsonField.POLYGON);
        geomTypes.add(GeoJsonField.MULTIPOLYGON);
    }
}

