/*
 * $Header: /cvshome/repository/org/osgi/framework/Filter.java,v 1.7 2001/08/10 09:29:06 pkriens Exp $
 *
 * Copyright (c) The Open Services Gateway Initiative (2000-2001).
 * All Rights Reserved.
 *
 * Implementation of certain elements of the Open Services Gateway Initiative
 * (OSGI) Specification may be subject to third party intellectual property
 * rights, including without limitation, patent rights (such a third party may
 * or may not be a member of OSGi). OSGi is not responsible and shall not be
 * held responsible in any manner for identifying or failing to identify any or
 * all such third party intellectual property rights.
 *
 * This document and the information contained herein are provided on an "AS
 * IS" basis and OSGI DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION HEREIN WILL
 * NOT INFRINGE ANY RIGHTS AND ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT WILL OSGI BE LIABLE FOR ANY
 * LOSS OF PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTIAL,
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH THIS
 * DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH LOSS OR DAMAGE.
 *
 * All Company, brand and product names may be trademarks that are the sole
 * property of their respective owners. All rights reserved.
 */

package org.osgi.framework;

import java.util.Dictionary;

/**
 * An RFC 1960-based Filter.
 * <p><tt>Filter</tt> objects can be created by calling
 * {@link BundleContext#createFilter} with the chosen filter string.
 * <p>A <tt>Filter</tt> object can be used numerous times to determine if the
 * match argument matches the filter string that was used to create the <tt>Filter</tt>
 * object.
 *
 * <p>The syntax of a filter string is the string representation of LDAP search filters
 * as defined in RFC 1960: <i>A String Representation of LDAP Search Filters</i> (available
 * at http://www.ietf.org/rfc/rfc1960.txt).
 * It should be noted that RFC 2254: <i>A String Representation of LDAP Search Filters</i>
 * (available at http://www.ietf.org/rfc/rfc2254.txt) supercedes
 * RFC 1960 but only adds extensible matching and is not applicable for this OSGi Framework API.
 *
 * <p>The string representation of an LDAP search filter uses a prefix format, and
 * is defined with the following grammar.
 * <pre>
 * &lt;filter&gt; ::= '(' &lt;filtercomp&gt; ')'
 * &lt;filtercomp&gt; ::= &lt;and&gt; | &lt;or&gt; | &lt;not&gt; | &lt;item&gt;
 * &lt;and&gt; ::= '&' &lt;filterlist&gt;
 * &lt;or&gt; ::= '|' &lt;filterlist&gt;
 * &lt;not&gt; ::= '!' &lt;filter&gt;
 * &lt;filterlist&gt; ::= &lt;filter&gt; | &lt;filter&gt; &lt;filterlist&gt;
 * &lt;item&gt; ::= &lt;simple&gt; | &lt;present&gt; | &lt;substring&gt;
 * &lt;simple&gt; ::= &lt;attr&gt; &lt;filtertype&gt; &lt;value&gt;
 * &lt;filtertype&gt; ::= &lt;equal&gt; | &lt;approx&gt; | &lt;greater&gt; | &lt;less&gt;
 * &lt;equal&gt; ::= '='
 * &lt;approx&gt; ::= '~='
 * &lt;greater&gt; ::= '&gt;='
 * &lt;less&gt; ::= '&lt;='
 * &lt;present&gt; ::= &lt;attr&gt; '=*'
 * &lt;substring&gt; ::= &lt;attr&gt; '=' &lt;initial&gt; &lt;any&gt; &lt;final&gt;
 * &lt;initial&gt; ::= NULL | &lt;value&gt;
 * &lt;any&gt; ::= '*' &lt;starval&gt;
 * &lt;starval&gt; ::= NULL | &lt;value&gt; '*' &lt;starval&gt;
 * &lt;final&gt; ::= NULL | &lt;value&gt;
 * </pre>
 * <p><tt>&lt;attr&gt;</tt> is a string representing an attribute, or
 * key, in the properties objects of the services registered in the Framework.
 * Attribute names are not case sensitive; that is, <tt>cn</tt> and <tt>CN</tt> both refer to the same attribute.
 * <tt>&lt;attr&gt;</tt> should contain no spaces though white space is allowed between the initial parenthesis &quot;(&quot;
 * and the start of the key, and between the end of the key and the equal sign &quot;=&quot;.
 * <tt>&lt;value&gt;</tt> is a string representing the value, or part of
 * one, of a key in the properties objects of the registered services.
 * If a <tt>&lt;value&gt;</tt> must contain one of the characters '<tt>*</tt>'
 * or '<tt>(</tt>' or '<tt>)</tt>', these characters should be escaped by preceding
 * them with the backslash '<tt>\</tt>' character. Spaces are significant in <tt>&lt;value&gt;</tt>.
 * Space charactes are defined by <tt>java.lang.Character.isWhiteSpace()</tt>.
 * Note that although both the <tt>&lt;substring&gt;</tt> and <tt>&lt;present&gt;</tt> productions
 * can produce the <tt>'attr=*'</tt> construct; this construct is used only to denote a presence filter.
 *
 * <p>Examples of LDAP filters are:
 *
 * <pre>
 *   &quot;(cn=Babs Jensen)&quot;
 *   &quot;(!(cn=Tim Howes))&quot;
 *   &quot;(&(&quot; + Constants.OBJECTCLASS + &quot;=Person)(|(sn=Jensen)(cn=Babs J*)))&quot;
 *   &quot;(o=univ*of*mich*)&quot;
 * </pre>
 *
 * <p>The approximate match (<tt>~=</tt>) is implementation specific but should at least ignore
 * case and white space differences. Optional are codes like soundex or other smart "closeness" comparisons.
 *
 * <p>Comparison of values is not straightforward. Strings are compared differently than numbers and it is
 * possible for a key to have multiple values. Note that keys in the match argument must always be strings.
 * The comparison is defined by the object type of the key's value. The following rules apply for comparison:
 * <pre>
 * Property Value Type    Comparison Type
 * String   			  String comparison
 * Integer, Long, Float,
 * Double, Byte, Short,
 * BigInteger,BigDecimal  Numerical comparison
 * Character			  Character comparison
 * Boolean  			  Equality comparisons only
 * [ ] (array)  		  Recursively applied to values
 * Vector   			  Recursively applied to elements
 * </pre>
 * Arrays of primitives are also supported.
 *
 * A filter matches a key that has multiple values if it matches at least one of those values. For example,
 * <pre>
 * Dictionary d = new Hashtable();
 * d.put( "cn", new String[] { "a", "b", "c" } );
 * </pre>
 * <tt>d</tt> will match <tt>(cn=a)</tt> and also <tt>(cn=b)</tt>
 *
 * <p>A filter component that references a key having an unrecognizable data type will evaluate to
 * <tt>false</tt>.
 *
 * @version $Revision: 1.7 $
 * @author Open Services Gateway Initiative
 * @since 1.1
 */

public interface Filter
{
	/**
	 * Filter using a service's properties.
	 * <p>The filter is executed using properties of the referenced service.
	 *
	 * @param reference The reference to the service whose properties are used in the match.
	 *
	 * @return <tt>true</tt> if the service's properties match this filter;
	 * <tt>false</tt> otherwise.
	 */
	public boolean match(ServiceReference reference);

	/**
	 * Filter using a <tt>Dictionary</tt> object.
	 * The Filter is executed using the <tt>Dictionary</tt> object's keys and values.
	 *
	 * @param dictionary The <tt>Dictionary</tt> object whose keys are used in the match.
	 *
	 * @return <tt>true</tt> if the <tt>Dictionary</tt> object's keys and values match this filter;
	 * <tt>false</tt> otherwise.
	 *
	 * @exception IllegalArgumentException If <tt>dictionary</tt> contains case
	 * variants of the same key name.
	 */
	public boolean match(Dictionary dictionary);

	/**
	 * Returns this <tt>Filter</tt> object's filter string.
	 * <p>The filter string is normalized by removing whitespace which does
	 * not affect the meaning of the filter.
	 *
	 * @return Filter string.
	 */
	public String toString();

	/**
	 * Compares this <tt>Filter</tt> object to another object.
	 *
	 * @param obj The object to compare against this <tt>Filter</tt> object.
	 *
	 * @return If the other object is a <tt>Filter</tt> object, then
	 * returns <tt>this.toString().equals(obj.toString()</tt>;
	 * <tt>false</tt> otherwise.
	 */
	public boolean equals(Object obj);

	/**
	 * Returns the hashCode for this <tt>Filter</tt> object.
	 *
	 * @return The hashCode of the filter string; that is, <tt>this.toString().hashCode()</tt>.
	 */
	public int hashCode();
}


