/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: cmi@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: DefaultCMIPRODelegate.java 2438 2010-02-24 16:20:30Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.cmi.rmi;

import java.rmi.NoSuchObjectException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.ExportException;
import java.rmi.server.UnicastRemoteObject;

import javax.rmi.CORBA.PortableRemoteObjectDelegate;
import javax.rmi.CORBA.Tie;
import javax.rmi.CORBA.Util;

import org.omg.CORBA.ORB;
import org.omg.CORBA.SystemException;
import org.ow2.cmi.config.JNDIConfig;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import com.sun.corba.se.impl.util.RepositoryId;
import com.sun.corba.se.impl.util.Utility;
import com.sun.corba.se.spi.presentation.rmi.StubAdapter;

/**
 * Define a delegate for supporting multiprotocols based on the default PRODelegate of Sun.
 * @author The new CMI team
 * @see com.sun.corba.se.impl.javax.rmi.PortableRemoteObject
 */
public class DefaultCMIPRODelegate implements PortableRemoteObjectDelegate {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(DefaultCMIPRODelegate.class);


    /**
     * Makes a Remote object ready for remote communication. This normally
     * happens implicitly when the object is sent or received as an argument
     * on a remote method call, but in some circumstances it is useful to
     * perform this action by making an explicit call.  See the
     * {@link Stub#connect} method for more information.
     * @param target the object to connect.
     * @param source a previously connected object.
     * @throws RemoteException if <code>source</code> is not connected
     * or if <code>target</code> is already connected to a different ORB than
     * <code>source</code>.
     */
    public void connect(final Remote target, final Remote source) throws RemoteException {
        if (target == null || source == null) {
            logger.error("invalid argument");
            throw new NullPointerException("invalid argument");
        }
        String protocol = JNDIConfig.getCurrentProtocolName();
        if(protocol.equals("iiop")) {
            ORB orb = null;
            try {
                if (StubAdapter.isStub(source)) {
                    orb = StubAdapter.getORB(source);
                } else {
                    // Is this a servant that was exported to iiop?
                    Tie tie = Util.getTie(source);
                    if (tie != null) {
                        orb = tie.orb();
                    }
                }
            } catch (SystemException e) {
                logger.error("'source' object not connected", e);
                throw new RemoteException("'source' object not connected", e);
            }
            boolean targetIsIIOP = false;
            Tie targetTie = null;
            if (StubAdapter.isStub(target)) {
                targetIsIIOP = true;
            } else {
                targetTie = Util.getTie(target);
                if (targetTie != null) {
                    targetIsIIOP = true;
                }
            }
            if (!targetIsIIOP) {
                // Yes. Do we have an ORB from the source object?
                        // If not, we're done - there is nothing to do to
                // connect a JRMP object. If so, it is an error because
                // the caller mixed JRMP and IIOP...
                if (orb != null) {
                    logger.error("'source' object exported to IIOP, 'target' is JRMP");
                    throw new RemoteException(
                    "'source' object exported to IIOP, 'target' is JRMP");
                }
            } else {
                // The target object is IIOP. Make sure we have a
                // valid ORB from the source object...
                if (orb == null) {
                    logger.error("'source' object is JRMP, 'target' is IIOP");
                    throw new RemoteException(
                            "'source' object is JRMP, 'target' is IIOP");
                }
                try {
                    if (targetTie != null) {
                        // Is the tie already connected?
                        try {
                            ORB existingOrb = targetTie.orb();

                            // Yes. Is it the same orb?
                            if (existingOrb == orb) {

                                // Yes, so nothing to do...
                                return;
                            } else {
                                // No, so this is an error...
                                logger.error("'target' object was already connected");
                                throw new RemoteException(
                                "'target' object was already connected");
                            }
                        } catch (SystemException e) {
                            logger.error(e);
                        }
                        // No, so do it...
                        targetTie.orb(orb);
                    } else {
                        StubAdapter.connect(target, orb);
                    }
                } catch (SystemException e) {
                    // The stub or tie was already connected...
                    logger.error("'target' object was already connected", e);
                    throw new RemoteException(
                            "'target' object was already connected", e);
                }
            }
        }
    }

    /**
     * Makes a server object ready to receive remote calls. Note
     * that subclasses of PortableRemoteObject do not need to call this
     * method, as it is called by the constructor.
     * @param obj the server object to export.
     * @exception RemoteException if export fails.
     */
    public void exportObject(final Remote obj) throws RemoteException {
        if (obj == null) {
            logger.error("invalid argument");
            throw new NullPointerException("invalid argument");
        }
        String protocol = JNDIConfig.getCurrentProtocolName();
        if(protocol.equals("iiop")) {
            logger.debug("Exporting to iiop...");
            // Has this object already been exported to IIOP?
            if (Util.getTie(obj) != null) {
                // Yes, so this is an error...
                logger.error("{0} already exported", obj.getClass().getName());
                throw new ExportException(obj.getClass().getName() + " already exported");
            }
            // Can we load a Tie?
            Tie theTie = Utility.loadTie(obj);
            if (theTie != null) {
                // Yes, so export it to IIOP...
                Util.registerTarget(theTie, obj);
            } else {
                logger.error("Tie not found");
                throw new ExportException("Tie not found");
            }
        } else if(protocol.equals("jrmp")) {
            logger.debug("Exporting to jrmp...");
            UnicastRemoteObject.exportObject(obj);
        } else {
            logger.error("{0} is not a supported protocol. Use jrmp, iiop or irmi.", protocol);
            throw new ExportException(protocol+" is not a supported protocol. Use jrmp, iiop or irmi.");
        }

    }

    /**
     * Checks to ensure that an object of a remote or abstract interface type
     * can be cast to a desired type.
     * @param narrowFrom the object to check.
     * @param narrowTo the desired type.
     * @return an object which can be cast to the desired type.
     * @throws ClassCastException if narrowFrom cannot be cast to narrowTo.
     */
    @SuppressWarnings("unchecked")
    public Object narrow(final Object narrowFrom, final Class narrowTo)
    throws ClassCastException {

        if (narrowFrom == null) {
            return null;
        }
        if (narrowTo== null) {
            logger.error("invalid argument");
            throw new NullPointerException("invalid argument");
        }
        try {
            if (narrowTo.isAssignableFrom(narrowFrom.getClass())) {
                return narrowFrom;
            }
            // Is narrowTo an interface that might be
            // implemented by a servant running on iiop?
            if (narrowTo.isInterface()
                    && narrowTo != java.io.Serializable.class
                    && narrowTo != java.io.Externalizable.class) {

                org.omg.CORBA.Object narrowObj
                = (org.omg.CORBA.Object) narrowFrom;

                // Create an id from the narrowTo type...
                String id = RepositoryId.createForAnyType(narrowTo);

                if (narrowObj._is_a(id)) {
                    return Utility.loadStub(narrowObj, narrowTo);
                } else {
                    logger.error("Object is not of remote type {0}", narrowTo.getName());
                    throw new ClassCastException("Object is not of remote type "
                            + narrowTo.getName());
                }
            } else {
                logger.error("Class {0} is not a valid remote interface", narrowTo.getName());
                throw new ClassCastException("Class "+narrowTo.getName()
                        +" is not a valid remote interface");
            }
        } catch(Exception error) {
            logger.error(error);
            ClassCastException cce = new ClassCastException();
            cce.initCause(error);
            throw cce;
        }
    }

    /**
     * Returns a stub for the given server object.
     * @param obj the server object for which a stub is required. Must either be a subclass
     * of PortableRemoteObject or have been previously the target of a call to
     * {@link #exportObject}.
     * @return the most derived stub for the object.
     * @exception NoSuchObjectException if a stub cannot be located for the given server object.
     */
    public Remote toStub(final Remote obj) throws NoSuchObjectException {
        Remote result = null;
        if (obj == null) {
            logger.error("invalid argument");
            throw new NullPointerException("invalid argument");
        }
        // If the class is already an IIOP stub then return it.
        if (StubAdapter.isStub(obj)) {
            return obj;
        }
        // If the class is already a JRMP stub then return it.
        if (obj instanceof java.rmi.server.RemoteStub) {
            return obj;
        }
        String protocol = JNDIConfig.getCurrentProtocolName();
        if(protocol.equals("iiop")) {
            // Has it been exported to IIOP?
            Tie theTie = Util.getTie(obj);
            if (theTie != null) {
                result = Utility.loadStub(theTie, null, null, true);
            }
        } else if(protocol.equals("jrmp")) {
            result = java.rmi.server.RemoteObject.toStub(obj);
        } else {
            logger.error("Unknown object for {0}", protocol);
            throw new NoSuchObjectException("Unknown object for "+protocol);
        }
        if (result == null) {
            logger.error("Object not exported.");
            throw new NoSuchObjectException("object not exported");
        }
        return result;
    }

    /**
     * Deregisters a server object from the runtime, allowing the object to become
     * available for garbage collection.
     * @param obj the object to unexport.
     * @exception NoSuchObjectException if the remote object is not
     * currently exported.
     */
    public void unexportObject(final Remote obj) throws NoSuchObjectException {
        if (obj == null) {
            logger.error("invalid argument");
            throw new NullPointerException("invalid argument");
        }
        if (StubAdapter.isStub(obj)
                || obj instanceof java.rmi.server.RemoteStub) {
            logger.error("Can only unexport a server object.");
            throw new NoSuchObjectException(
            "Can only unexport a server object.");
        }
        String protocol = JNDIConfig.getCurrentProtocolName();
        if(protocol.equals("iiop")) {
            logger.debug("Unexporting for iiop...");
            Tie theTie = Util.getTie(obj);
            if (theTie != null) {
                Util.unexportObject(obj);
            } else {
                if (Utility.loadTie(obj) != null) {
                    logger.error("Object not exported.");
                    throw new NoSuchObjectException("Object not exported.");
                }
            }
        } else if(protocol.equals("jrmp")) {
            logger.debug("Unexporting for jrmp...");
            UnicastRemoteObject.unexportObject(obj, true);
        } else {
            logger.error("{0} is not a supported protocol. Use jrmp, iiop or irmi.", protocol);
            throw new NoSuchObjectException(protocol+" is not a supported protocol. Use jrmp, iiop or irmi.");
        }
    }

}
