/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2023 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author Marc Jambert - EBM WebSourcing
 */
public final class ByteChannelDumpHelper {

    private ByteChannelDumpHelper() {
    }

    public static final FileChannel createDumpFileChannel(File dumpFile) {
        assert dumpFile != null;
        assert !dumpFile.isDirectory() : "Dump file can't be a directory";
        dumpFile.getParentFile().mkdirs();
        FileOutputStream dumpFileOutputStream;
        try {
            dumpFileOutputStream = new FileOutputStream(dumpFile);
        } catch (FileNotFoundException e) {
            throw new UncheckedException(e);
        }
        return dumpFileOutputStream.getChannel();
    }

    public static final int dumpAndRead(ReadableByteChannel channelToDump, ByteBuffer readBuffer,
            FileChannel dumpChannel) throws IOException {
        ensureChannelIsOpen(channelToDump);
        ByteBuffer readBufferView = readBuffer.duplicate();
        int nbBytesRead = channelToDump.read(readBuffer);
        if (nbBytesRead != -1) {
            readBufferView.limit(readBufferView.position() + nbBytesRead);
            dumpChannel.write(readBufferView);
        }
        return nbBytesRead;
    }

    public static final int dumpAndWrite(WritableByteChannel channelToDump, ByteBuffer writeBuffer,
            FileChannel dumpChannel) throws IOException {
        ensureChannelIsOpen(channelToDump);
        ByteBuffer writeBufferView = writeBuffer.duplicate();
        dumpChannel.write(writeBufferView);
        int nbBytesWritten = channelToDump.write(writeBuffer);
        return nbBytesWritten;
    }
    
    
    private static final void ensureChannelIsOpen(Channel channel) throws ClosedChannelException {
        if (!channel.isOpen()) {
            throw new ClosedChannelException();
        }
    }
    

}
