/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2023 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.io;

import static java.io.File.separator;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/**
 * @author Marc Jambert - EBM WebSourcing
 * @author Christophe DENEUX - Linagora
 */
public final class FileSystemHelper {

    /**
     * Error message of {@link #getRelativePath(File, File)} when the source
     * path is null
     */
    private final static String ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NULL = "The source path is null";

    /**
     * Error message of {@link #getRelativePath(File, File)} when the target
     * path is null
     */
    private final static String ILLEGAL_ARGUMENT_EXCEPTION_MSG_TARGET_NULL = "The target path is null";

    /**
     * Error message of {@link #getRelativePath(File, File)} when the source
     * directory is not an absolute directory.
     */
    private final static String ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NOT_ABSOLUTE = "The source is not an absolute directory";

    /**
     * Error message of {@link #getRelativePath(File, File)} when the source
     * directory is not a directory.
     */
    private final static String ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NOT_DIR = "The source is not an directory";

    /**
     * Error message of {@link #getRelativePath(File, File)} when the target
     * directory is not an absolute directory.
     */
    private final static String ILLEGAL_ARGUMENT_EXCEPTION_MSG_TARGET_NOT_ABSOLUTE = "The target is not an absolute path";

    private final static String FILE_SEPARATOR_AS_REGEXP = Pattern.quote(separator);

    private static final Random random = new Random();

    private static final boolean IS_WINDOWS;

    static {
        boolean res;
        try {
            res = System.getProperty("os.name").startsWith("Windows");
        } catch (final SecurityException e) {
            // so that we are 100% sure that the class loading does not fail!
            Logger.getLogger(FileSystemHelper.class.getName())
                    .log(Level.SEVERE,
                            "SecurityException while getting the os.name system property, defaulting to false.", e);
            res = false;
        }
        IS_WINDOWS = res;
    }

    private FileSystemHelper() {
    }

    public static File createTempDir() throws IOException {
        return createTempDir("tmp"+random.nextInt());
    }

    public static File createTempDir(String prefix) throws IOException {
    	return Files.createTempDirectory(prefix).toFile();
    }

    public static File createTempDir(File parentDir, String prefix) throws IOException {
    	return Files.createTempDirectory(parentDir.toPath(), prefix).toFile();
    }

    /**
     * Cleans a directory without deleting it.
     * 
     * @param directory
     *            directory to clean
     * @throws IOException
     *             in case cleaning is unsuccessful
     */
    public static void cleanDirectory(File directory) throws IOException {
        assert directory != null;
        if (!directory.exists()) {
            String message = directory + " does not exist";
            throw new IOException(message);
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IOException(message);
        }

        File[] files = directory.listFiles();
        if (files == null) { // null if security restricted
            throw new IOException("Failed to list contents of " + directory);
        }

        IOException exception = null;
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            try {
                forceDelete(file);
            } catch (IOException ioe) {
                exception = ioe;
            }
        }

        if (null != exception) {
            throw exception;
        }
    }

    /**
     * Deletes a file. If file is a directory, delete it and all
     * sub-directories.
     * <p>
     * The difference between File.delete() and this method are:
     * <ul>
     * <li>A directory to be deleted does not have to be empty.</li>
     * <li>You get exceptions when a file or directory cannot be deleted.
     * (java.io.File methods returns a boolean)</li>
     * </ul>
     * 
     * @param file
     *            file or directory to delete, must not be <code>null</code>
     * @throws NullPointerException
     *             if the directory is <code>null</code>
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    public static void forceDelete(File file) throws IOException {
        assert file != null;
        if (file.isDirectory()) {
            deleteDirectory(file);
        } else {
            if (!file.exists()) {
                return;
            }
            if (!file.delete()) {
                String message = "Unable to delete file: " + file;
                throw new IOException(message);
            }
        }
    }

    /**
     * Deletes a directory recursively.
     * 
     * @param directory
     *            directory to delete
     * @throws IOException
     *             in case deletion is unsuccessful
     */
    private static void deleteDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }

        cleanDirectory(directory);
        if (!directory.delete()) {
            String message = "Unable to delete directory " + directory + ".";
            throw new IOException(message);
        }
    }

    /**
     * Return relative path.
     * <p>
     * <b>WARNING:</b> Folder path must finished with the character: {@link File#separator}
     * </p>
     * <table border="1">
     * <caption>Examples</caption>
     * <tr>
     * <td>source</td>
     * <td>target</td>
     * <td>relative path</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/dir1/</td>
     * <td>../</td>
     * </tr>
     * <tr>
     * <td>/root/</td>
     * <td>/root/dir1/dir2/</td>
     * <td>dir1/dir2/</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/test.xml</td>
     * <td>../../test.xml</td>
     * </tr>
     * </table>
     * 
     * 
     * @param source
     *            path of the source folder
     * @param target
     *            path of the target folder/file
     * @return the relative path
     * @deprecated Use {@link FileSystemHelper#getRelativePath(File, File)} that
     *             manages correctly non-relative paths and returns a correct
     *             value for relative sub-directories (<code>./dir1/dir2/</code>
     *             )
     */
    @Deprecated
    public static String getRelativePath(String source, String target) {
        // Determine if the target is a file
        String targetFile = null;
        if (!source.endsWith(separator)) {
            throw new IllegalArgumentException();
        }
        if (!target.endsWith(separator)) {
            int lastFolderIndex = target.lastIndexOf(separator) + 1;
            targetFile = target.substring(lastFolderIndex);
            target = target.substring(0, lastFolderIndex);
        }
        String[] sources = source.split(FILE_SEPARATOR_AS_REGEXP);
        String[] targets = target.split(FILE_SEPARATOR_AS_REGEXP);

        // Get the shortest of the two paths
        int length = sources.length < targets.length ? sources.length : targets.length;
        // Get the common part
        int common = 0;
        while (common < length) {
            if (sources[common].equals(targets[common])) {
                common++;
            } else {
                break;
            }
        }
        StringBuilder relativePathBuilder = new StringBuilder();
        for (int i = common; i < sources.length; i++) {
            relativePathBuilder.append(".." + separator);
        }

        for (int i = common; i < targets.length; i++) {
            relativePathBuilder.append(targets[i] + separator);
        }
        if (targetFile != null) {
            relativePathBuilder.append(targetFile);
        }
        return relativePathBuilder.toString();
    }

    /**
     * <p>Return relative path.</p>
     * <table border="1">
     * <caption>Examples</caption>
     * <tr>
     * <td><code>sourcePath</code></td>
     * <td><code>targetPath</code></td>
     * <td>relative path</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/dir1/</td>
     * <td>../</td>
     * </tr>
     * <tr>
     * <td>/root/</td>
     * <td>/root/dir1/dir2/</td>
     * <td>./dir1/dir2/</td>
     * </tr>
     * <tr>
     * <td>/root/dir1/dir2/</td>
     * <td>/root/test.xml</td>
     * <td>../../test.xml</td>
     * </tr>
     * <tr>
     * <td>/root/</td>
     * <td>/notrelative/dir1/</td>
     * <td>/notrelative/dir1/</td>
     * </tr>
     * <tr>
     * <td>/root/dir1</td>
     * <td>/notrelative/test.xml</td>
     * <td>/notrelative/test.xml</td>
     * </tr>
     * </table>
     * 
     * 
     * @param sourcePath
     *            path of the source folder. <b>MUST</b> be an absolute directory. Not null.
     * @param targetPath
     *            path of the target folder/file to relativize against <code>sourcePath</code>. <b>MUST</b> be an
     *            absolute {@link File}. Not null
     * @return the relative path, or <code>targetPath</code> if paths are not relative
     * @throws IllegalArgumentException
     *             if parameters are invalid
     */
    public static String getRelativePath(final File sourcePath, final File targetPath) {
        // Check that the source path is not null
        if (sourcePath == null) {
            throw new IllegalArgumentException(ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NULL);
        }

        // Check that the target path is not null
        if (targetPath == null) {
            throw new IllegalArgumentException(ILLEGAL_ARGUMENT_EXCEPTION_MSG_TARGET_NULL);
        }

        // Check that the source path is an absolute path
        if (!sourcePath.isAbsolute()) {
            throw new IllegalArgumentException(ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NOT_ABSOLUTE);
        }

        // Check that the source path is a directory
        if (!sourcePath.isDirectory()) {
            throw new IllegalArgumentException(ILLEGAL_ARGUMENT_EXCEPTION_MSG_SRC_NOT_DIR);
        }

        // Check that the target path is an absolute path
        if (!targetPath.isAbsolute()) {
            throw new IllegalArgumentException(ILLEGAL_ARGUMENT_EXCEPTION_MSG_TARGET_NOT_ABSOLUTE);
        }

        final String source;
        final String target;
        final String sourceAbsolutePath = sourcePath.getAbsolutePath();
        final String targetAbsolutePath = targetPath.getAbsolutePath();
        if (IS_WINDOWS) {
            // On Windows, relative paths are on the same drive.
            // If the drive letters of the absolute paths are not the same, the paths will be not relative
            final String sourceDrive = sourceAbsolutePath.substring(0, 2);
            final String targetDrive = targetAbsolutePath.substring(0, 2);
            if (targetDrive.equals(sourceDrive)) {
                // Drive letters are the same
                source = sourceAbsolutePath.substring(2);
                target = targetAbsolutePath.substring(2);
            } else {
                // Drive letters are different
                return targetPath.getAbsolutePath();
            }
        } else {
            source = sourceAbsolutePath.substring(1);
            target = targetAbsolutePath.substring(1);
        }

        final String[] sources = source.split(FILE_SEPARATOR_AS_REGEXP);
        final String[] targets = target.split(FILE_SEPARATOR_AS_REGEXP);

        // Get the shortest of the two paths
        final int length = sources.length < targets.length ? sources.length : targets.length;
        // Get the common part
        int common = 0;
        while (common < length) {
            if (sources[common].equals(targets[common])) {
                common++;
            } else {
                break;
            }
        }
        if (common > 0) {
            final StringBuilder relativePathBuilder = new StringBuilder();
            if (common == sources.length) {
                // target is a subdir of source
                relativePathBuilder.append(".").append(separator);
            }
            for (int i = common; i < sources.length; i++) {
                relativePathBuilder.append("..").append(separator);
            }

            for (int i = common; i < targets.length; i++) {
                relativePathBuilder.append(targets[i]);
                if (i < targets.length - 1) {
                    relativePathBuilder.append(separator);
                }
            }
            return relativePathBuilder.toString();
        } else {
            return targetPath.getAbsolutePath();
        }
    }
}
