/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2023 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.io;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

/**
 * @author Marc Jambert - EBM WebSourcing
 */
public class IOHelper {

    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

    private IOHelper() {
    }

    public static void close(Closeable closeable) {
        if (closeable == null)
            return;
        try {
            closeable.close();
        } catch (IOException e) {
            throw new UncheckedException(String.format(
                    "Cannot close object of class '%s'.", closeable.getClass()
                            .getSimpleName()));
        }
    }

    /**
     * Get the line of the specified file corresponding to the specified line
     * number. Each time this method is called, the file is open, read until the
     * specified line number and closed.
     * 
     * @param file
     *            the file to read the line
     * @param lineNum
     *            the number of the line in the specified file
     * @return a String containing the contents of the line, not including any
     *         line-termination characters, or null if the specified line number
     *         is greater than the line number of the file
     * 
     * @throws IOException
     *             if the file does not exist, is a directory rather than a
     *             regular file, or for some other reason cannot be opened for
     *             reading, if a security manager exists and its checkRead
     *             method denies read access to the file or if an I/O error
     *             occurs
     */
    public static final String getLine(File file, int lineNum) throws IOException {
        assert file != null;
        assert lineNum > 0;
        
        String strLine = null;

        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(new FileInputStream(file)));

            int counter = 1;
            strLine = br.readLine();
            while (strLine != null && counter < lineNum) {
                strLine = br.readLine();
                counter++;
            }
        } finally {
            if (br != null) {
                br.close();
            }
        }

        return strLine;
    }

    
    /**
     * Read the specified file
     * 
     * @param file
     *            the file to read
     * 
     * @return the content of the file
     * 
     * @throws IOException
     *             if the file does not exist, is a directory rather than a
     *             regular file, or for some other reason cannot be opened for
     *             reading, if a security manager exists and its checkRead
     *             method denies read access to the file or if an I/O error
     *             occurs
     */
    public static final String readFileAsString(File file) throws IOException {
    	return new String(readFileAsBytes(file));
    }

    
	public static byte[] readFileAsBytes(File file) throws IOException {
		assert file != null;
		if (file.canRead() && (file.length() == 0)) return EMPTY_BYTE_ARRAY;
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			byte[] readBytes = new byte[(int) file.length()];
			int nbRead = fis.read(readBytes);
			assert nbRead == readBytes.length;
			return readBytes;
		} finally {
			IOHelper.close(fis);
		}
	}
}
