/**
 * Copyright (C) 2004 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.javaLog;

import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.logging.ErrorManager;
import java.util.logging.LogRecord;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.wrapper.common.OutputStreamSwitcher;

/**
 * The aim of this class is to permit to specify the output for the console
 * handler of the java.util.logging system.
 *
 * This console handler is also able to choose the right ouput (System.err
 * or System.out) depending on the message level.
 *
 * @author S.Chassande-Barrioz
 */
public class ConsoleHandler extends java.util.logging.Handler {

	protected OutputStreamSwitcher oss;
	private Writer writer;

	public ConsoleHandler() {
		super();
	}

	public void desactivateSwitching(OutputStream newOut) {
		if (oss != null) {
			oss = null;
			setOutput(newOut);
		}
	}

	public void activateSwitching() {
		if (oss == null) {
			oss = new OutputStreamSwitcher();
			setOutput(oss);
		}
	}

	/**
	 * Assign the Outputstream by calling a protected method from the super
	 * class.
	 */
	public void setOutput(OutputStream out) throws SecurityException {
		if (out == null) {
			throw new NullPointerException();
		}
		flush();
		String encoding = getEncoding();
		if (encoding == null) {
			writer = new OutputStreamWriter(out);
		} else {
			try {
				writer = new OutputStreamWriter(out, encoding);
			} catch (UnsupportedEncodingException ex) {
			// This shouldn't happen.  The setEncoding method
			// should have validated that the encoding is OK.
			throw new Error("Unexpected exception " + ex);
			}
		}
	}


	public void publish(LogRecord record) {
		if (!isLoggable(record)) {
			return;
		}
		String msg;
		try {
			msg = getFormatter().format(record);
		} catch (Exception ex) {
			// We don't want to throw an exception here, but we
			// report the exception to any registered ErrorManager.
			reportError(null, ex, ErrorManager.FORMAT_FAILURE);
			return;
		}

		if (oss != null) {
			if (record.getLevel().intValue() >= BasicLevel.WARN) {
				oss.switchOutput(System.err);
			} else {
				oss.switchOutput(System.out);
			}
		}
		try {
			writer.write(msg);
		} catch (Exception ex) {
			// We don't want to throw an exception here, but we
			// report the exception to any registered ErrorManager.
			reportError(null, ex, ErrorManager.WRITE_FAILURE);
		}
		flush();
	}

	public void flush() {
		if (writer != null) {
			try {
				writer.flush();
			} catch (Exception ex) {
				// We don't want to throw an exception here, but we
				// report the exception to any registered ErrorManager.
				reportError(null, ex, ErrorManager.FLUSH_FAILURE);
			}
		}
	}

	public void close() throws SecurityException {
		flush();
	}
}