/**
 * Copyright (C) 2001-2006 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.javaLog;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

import org.objectweb.util.monolog.Monolog;
import org.objectweb.util.monolog.api.LogInfo;
import org.objectweb.util.monolog.file.api.Pattern;
import org.objectweb.util.monolog.wrapper.common.AbstractFactory;

/**
 * The goal of this class is to format a LogRecord with regard to a pattern.
 * In order to support support additional logging layer on top monolog, this
 * class permits to specify package name or class names of its own logging layer.
 * By default the org.apache.commons.logging package is included. To specify
 * the list of names you have to assign the system property 'monolog.wrappers'
 * with a list (separated with commas or white spaces) of your names.
 *
 * @author S.Chassande-Barrioz
 * @author Mohammed Boukada (Log extension)
 */
public class MonologFormatter extends Formatter {

    public final static String WRAPPERS_PROPERTY = "monolog.wrappers";

    /**
     * Number of items to get back in stack trace to see the caller
     */
    private static final int STACK_TRACE_ITEMS = 8;

    private static String[] LOGWRAPPER = {
        "org.objectweb.util.monolog.wrapper",
        "org.apache.commons.logging",
        "java.util.logging.Logger"
    };

    static {
        String wrap = System.getProperty(WRAPPERS_PROPERTY);
        if (wrap != null) {
            ArrayList ws = new ArrayList(5);
            StringTokenizer st = new StringTokenizer(wrap,",;: /", false);
            while (st.hasMoreTokens()) {
                ws.add(st.nextToken());
            }
            String[] wsa = new String[LOGWRAPPER.length + ws.size()];
            System.arraycopy(LOGWRAPPER, 0, wsa, 0, LOGWRAPPER.length);
            for(int i=0; i<ws.size(); i++) {
                wsa[LOGWRAPPER.length + i] = (String) ws.get(i);
            }
            LOGWRAPPER = wsa;
        }
    }

    private final static int PATTERN_ID_LEVEL = -100;
    private final static int PATTERN_ID_TOPIC = -200;
    private final static int PATTERN_ID_DATE = -300;
    private final static int PATTERN_ID_THREAD = -400;
    private final static int PATTERN_ID_MESSAGE = -500;
    private final static int PATTERN_ID_METHOD = -600;
    private final static int PATTERN_ID_OBJECT = -700;
    private final static int PATTERN_ID_LINE_NUMBER = -800;
    private final static int PATTERN_ID_NEW_LINE = -900;
    private final static int PATTERN_ID_INTERVAL = 100;

    private final static String TOKENS =
            "{}"
            + Pattern.LEVEL
            + Pattern.TOPIC
            + Pattern.DATE
            + Pattern.THREAD
            + Pattern.MESSAGE
            + Pattern.METHOD
            + Pattern.OBJECT
            + Pattern.LINE_NUMBER
            + Pattern.PREFIX
            + Pattern.NEW_LINE;

    private final static String patternIdToString(int id) {
        switch(id) {
        case PATTERN_ID_LEVEL: return "" + Pattern.LEVEL;
        case PATTERN_ID_TOPIC: return "" + Pattern.TOPIC;
        case PATTERN_ID_DATE: return "" + Pattern.DATE;
        case PATTERN_ID_THREAD: return "" + Pattern.THREAD;
        case PATTERN_ID_MESSAGE: return "" + Pattern.MESSAGE;
        case PATTERN_ID_METHOD: return "" + Pattern.METHOD;
        case PATTERN_ID_OBJECT: return "" + Pattern.OBJECT;
        case PATTERN_ID_LINE_NUMBER: return "" + Pattern.LINE_NUMBER;
        case PATTERN_ID_NEW_LINE: return "" + Pattern.NEW_LINE;
        default:
            return null;
        }
    }


    Calendar calendar;

    private static long previousTime;

    private static char[] previousTimeWithoutMillis = new char[20]; // "YYYY-MM-DD HH:mm:ss."

    private static boolean debug = Boolean.getBoolean("monolog.pattern.debug");

    /**
     * the pattern in the string format (user value)
     */
    String strPattern;

    /**
     * An array of pattern id representing the user pattern
     */
    int[] pattern;

    /**
     * An array of String used into the pattern
     */
    String[] strings;

    SimpleFormatter simpleFormatter = new SimpleFormatter();

    public MonologFormatter() {
        calendar = Calendar.getInstance();
    }

    public MonologFormatter(String strPattern) {
        this();
        setPattern(strPattern);
    }


    public String getPattern() {
        return strPattern;
    }

    public void setPattern(String p) {
        if (debug) {
            AbstractFactory.debug("Pattern=" + p);
        }
        this.strPattern = p;
        if (strPattern == null) {
            pattern = new int[0];
        } else {
            // Add extension patterns to TOKENS
            String unknownTokens = "";
            String tmp = p.replace(" ", "");
            for (int i=0; i < tmp.length(); i++) {
                if (Character.isLetter(tmp.charAt(i)) && TOKENS.indexOf(tmp.charAt(i)) == -1) {
                    // It is an extension patter, add it
                    unknownTokens += tmp.charAt(i);
                }
            }
            StringTokenizer st = new StringTokenizer(p, TOKENS + unknownTokens, true);
            ArrayList sections = new ArrayList();
            boolean isPrefix = false;
            boolean isObject = false;
            boolean isInSubObject = false;
            int subObjectNumber = 1;
            while (st.hasMoreElements()) {
                String token = st.nextToken();
                if (debug) {
                    AbstractFactory.debug("token=<" + token + ">");
                }
                if (isObject && token.equals("{")) {
                    isInSubObject = true;
                    isObject = false;
                }
                if (token.length() == 1) {
                    char c = token.charAt(0);
                    switch (c) {
                    case '{':
                        if (!isInSubObject) {
                            addSection(sections, token);
                        }
                        break;
                    case '}':
                        if (isInSubObject) {
                            int old = ((Integer) sections.get(sections.size() - 1)).intValue();
                            sections.set(sections.size() - 1,
                                    new Integer(old - subObjectNumber));
                            isInSubObject = false;
                        } else {
                            addSection(sections, token);
                        }
                        break;
                    case Pattern.PREFIX:
                        if (isPrefix) {
                            sections.add(String.valueOf(Pattern.PREFIX));
                        }
                        isPrefix = !isPrefix;
                        break;
                    case Pattern.LEVEL:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_LEVEL, isPrefix);
                        break;
                    case Pattern.TOPIC:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_TOPIC, isPrefix);
                        break;
                    case Pattern.DATE:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_DATE, isPrefix);
                        break;
                    case Pattern.THREAD:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_THREAD, isPrefix);
                        break;
                    case Pattern.MESSAGE:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_MESSAGE, isPrefix);
                        break;
                    case Pattern.METHOD:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_METHOD, isPrefix);
                        break;
                    case Pattern.OBJECT:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_OBJECT, isPrefix);
                        isObject = true;
                        break;
                    case Pattern.LINE_NUMBER:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_LINE_NUMBER, isPrefix);
                        break;
                    case Pattern.NEW_LINE:
                        isPrefix = treatPattern(
                                sections, token, PATTERN_ID_NEW_LINE, isPrefix);
                        break;
                    default:
                        if (Character.isLetter(c)){
                            addSection(sections, new Character(token.charAt(0)));
                            isPrefix = false;
                        }
                        else if (isInSubObject) {
                            subObjectNumber = Integer.parseInt(token);
                        } else {
                            addSection(sections, token);
                        }
                        break;
                    }
                } else if (isObject) {
                    //Ignore
                } else if (isInSubObject) {
                    subObjectNumber = Integer.parseInt(token);
                } else {
                    addSection(sections, token);
                }
            }
            pattern = new int[sections.size()];
            if (debug) {
                AbstractFactory.debug("building pattern array...");
                AbstractFactory.debug("nb of pattern:" + pattern.length);
            }
            ArrayList stringList = new ArrayList(sections.size());
            int cpt = 0;
            for (int i = 0; i < pattern.length; i++) {
                Object o = sections.get(i);
                if (o instanceof String) {
                    if (debug) {
                        AbstractFactory.debug("add current pattern into strings: [" + cpt + ", " + o + "]");
                    }
                    stringList.add(o);
                    pattern[i] = cpt;
                    cpt++;
                } else if (o instanceof Integer) {
                    if (debug) {
                        AbstractFactory.debug("add current pattern as negative number:" + o);
                    }
                    pattern[i] = ((Integer) o).intValue();
                } else if (o instanceof Character) {
                    // For extensions pattern
                    if (debug) {
                        AbstractFactory.debug("add current extension pattern into strings: [" + cpt + ", " + o + "]");
                    }
                    stringList.add(""+o);
                    pattern[i] = cpt;
                    cpt++;
                }
            }
            strings = (String[]) stringList.toArray(new String[cpt]);
            if (debug) {
                AbstractFactory.debug("nb of string:" + strings.length);
            }
        }
    }

    private boolean treatPattern(List sections,
                                 String token,
                                 int tokenId,
                                 boolean isPrefix) {
        if (debug) {
            AbstractFactory.debug("treatPttern(" + tokenId + "):"
                    + " isPrefix=" + isPrefix
                    + " token=" + token
                    + " sections=" + sections
                );
        }
        if (isPrefix) {
            sections.add(new Integer(tokenId));
            return false;
        } else {
            addSection(sections, token);
            return isPrefix;
        }
    }

    private void addSection(List sections, String s) {
        int size = sections.size();
        if (size == 0) {
            if (debug) {
                AbstractFactory.debug("addSection(" + s + ", " + sections + "): first elem");
            }
            sections.add(s);
        } else {
            Object last = sections.get(size - 1);
            if (last instanceof String) {
                sections.set(size - 1, last + s);
                if (debug) {
                    AbstractFactory.debug("addSection(" + s + ", " + sections + "): concat: " + sections.get(size - 1));
                }
            } else {
                if (debug) {
                    AbstractFactory.debug("addSection(" + s + ", " + sections + "): new elem");
                }
                sections.add(s);
            }
        }
    }

    private void addSection(List sections, Character s) {
        int size = sections.size();
        if (size == 0) {
            if (debug) {
                AbstractFactory.debug("addSection(" + s + ", " + sections + "): first elem");
            }
            sections.add(s);
        } else {
            if (debug) {
                AbstractFactory.debug("addSection(" + s + ", " + sections + "): new elem");
            }
            sections.add(s);
        }
    }

    /**
     * Format the given log record and return the formatted string.
     * <p>
     * The resulting formatted String will normally include a
     * localized and formated version of the LogRecord's message field.
     * The Formatter.formatMessage convenience method can (optionally)
     * be used to localize and format the message field.
     *
     * @param record the log record to be formatted.
     * @return the formatted log record
     */
    public String format(LogRecord record) {
        StringBuffer sb = new StringBuffer();
        String[] ctx = null;
        for (int i = 0; i < pattern.length; i++) {
            int p = pattern[i];
            if (debug) {
                AbstractFactory.debug("format: pattern=" + p + "=" + patternIdToString(p));
            }
            switch (p) {
            case PATTERN_ID_LEVEL:
                sb.append(record.getLevel().getName());
                break;
            case PATTERN_ID_TOPIC:
                sb.append(record.getLoggerName());
                break;
            case PATTERN_ID_DATE:
                format(new Date(record.getMillis()), sb);
                break;
            case PATTERN_ID_THREAD:
                sb.append(Thread.currentThread().getName());
                break;
            case PATTERN_ID_MESSAGE:
                sb.append(simpleFormatter.formatMessage(record));
                break;
            case PATTERN_ID_METHOD:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[1]);
                break;
            case PATTERN_ID_OBJECT:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[0]);
                break;
            case PATTERN_ID_LINE_NUMBER:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[2]);
                break;
            case PATTERN_ID_NEW_LINE:
                sb.append("\n");
                break;
            default:
                if (p < 0) {
                    if (ctx == null) {
                        ctx = getContext();
                    }
                    if (p > (PATTERN_ID_OBJECT - PATTERN_ID_INTERVAL)
                            && p<PATTERN_ID_OBJECT) {
                        p = PATTERN_ID_OBJECT - p;
                        String res = ctx[0];
                        if (p == 1) {// optimize the usual case
                            int idx = res.lastIndexOf('.');
                            if (idx != -1) {
                                res = res.substring(idx + 1);
                            }
                        } else if (p == 0) {
                            //Nothing to do
                        } else {
                            int idx = res.lastIndexOf('.');
                            for(;p>1 && idx != -1;p--) {
                                if (idx != -1) {
                                    idx = res.lastIndexOf('.', idx - 1);
                                }
                            }
                            if (idx != -1) {
                                res = res.substring(idx + 1);
                            }
                        }
                        sb.append(res);
                    }
                } else if (p >= strings.length) {
                    System.err.println("ERROR: String identifier unknown: " + p);
                } else {
                    String str = strings[p];
                    if (str.length() == 1 && Character.isLetter(str.charAt(0))){
                        String value = getLogInfoValue(str.charAt(0));
                        sb.append(value);
                    } else {
                        sb.append(str);
                    }
                }
            }
        }
        if (record.getThrown() != null) {
            StringWriter sw = new StringWriter();
            record.getThrown().printStackTrace(new PrintWriter(sw));
            sb.append(sw.getBuffer());
        }
        return sb.toString();
    }

    /**
     * Calculate the class name, the method name and the line number of the
     * logger user.
     * @return a string array containing 3 String
     * [ "classname", "method name", "line number"]
     * ex: ["com.foo.Bar", "myMethod", "512"]
     */
    public static String[] getContext() {

        Throwable t = new Throwable().fillInStackTrace();
        StringWriter sw = new StringWriter();
        t.printStackTrace(new PrintWriter(sw));
        String m = sw.getBuffer().toString();
        int fin = 0;
        int deb = 0;
        // Remove JDK log methods from the stack trace
        for (int i = 0; i < STACK_TRACE_ITEMS; i++) {
            deb = m.indexOf("\n", deb) + 1;
        }
        boolean isWrapper = true;
        deb = m.indexOf("at ", deb) + 3;

        // Remove wrapper methods (including Monolog)
        while (isWrapper) {
            isWrapper = false;
            for (int i=0; i<LOGWRAPPER.length && !isWrapper; i++) {
                isWrapper |= m.startsWith(LOGWRAPPER[i], deb);
            }
            if (isWrapper) {
                deb = m.indexOf("at ", deb) + 3;
            }
        }
        fin = m.indexOf("\n", deb);
        m = m.substring(deb, fin);

        //m = %C.%M(Toto.java:%L)
        deb = m.indexOf("(");
        fin = m.indexOf(":");
        // If the first '(' is not found, print a warning.
        // If the end is not found, it will use the 'unknown' keyword.
        // Some JVMs optimize methods and then the line number is not available (and fin == -1)
        if (deb == -1) {
            AbstractFactory.warn("Bad stack trace. '(' and ':' expected in the string '" + m
                    + "'. The full stack trace is the following:\n"
                    + sw.getBuffer().toString());
            return new String[]{"","",""};
        }
        String[] res = new String[3];
        res[2] = (fin == -1 ? "unknown" : m.substring(fin + 1, m.length() - 1));
        m = m.substring(0, deb);

        //m = %C.%M
        fin = m.lastIndexOf('.');
        res[0] = m.substring(0, fin);
        res[1] = m.substring(fin + 1);
        return res;
    }

    /**
     Appends to <code>sbuf</code> the time in the format
     "YYYY-MM-DD HH:mm:ss,SSS" for example, "2004-04-28 15:49:37,459"

     @param date the date to format
     @param sbuf the string buffer to write to
     */
    public void format(Date date, StringBuffer sbuf) {
        long now = date.getTime();
        int millis = (int) (now % 1000);

        if ((now - millis) != previousTime) {
            // We reach this point at most once per second
            // across all threads instead of each time format()
            // is called. This saves considerable CPU time.

            calendar.setTime(date);

            int start = sbuf.length();

            int year = calendar.get(Calendar.YEAR);
            sbuf.append(year);
            sbuf.append('-');
            int month = calendar.get(Calendar.MONTH);
            month++;
            if (month < 10) {
                sbuf.append('0');
            }
            sbuf.append(month);
            sbuf.append('-');
            int day = calendar.get(Calendar.DAY_OF_MONTH);
            if (day < 10) {
                sbuf.append('0');
            }
            sbuf.append(day);

            sbuf.append(' ');

            int hour = calendar.get(Calendar.HOUR_OF_DAY);
            if (hour < 10) {
                sbuf.append('0');
            }
            sbuf.append(hour);
            sbuf.append(':');

            int mins = calendar.get(Calendar.MINUTE);
            if (mins < 10) {
                sbuf.append('0');
            }
            sbuf.append(mins);
            sbuf.append(':');

            int secs = calendar.get(Calendar.SECOND);
            if (secs < 10) {
                sbuf.append('0');
            }
            sbuf.append(secs);
            sbuf.append(',');

            // store the time string for next time to avoid recomputation
            sbuf.getChars(start, sbuf.length(), previousTimeWithoutMillis, 0);

            previousTime = now - millis;
        } else {
            sbuf.append(previousTimeWithoutMillis);
        }
        if (millis < 100) {
            sbuf.append('0');
        }
        if (millis < 10) {
            sbuf.append('0');
        }
        sbuf.append(millis);
    }

    public String format(String msg, String levelName, String topic, long time) {
        StringBuffer sb = new StringBuffer();
        String[] ctx = null;
        for (int i = 0; i < pattern.length; i++) {
            int p = pattern[i];
            if (debug) {
                AbstractFactory.debug("format: pattern=" + p + "=" + patternIdToString(p));
            }
            switch (p) {
            case PATTERN_ID_LEVEL:
                sb.append(levelName);
                break;
            case PATTERN_ID_TOPIC:
                sb.append(topic);
                break;
            case PATTERN_ID_DATE:
                format(new Date(time), sb);
                break;
            case PATTERN_ID_THREAD:
                sb.append(Thread.currentThread().getName());
                break;
            case PATTERN_ID_MESSAGE:
                sb.append(msg);
                break;
            case PATTERN_ID_METHOD:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[1]);
                break;
            case PATTERN_ID_OBJECT:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[0]);
                break;
            case PATTERN_ID_LINE_NUMBER:
                if (ctx == null) {
                    ctx = getContext();
                }
                sb.append(ctx[2]);
                break;
            case PATTERN_ID_NEW_LINE:
                sb.append("\n");
                break;
            default:
                if (p < 0) {
                    if (ctx == null) {
                        ctx = getContext();
                    }
                    if (p > (PATTERN_ID_OBJECT - PATTERN_ID_INTERVAL)
                            && p<PATTERN_ID_OBJECT) {
                        p = PATTERN_ID_OBJECT - p;
                        String res = ctx[0];
                        if (p == 1) {// optimize the usual case
                            int idx = res.lastIndexOf('.');
                            if (idx != -1) {
                                res = res.substring(idx + 1);
                            }
                        } else if (p == 0) {
                            //Nothing to do
                        } else {
                            int idx = res.lastIndexOf('.');
                            for(;p>1 && idx != -1;p--) {
                                if (idx != -1) {
                                    idx = res.lastIndexOf('.', idx - 1);
                                }
                            }
                            if (idx != -1) {
                                res = res.substring(idx + 1);
                            }
                        }
                        sb.append(res);
                    }
                } else if (p >= strings.length) {
                    System.err.println("ERROR: String identifier unknown: " + p);
                } else {
                    String str = strings[p];
                    if (str.length() == 1 && Character.isLetter(str.charAt(0))){
                        String value = getLogInfoValue(str.charAt(0));
                        sb.append(value);
                    } else {
                        sb.append(str);
                    }
                }
            }
        }
        return sb.toString();
    }

    /**
     * Gets value of an extension log info
     * @param pattern log info pattern
     * @return log info value
     */
    protected String getLogInfoValue(Character pattern) {
        LogInfo logInfoProvider = Monolog.monologFactory.getLogInfo(pattern);

        if (logInfoProvider != null) {
            return logInfoProvider.getValue();
        } else {
            return "";
        }
    }

}