/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.log4j;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.log4j.Appender;
import org.apache.log4j.Layout;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.spi.ErrorHandler;
import org.apache.log4j.spi.Filter;
import org.apache.log4j.spi.LoggingEvent;
import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.wrapper.common.AbstractFactory;

/**
 * This class is a generic implementation of the Handler interface. This class
 * delegates all calls on a log4j Appender. It is also an Appender interceptor.
 * This class can therefore be referenced into the log4j struture as an
 * Appender.
 * There are three ways to specify the inner Appender:
 * <ul>
 * <li>by the construstor with the appender instance</li>
 * <li>by the setAppender method with the appender instance</li>
 * <li>by the setAttribute method with the appender class name. This method
 * tries to instanciate the class, and initializes the new Appender with all
 * attribute which has been specified before. Even the filters and the layout
 * are memorized.</li>
 *
 * @author Sebastien Chassande-Barrioz
 */
public class GenericHandler implements Appender, Handler {

	/**
	 * This constant can be used to specify the class name of the inner appender
	 */
	public static final String APPENDER_CLASS_NAME_ATTR = "appenderClassName";

	/**
	 * The inner appender
	 */
	protected Appender appender = null;

	/**
	 * The appender name
	 */
	protected String name = null;

	/**
	 * The properties of the appender
	 */
	protected HashMap prop = null;

	protected ArrayList filters = null;
	protected Layout layout = null;

	public GenericHandler() {
	}

	public GenericHandler(String name) {
		this.name = name;
	}

	public GenericHandler(Appender a) {
		appender = a;
		prop = new HashMap();
	}

	public Appender getAppender() {
		return appender;
	}

	public void setAppender(Appender a) {
		appender = a;
		if (layout != null) {
			appender.setLayout(layout);
		}
		if (filters != null) {
			for (Iterator it = filters.iterator(); it.hasNext();) {
				appender.addFilter((Filter) it.next());
			}
		}
	}

	// IMPLEMENTATION OF THE Appender INTERFACE //
	//------------------------------------------//

	public String getName() {
		return name = appender.getName();
	}

	public void setName(String n) {
		name = n;
		if (appender!=null)
			appender.setName(n);
	}

	public String getType() {
		return "generic";
	}

	public String[] getAttributeNames() {
		return (String[]) prop.keySet().toArray(new String[0]);
	}

	public Object getAttribute(String key) {
		return prop.get(key);
	}

	public Object setAttribute(String key, Object value) {
		if (prop == null) {
			prop = new HashMap();
		}
		if (!key.equalsIgnoreCase("activation")) {
			return prop.put(key, value);
		} else if (prop.containsKey(key)) {
		    return null; //already activated
		}

		if (APPENDER_CLASS_NAME_ATTR.equalsIgnoreCase(key)) {
			try {
				Appender _appender=(Appender)Class.forName( (String) value).newInstance();
				if (name!=null){
					_appender.setName(name);
				}
				setAppender(_appender);				
			}
			catch (Exception e) {
				AbstractFactory.warn("setAttribute failed " +e.getMessage());
			}
			
		} 
		String pattern = (String) prop.get(Handler.PATTERN_ATTRIBUTE);
		setLayout(new PatternLayout(PatternConverter.monolog2log4j(pattern)));

		return null;
	}


	// IMPLEMENTATION OF THE Appender INTERFACE //
	//------------------------------------------//

	public void addFilter(Filter newFilter) {
		if (appender != null) {
			appender.addFilter(newFilter);
		}
		else {
			if (filters == null)
				filters = new ArrayList();
			filters.add(newFilter);
		}
	}

	public void clearFilters() {
		if (appender != null) {
			appender.clearFilters();
		}
		else {
			if (filters != null)
				filters.clear();
		}
	}

	public void close() {
		if (appender != null) {
			appender.close();
		}
	}

	public void doAppend(LoggingEvent event) {
		if (appender != null) {
			appender.doAppend(event);
		}
	}

	public void setErrorHandler(ErrorHandler errorHandler) {
		if (appender != null) {
			appender.setErrorHandler(errorHandler);
		}
	}

	public void setLayout(Layout layout) {
		if (appender != null) {
			appender.setLayout(layout);
		}
		else {
			this.layout = layout;
		}
	}

	public Filter getFilter() {
		if (appender != null) {
			return appender.getFilter();
		}
		else if (filters != null && filters.size()>0) {
			return (Filter) filters.get(0);
		}
		return null;
	}

	public ErrorHandler getErrorHandler() {
		return (appender != null ? appender.getErrorHandler():null);
	}

	public Layout getLayout() {
		return layout;
	}

	public boolean requiresLayout() {
		return (appender != null ? appender.requiresLayout() : true);
	}
}
