/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.log4j;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.log4j.Appender;
import org.apache.log4j.Priority;
import org.apache.log4j.spi.LoggingEvent;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.api.Level;
import org.objectweb.util.monolog.api.TopicalLogger;
import org.objectweb.util.monolog.wrapper.common.AbstractFactory;
import org.objectweb.util.monolog.wrapper.common.EnumrationImpl;

/**
 * This class wraps the Logger concept into the log4j world. This class extends
 * therefore the Logger class. This implementation supports
 * <ul>
 * <li>The multiple topic feature by adding a parent list into each node.</li>
 * <li>The inheritance model: The same instance represents the monolog aspect
 * (Logger) and the Log4j aspect (Logger).
 * <li>The delegation model: There are two instances, a Logger object (or a
 * class which inherits from Logger as RootLogger), and a MonologCategory
 * which delegates calls to the first objects.</li>
 * </ul>
 *
 * @author Sebastien Chassande-Barrioz
 */
public class MonologCategory
	extends org.apache.log4j.Logger
	implements TopicalLogger {

	protected boolean enable = true;
	protected final static int DISABLE_OFF = -1;
	protected final static int DISABLE_ON = org.apache.log4j.Level.FATAL.toInt();
	protected OwPriority interPriority = null;
	protected byte depth = 2;

	/**
	 * This field references all parent of this Logger.
	 * key = topic of the current logger
	 * value = its parent either the topic. A parent can be a MonologCategory or
	 * a Logger.
	 */
	protected HashMap topicToparents = null;

	/**
	 * This field references all appenders associated to the current Logger
	 */
	protected ArrayList appenders = null;

	/**
	 * This field references the inner Logger if the delegation was choosen.
	 */
	protected org.apache.log4j.Logger categ = null;

	/**
	 * This field is the class name of the class localized in the top of the
	 * log system layer. The initial value is by default the class name of this
	 * class. Indeed the monolog call directly this class. This information
	 * is used to knwon the caller in the execution stack.
	 */
	private final static String instanceFQN
		= "org.objectweb.util.monolog.wrapper.log4j.MonologCategory";

	/**
	 * This constructor initializes the instance in inheritance mode. It
	 * initializes the instanceFQN, and struture of parents.
	 */
	public MonologCategory(String _initialName) {
		super(_initialName);
		topicToparents = new HashMap();
	}

	/**
	 * This constructor initializes the instance in delegation mode.
	 * @param c is the inner Logger. All calls will be foward to this instance
	 */
	public MonologCategory(org.apache.log4j.Logger c) {
		super(c.getName());
		categ = c;
	}

	/**
	 * It formats a message by adding the object and the method name where the
	 * call to monolog was done.
	 * @param msg is the original message
	 * @param removeTopStack is the number of monolog method call. Indeed this method
	 * fetch a stack trace. This method fetches one line in this stack. The
	 * parameter is the line number in this stack.
	 */
	public static String format(String msg, int removeTopStack) {
		Throwable t = new Throwable().fillInStackTrace();
		StringWriter sw = new StringWriter();
		t.printStackTrace(new PrintWriter(sw));
		String m = sw.getBuffer().toString();

		int deb = -1,fin = 0;

		// take the right line
		deb = -1;
		for (int i = 0; i < (removeTopStack + 1); i++) {
			deb = m.indexOf("\n", deb + 1);
		}

		deb = m.indexOf("at ", deb);
		fin = m.indexOf("\n", deb);
		m = m.substring(deb + 3, fin);

		// remove param
		deb = m.indexOf("(");
		fin = m.indexOf(":");
		m = m.substring(0, deb + 1) + m.substring(fin + 1, m.length());

		// remove package name
		deb = m.indexOf("(");
		int c1 = 0;
		int c2 = 0;
		int current = m.indexOf(".");
		while (current != -1 && current < deb) {
			c1 = c2;
			c2 = current;
			current = m.indexOf(".", current + 1);
		}
		m = m.substring(c1 + 1, m.length());

		return m + ": " + msg;
	}

	private org.apache.log4j.Level convertToLog4jLevel(int value) {
		switch (value) {
		case org.apache.log4j.Level.DEBUG_INT:
			return org.apache.log4j.Level.DEBUG;
		case org.apache.log4j.Level.INFO_INT:
			return org.apache.log4j.Level.INFO;
		case org.apache.log4j.Level.WARN_INT:
			return org.apache.log4j.Level.WARN;
		case org.apache.log4j.Level.ERROR_INT:
			return org.apache.log4j.Level.ERROR;
		case org.apache.log4j.Level.FATAL_INT:
			return org.apache.log4j.Level.FATAL;
		default:
			if (interPriority == null)
				interPriority = new OwPriority(value);
			else
				interPriority.level = value;
			return interPriority;
		}
	}

	// IMPLEMENTATION OF CLASS org.apache.log4j.Logger

	/**
	 Starting from this Logger, search the Logger hierarchy for a
	 non-null priority and return it. Otherwise, return the priority of the
	 root Logger.

	 <p>The Logger class is designed so that this method executes as
	 quickly as possible.
	 */
	public org.apache.log4j.Level getChainedLevel() {
		if (categ != null) {
			return categ.getEffectiveLevel();
		}
		if (level != null) {
			return level;
		}
		org.apache.log4j.Level current = parent.getEffectiveLevel();
		org.apache.log4j.Logger[] cats = (org.apache.log4j.Logger[])
			topicToparents.values().toArray(new org.apache.log4j.Logger[0]);
		for (int i = 0; i < cats.length; i++) {
			org.apache.log4j.Level neo = cats[i].getEffectiveLevel();
			if (neo.isGreaterOrEqual(current)) {
				current = neo;
			}
		}
		return current;
	}

	/**
	 * In inheritance mode this method delegates the treatment to the other
	 * callAppendes methods.
	 * In delegation mode, the call is forwarded on the inner Logger instance.
	 */
	public void callAppenders(LoggingEvent event) {
		if (categ != null) {
			categ.callAppenders(event);
		}
		callAppenders(event, false);
	}

	/**
	 * This method calls all the parent loggers and call its appender either
	 * the followin condition:
	 * <ul>
	 * <li>if the called parameter is equals to true then all parent are call
	 * with the same value, and the logging event are transmitted to the
	 * appenders. The true is return beacause the event must be transmitted</li>
	 *
	 * <li>Or if the current priority is define and the message priority is
	 * equals or greater then the current priority then all parent are call
	 * with the same value, and the logging event are transmitted to the
	 * appenders. The true is return beacause the event must be transmitted</li>
	 *
	 * <li>Else It is needed to check one of the parent is enable for the
	 * logging event. This is done by the recall of each parent. If one of the
	 * parent return true, then the event must be logged.</li>
	 * </ul>
	 * @param event is the logging event
	 * @param called is the boolean which permits to know if the current logger must
	 * call or not its appender without check its priority.
	 * return true is the logging event is enabled in the current logger or one
	 * of its ancestors.
	 */
	public synchronized boolean callAppenders(LoggingEvent event, boolean called) {
		org.apache.log4j.Level l = event.getLevel();
		if (called
			|| (level != null
			&& l.isGreaterOrEqual(level))) {
			for (Enumeration en = getAllAppenders();
				 en.hasMoreElements();) {
				((Appender) en.nextElement()).doAppend(event);
			}
			if (additive) {
				if (parent instanceof MonologCategory)
					((MonologCategory) parent).callAppenders(event, true);
				else
					parent.callAppenders(event);
				for (Iterator it = topicToparents.values().iterator(); it.hasNext();) {
					org.apache.log4j.Logger c = (org.apache.log4j.Logger) it.next();
					if (c instanceof MonologCategory)
						((MonologCategory) c).callAppenders(event, true);
					else
						c.callAppenders(event);
				}
			}
			return true;
		}
		else if (level == null && additive) {
			if (parent instanceof MonologCategory) {
				called |= ((MonologCategory) parent).callAppenders(event, false);
			}
			else if (parent.isEnabledFor(l)) {
				called = true;
				parent.callAppenders(event);
			}
			for (Iterator it = topicToparents.values().iterator(); it.hasNext();) {
				org.apache.log4j.Logger c = (org.apache.log4j.Logger) it.next();
				if (c instanceof MonologCategory) {
					called |= ((MonologCategory) c).callAppenders(event, false);
				}
				else if (c.isEnabledFor(l)) {
					called = true;
					c.callAppenders(event);
				}
			}
			if (called) {
				for (Enumeration en = getAllAppenders();
					 en.hasMoreElements();) {
					((Appender) en.nextElement()).doAppend(event);
				}
			}
		}
		return called;
	}


	// IMPLEMENTATION OF INTERFACE Logger

	/**
	 * Check if the level parameter are not filtered by the logger
	 */
	public boolean isLoggable(int l) {
		if (categ != null) {
			return l >= categ.getEffectiveLevel().toInt();
		}
		return l >= getEffectiveLevel().toInt();
	}

	public boolean isLoggable(Level l) {
		if (categ != null) {
			return l.getIntValue() >= categ.getEffectiveLevel().toInt();
		}
		return l.getIntValue() >= getEffectiveLevel().toInt();
	}

	/**
	 * Is the handler enabled
	 */
	public boolean isOn() {
		return enable;
	}

	/**
	 * Log an object with a specific level. If the level parameter is
	 * loggable the object is handled.
	 */
	public void log(int l, Object o) {
		if (!enable || !isLoggable(l)) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l), (o == null?o:o.toString()), null);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l), (o == null?o:o.toString()), null);
	}

	public void log(Level l, Object o) {
		if (!enable || !isLoggable(l.getIntValue())) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l.getIntValue()),
					(o == null?o:o.toString()), null);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l.getIntValue()),
					(o == null?o:o.toString()), null);
	}

	/**
	 * Log an object and a trowable with a specific level.
	 */
	public void log(int l, Object o, Throwable t) {
		if (!enable || !isLoggable(l)) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l),
					(o == null?o:o.toString()), t);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l),
					(o == null?o:o.toString()), t);
	}

	public void log(Level l, Object o, Throwable t) {
		if (!enable || !isLoggable(l.getIntValue())) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l.getIntValue()),
					(o == null?o:o.toString()), t);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l.getIntValue()),
					(o == null?o:o.toString()), t);
	}

	/**
	 * Log an object and a trowable with a specific level. This method
	 * permits to specify an object instance and a method.
	 */
	public void log(int l, Object o, Object location, Object method) {
		if (!enable || !isLoggable(l)) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l), (o == null?o:o.toString()), null);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l),
				(location == null?"":location.toString())
				+ (method == null?"":method.toString())
				+ (o == null?o:o.toString()),
				null);
	}

	public void log(Level l, Object o, Object location, Object method) {
		if (!enable || !isLoggable(l.getIntValue())) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l.getIntValue()), (o == null?o:o.toString()), null);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l.getIntValue()),
				(location == null?"":location.toString())
				+ (method == null?"":method.toString())
				+ (o == null?o:o.toString()),
				null);
	}

	/**
	 * Log an object and a trowable with a specific level. This method
	 * permits to specify an object instance and a method.
	 */
	public void log(int l, Object o, Throwable t, Object location,
					Object method) {
		if (!enable || !isLoggable(l)) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l), (o == null?o:o.toString()), t);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l),
				(location == null?"":location.toString())
				+ (method == null?"":method.toString())
				+ (o == null?o:o.toString()),
				t);
	}

	public void log(Level l, Object o, Throwable t, Object location,
					Object method) {
		if (!enable || !isLoggable(l.getIntValue())) {
			return;
		}
		if (categ != null)
			categ.log(convertToLog4jLevel(l.getIntValue()), (o == null?o:o.toString()), t);
		else
			forcedLog(instanceFQN, convertToLog4jLevel(l.getIntValue()),
				(location == null?"":location.toString())
				+ (method == null?"":method.toString())
				+ (o == null?o:o.toString()),
				t);
	}

	/**
	 * Enable the handler
	 */
	public void turnOn() {
		enable = true;
	}

	/**
	 * Disable the handler
	 */
	public void turnOff() {
		enable = false;
	}

	// IMPLEMENTATION OF INTERFACE TopicalLogger

	/**
	 * Set the current level of the logger
	 */
	public void setIntLevel(int level) {
		if (level == BasicLevel.INHERIT) {
			if (categ != null)
				categ.setLevel(null);
			else
				super.setLevel(null);
			return;
		}
		if (categ != null)
			categ.setLevel(convertToLog4jLevel(level));
		else
			super.setLevel(convertToLog4jLevel(level));
	}

	public void setLevel(Level l) {
		if (l == null || l.getIntValue() == BasicLevel.INHERIT) {
			if (categ != null)
				categ.setLevel(null);
			else
				super.setLevel(null);
			return;
		}
		if (categ != null)
			categ.setLevel(convertToLog4jLevel(l.getIntValue()));
		else
			super.setLevel(convertToLog4jLevel(l.getIntValue()));
	}

	/**
	 *  Return the current Level of the logger
	 */
	public int getCurrentIntLevel() {
		if (categ != null)
			return categ.getLevel().toInt();
		else
			return (level != null ? level.toInt() : BasicLevel.INHERIT);
	}

	public Level getCurrentLevel() {
		org.apache.log4j.Level p = null;
		if (categ != null)
			p = categ.getLevel();
		else
			p = level;
		if (p != null)
			return LevelImpl.getLevel(p.toInt());
		else
			return BasicLevel.LEVEL_INHERIT;
	}

	/**
	 * Add a handler in the Handler list of the topicalLogger
	 */
	public void addHandler(Handler h) throws Exception {
		if (h instanceof Appender) {
			if (categ != null)
				categ.addAppender((Appender) h);
			else
				super.addAppender((Appender) h);
		}
		else
			throw new UnsupportedOperationException(
				"The type of the handler does not match with this wrapper");
	}

	/**
	 * Add a topic to the topicalLogger
	 */
	public void addTopic(String topic) throws Exception {
		if (categ == null) {
			Object p = parent;
			String n = name;
			org.apache.log4j.Logger.getLogger(topic, new BetaCF(this));
			topicToparents.put(topic, parent);
			parent = (org.apache.log4j.Logger) p;
			name = n;
		}
	}

	public Handler[] getHandler() {
		ArrayList al = new ArrayList();
		if (categ != null) {
			for (Enumeration en = categ.getAllAppenders(); en.hasMoreElements();) {
				Appender a = (Appender) en.nextElement();
				if (a instanceof Handler) {
					al.add(a);
				}
				else {
					al.add(new GenericHandler(a));
				}
			}
		}
		else {
			for (Enumeration en = getAllAppenders(); en.hasMoreElements();) {
				Appender a = (Appender) en.nextElement();
				if (a instanceof Handler) {
					al.add(a);
				}
				else {
					al.add(new GenericHandler(a));
				}
			}
		}
		return (Handler[]) al.toArray(new Handler[0]);
	}

	public Handler getHandler(String hn) {
		Appender a = null;
		if (categ != null) {
			a = categ.getAppender(hn);
		}
		else {
			a = getAppender(hn);
		}
		if (a instanceof Handler) {
			return (Handler) a;
		}
		else {
			return new GenericHandler(a);
		}
	}

	public void removeAllHandlers() throws Exception {
		if (categ != null) {
			categ.removeAllAppenders();
		}
		else {
			removeAllAppenders();
		}
	}

	/**
	 * Returns the list of the different names of the topicalLogger
	 */
	public Enumeration getTopics() {
		return new EnumrationImpl(
			topicToparents.keySet().toArray(new String[0]));
	}

	/**
	 * Returns the list of the different names of the topicalLogger
	 */
	public String[] getTopic() {
		String[] res = null;
		if (categ != null) {
			res = new String[1];
			res[0] = AbstractFactory.getTopicWithoutPrefix(categ.getName());
            return res;
		}
        res = new String[topicToparents.size() + 1];
        res[0] = AbstractFactory.getTopicWithoutPrefix(name);
        int i=1;
        for (Iterator it = topicToparents.keySet().iterator(); it.hasNext();) {
            res[i] = AbstractFactory.getTopicWithoutPrefix((String) it.next());
            i++;
        }
		return res;
	}

	/**
	 * Remove a handler from the Handler list of the topicalLogger
	 */
	public void removeHandler(Handler h) throws Exception {
		if (h instanceof Appender) {
			if (categ != null) {
				categ.removeAppender((Appender) h);
				if (h instanceof GenericHandler) {
					categ.removeAppender(((GenericHandler) h).getAppender());
				}
			}
			else {
				super.removeAppender((Appender) h);
				if (h instanceof GenericHandler) {
					super.removeAppender(((GenericHandler) h).getAppender());
				}
			}
		}
		else
			throw new UnsupportedOperationException(
				"The type of the handler does not match with this wrapper");
	}

	/**
	 * Remove a topic from the topicalLogger
	 */
	public void removeTopic(String topic) throws Exception {
		if (categ == null)
			topicToparents.remove(topic);
	}

	// IMPLEMENTATION OF THE Handler INTERFACE //
	//-----------------------------------------//
	public void setName(String n) {
		name = n;
	}

	public String getType() {
		return "logger";
	}

	public String[] getAttributeNames() {
		return new String[0];
	}

	public Object getAttribute(String name) {
		return null;
	}

	public Object setAttribute(String name, Object value) {
		return null;
	}

	// INNER CLASSES //
	//---------------//
	public static class OwPriority extends org.apache.log4j.Level {

		protected int level = 10000;

		public OwPriority(int l) {
			super(l, "INTER", l);
			this.level = l;
		}

		public boolean isGreaterOrEqual(Priority pr) {
			return level >= pr.toInt();
		}
	}

	public static class BetaCF implements org.apache.log4j.spi.LoggerFactory {
		MonologCategory mc = null;

		public BetaCF(MonologCategory mc) {
			this.mc = mc;
		}

		public org.apache.log4j.Logger makeNewLoggerInstance(String name) {
			return mc;
		}
	}

}
