/**
 * Copyright (c) 2005-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see http://www.gnu.org/licenses/
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.engine.sampleclient;

import java.io.File;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.activation.DataHandler;
import javax.jbi.messaging.DeliveryChannel;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;

import org.ow2.petals.commons.log.FlowAttributes;
import org.ow2.petals.commons.log.FlowAttributesExchangeHelper;
import org.ow2.petals.commons.log.Level;
import org.ow2.petals.commons.log.PetalsExecutionContext;
import org.ow2.petals.component.framework.logger.ConsumeExtFlowStepEndLogData;
import org.ow2.petals.component.framework.logger.ConsumeExtFlowStepFailureLogData;
import org.ow2.petals.engine.sampleclient.gui.Console;

/**
 * Message listener of the SampleClient component.
 * @author Adrien Louis - EBM WebSourcing
 * @author ddesjardins - EBM WebSourcing
 */
public class SampleClientListener implements Runnable {

    private DeliveryChannel channel;

    private MessageExchange messageExchange;

    private boolean running = true;

    private Console console;

    private Logger logger;

    /**
     * Creates a new instance of SampleClientListener
     * 
     * @param channel
     * @param c
     * @param log
     */
    public SampleClientListener(DeliveryChannel channel, Console c, Logger log) {
        super();
        this.channel = channel;
        console = c;
        logger = log;
    }

    @Override
    public void run() {

        while (running) {
            try {
                messageExchange = channel.accept();
                if (messageExchange != null) {
                    process(messageExchange);
                }
            } catch (MessagingException e) {
                // search for an interrupted exception
                Throwable cause = e;
                while (cause != null) {
                    logger.info("Exception when accepting messages :" + cause.getClass().getName()
                            + " " + cause.getMessage());
                    cause = cause.getCause();
                    if (cause instanceof InterruptedException) {
                        logger.severe("Thread interrupted, stop listening.");
                        running = false;
                    }
                }
                logger.severe(e.getMessage());
            }
        }

    }

    /**
     * Process the received message exchange
     * 
     * @param msg
     */
    public void process(MessageExchange msg) {
        logger.log(Level.INFO, "Processing received message " + msg.getExchangeId());

        String response = null;
        Map<String, String> properties = null;

        final FlowAttributes flowAttributes = FlowAttributesExchangeHelper.getFlowAttributes(msg);
        // necessary to put the flow instance id in the context for the log handler
        PetalsExecutionContext.putFlowAttributes(flowAttributes);

        String flowInstanceId = null;
        String flowStepId = null;
        if (flowAttributes != null) {
            flowInstanceId = flowAttributes.getFlowInstanceId();
            flowStepId = flowAttributes.getFlowStepId();
        }

        if (ExchangeStatus.DONE.equals(msg.getStatus())) {
            response = "receive a DONE status.";
        } else if (ExchangeStatus.ERROR.equals(msg.getStatus())) {
            response = "receive an ERROR status :" + msg.getError().getMessage();

            logger.log(
                    Level.MONIT,
                    "",
                    new ConsumeExtFlowStepFailureLogData(flowInstanceId, flowStepId, "An error occurs"));
        } else {
            try {
                if (msg.getMessage("OUT") != null && (msg.getMessage("OUT").getContent() != null)) {
                    response = Utils.createString(msg.getMessage("OUT").getContent());
                    if (!msg.getMessage("OUT").getAttachmentNames().isEmpty()) {
                        logger.log(Level.INFO, msg.getMessage("OUT").getAttachmentNames().size()
                                + " attachment(s) found : "
                                + msg.getMessage("OUT").getAttachmentNames());
                        String folder = console.saveFolder();

                        for (Object object : msg.getMessage("OUT").getAttachmentNames()) {
                            String attachmentName = (String) object;
                            FileOutputStream fos = new FileOutputStream(folder + File.separator
                                    + attachmentName);

                            DataHandler data = msg.getMessage("OUT").getAttachment(attachmentName);
                            data.writeTo(fos);
                            fos.flush();
                            fos.close();
                        }
                    }

                    logger.log(
                            Level.MONIT,
                            "",
                            new ConsumeExtFlowStepEndLogData(flowInstanceId, flowStepId));

                } else if (msg.getFault() != null) {
                    response = "FAULT : " + Utils.createString(msg.getFault().getContent());
                    
                    logger.log(
                            Level.MONIT,
                            "",
                            new ConsumeExtFlowStepFailureLogData(flowInstanceId, flowStepId, "A fault occurs"));
                }

                @SuppressWarnings("unchecked")
                final Set<String> names = msg.getPropertyNames();
                if (names.size() > 0) {
                    properties = new HashMap<String, String>();
                    for (final String name : names) {
                        properties.put(name, String.valueOf(msg.getProperty(name)));
                    }
                }

                msg.setStatus(ExchangeStatus.DONE);

                channel.send(msg);
            } catch (Exception e) {
                logger.log(Level.SEVERE, e.getClass() + ":" + e.getMessage());
                response = e.getMessage();
            }

        }

        if (properties != null) {
            console.setResponse(response, properties);
        } else {
            console.setResponse(response);
        }
    }

    /**
     * Stop processing
     */
    public void stopProcessing() {
        this.running = false;
    }
}
