/**
 * Copyright (c) 2008-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see http://www.gnu.org/licenses/
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.jsr181;

import java.util.logging.Logger;

import javax.jbi.messaging.MessagingException;
import javax.xml.namespace.QName;

import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfOperation.MEPPatternConstants;
import org.ow2.petals.component.framework.api.configuration.SuConfigurationParameters;
import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.api.message.Exchange;

/**
 * @author Christophe HAMERLING - EBM WebSourcing
 */
public interface PetalsMessageSender {

	/**
	 * Creates an exchange.
	 *
	 * @param mep the invocation MEP (not null)
	 * @return the created exchange
	 * @throws MessagingException
	 * @throws PEtALSCDKException
	 * @since 1.1.0
	 */
	public Exchange createExchange(MEPPatternConstants mep) throws MessagingException, PEtALSCDKException;

	/**
     * Creates an exchange.
     *
     * @param interfaceName the target interface name (use null as wild card)
     * @param serviceName the target service name (use null as wild card)
     * @param endpointName the target end-point name (use null as wild card)
     * @param operationName the target operation name (can be null but not recommended)
     * @param mep the invocation MEP (not null)
     * @return the created exchange
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @since 1.3.0
     */
    public Exchange createExchange(QName interfaceName, QName serviceName, String endpointName, QName operationName,
            MEPPatternConstants mep) throws MessagingException, PEtALSCDKException;

	/**
	 * Sends an exchange in the bus (synchronous sending).
	 * @param exchange
	 * @return
	 * @throws MessagingException
	 * @since 1.1.0
	 */
	public boolean sendSync(Exchange exchange) throws MessagingException;

	/**
	 * Sends an exchange in the bus.
	 * @param exchange
	 * @throws MessagingException
	 * @since 1.1.0
	 */
	public void send(Exchange exchange) throws MessagingException;

	/**
	 * @return the configuration extensions
	 */
    public SuConfigurationParameters getExtensions();

	/**
	 * <p>Returns a logger to be used in the service unit (SU) classes.</p>
	 * <p>
	 * The created logger is a child of the JSR-181 components's logger.
	 * About the new logger's name, the following convention is taken:
	 * </p>
	 * <pre>
	 * Petals.Container.Components.petals-se-jsr181.interfaceName.serviceName.edptName
	 * </pre>
	 * <p>
	 * where:
	 * </p>
	 * <ul>
	 *     <li><b>interfaceName</b> is the local part of the SU's interface name.</li>
	 *     <li><b>serviceName</b> is the local part of the SU's service name.</li>
	 *     <li><b>edptName</b> is the SU's end-point name.</li>
	 * </ul>
	 * <p>
	 * This allows to set precisely the log levels in the loggers configuration.
	 * </p>
	 * <p>
	 * Notice that these logger names can be ugly in case of generated end-point names.
	 * </p>
	 *
	 * @return a logger (not null)
	 * @since 1.3.0
	 */
	public Logger getComponentChildLogger();
}
