package org.jvnet.basicjaxb.plugin.fluentapi;

import static java.lang.String.format;
import static org.jvnet.basicjaxb.plugin.fluentapi.Customizations.IGNORED_ELEMENT_NAME;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.FLUENT_COLLECTION_SETTER;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.FLUENT_LIST_SETTER;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.FLUENT_SETTER;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.GETTER_METHOD_PREFIX;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.GETTER_METHOD_PREFIX_LEN;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.PARAMETERIZED_LIST_PREFIX;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.SETTER_METHOD_PREFIX;
import static org.jvnet.basicjaxb.plugin.fluentapi.FluentMethodType.SETTER_METHOD_PREFIX_LEN;
import static org.jvnet.basicjaxb.plugin.util.FieldOutlineUtils.filter;
import static org.jvnet.basicjaxb.util.LocatorUtils.toLocation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.jvnet.basicjaxb.plugin.AbstractParameterizablePlugin;
import org.jvnet.basicjaxb.plugin.AbstractPlugin;
import org.jvnet.basicjaxb.plugin.Customizations;
import org.jvnet.basicjaxb.plugin.CustomizedIgnoring;
import org.jvnet.basicjaxb.plugin.Ignoring;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;

import com.sun.codemodel.JClass;
import com.sun.codemodel.JCodeModel;
import com.sun.codemodel.JDefinedClass;
import com.sun.codemodel.JMethod;
import com.sun.codemodel.JMod;
import com.sun.codemodel.JType;
import com.sun.codemodel.JVar;
import com.sun.tools.xjc.model.CClassInfo;
import com.sun.tools.xjc.model.CPropertyInfo;
import com.sun.tools.xjc.model.Model;
import com.sun.tools.xjc.outline.ClassOutline;
import com.sun.tools.xjc.outline.FieldOutline;
import com.sun.tools.xjc.outline.Outline;

/**
 * <p>Support a fluent api in addition to the default (JavaBean) setter
 * methods.</p>
 * 
 * <p>
 * The initial idea is simply to add a fluent method to the generated class for
 * every "set*" method encountered, with the only functional difference of
 * returning the class instance, instead of void.
 * </p>
 * 
 * <p>
 * <strong>Enhancement on 11 June 2006:</strong><br>
 * Provide fluent setter api for Lists, with support of variable arguments.
 * </p>
 * 
 * <p>
 * This enhancement was suggested by Kenny MacLeod &lt;kennym@kizoom.com&gt;, and
 * endorsed by Kohsuke Kawaguchi &lt;Kohsuke.Kawaguchi@sun.com&gt;. Here is quoted
 * from the original request:
 * </p>
 * 
 * <p>
 * By default, XJC represents Lists by generating a getter method, but no
 * setter. This is impossible to chain with fluent-api. How about the plugin
 * generates a useXYZ() method for List properties, taking as it's parameters a
 * vararg list. For example:
 * </p>
 * 
 * <pre>
 * // This method is generated by vanilla XJC
 * public List&lt;OtherType> getMyList()
 * {
 *     if (myList == null)
 *         myList = new ArrayList&lt;OtherType>();
 *     return myList;
 * }
 *
 * // This method is generated by XfluentAPI
 * public MyClass useMyList(OtherType... values)
 * {
 *     if (values != null)
 *     {
 *         for (OtherType value : values)
 *             getMyList().add(value);
 *     }
 *     return this;
 * }
 * </pre>
 * 
 * <p>
 * <strong>Enhancement on 11 Oct 2008:</strong><br>
 * Provide fluent setter api for Lists, with support of Collection argument in
 * addition to varargs arguments. This enhancement was suggested by Alex Wei
 * &lt;ozgwei@dev.java.net&gt; with patch submitted. See
 * <a href="https://jaxb2-commons.dev.java.net/issues/show_bug.cgi?id=12">Jira Issue 12</a>
 * for more details.
 * </p>
 * 
 * @author Hanson Char
 */
public class FluentApiPlugin extends AbstractParameterizablePlugin
{
	/** Name of Option to enable this plugin. */
	private static final String OPTION_NAME = "XfluentAPI";
	/** Description of Option to enable this plugin. */
	private static final String OPTION_DESC = "enable Fluent API, method chaining, for generated code";

	@Override
	public String getOptionName()
	{
		return OPTION_NAME;
	}

	@Override
	public String getUsage()
	{
		return format(USAGE_FORMAT, OPTION_NAME, OPTION_DESC);
	}

	private Ignoring ignoring = new CustomizedIgnoring
	(
		IGNORED_ELEMENT_NAME,
		Customizations.IGNORED_ELEMENT_NAME,
		Customizations.GENERATED_ELEMENT_NAME
	);
	public Ignoring getIgnoring()
	{
		return ignoring;
	}
	public void setIgnoring(Ignoring ignoring)
	{
		this.ignoring = ignoring;
	}
	
	@Override
	public Collection<QName> getCustomizationElementNames()
	{
		return Arrays.asList
		(
			IGNORED_ELEMENT_NAME,
			Customizations.IGNORED_ELEMENT_NAME,
			Customizations.GENERATED_ELEMENT_NAME
		);
	}
	
	// Plugin Processing
	
	@Override
	protected void beforeRun(Outline outline) throws Exception
	{
		if ( isInfoEnabled() )
		{
			StringBuilder sb = new StringBuilder();
			sb.append(LOGGING_START);
			sb.append("\nParameters");
			sb.append("\n  Verbose.: " + isVerbose());
			sb.append("\n  Debug...: " + isDebug());
			info(sb.toString());
		}
	}
	
	@Override
	protected void afterRun(Outline outline) throws Exception
	{
		if ( isInfoEnabled() )
		{
			StringBuilder sb = new StringBuilder();
			sb.append(LOGGING_FINISH);
			sb.append("\nResults");
			sb.append("\n  HadError.: " + hadError(outline.getErrorReceiver()));
			info(sb.toString());
		}
	}
	
	/**
	 * <p>
	 * Run the plugin with and XJC {@link Outline}.
	 * </p>
	 * 
	 * <p>
	 * Run an XJC plugin to add or modify the XJC {@link Outline}. An {@link Outline}
	 * captures which code is generated for which model component. A {@link Model} is
	 * a schema language neutral representation of the result of a schema parsing. XJC
	 * uses this model to turn this into a series of Java source code.
	 * </p>
	 * 
     * <p>
     * <b>Note:</b> This method is invoked only when a plugin is activated.
     * </p>
	 *
     * @param outline
     *      This object allows access to various generated code.
     * 
     * @return
     *      If the add-on executes successfully, return true.
     *      If it detects some errors but those are reported and
     *      recovered gracefully, return false.
     *
     * @throws Exception
     *      This 'run' method is a call-back method from {@link AbstractPlugin}
     *      and that method is responsible for handling all exceptions. It reports
     *      any exception to {@link ErrorHandler} and converts the exception to
     *      a {@link SAXException} for processing by {@link com.sun.tools.xjc.Plugin}.
	 */
	@Override
	public boolean run(Outline outline)
		throws Exception
	{
		// The Void class is an uninstantiable placeholder class to hold a reference
		// to the Class object representing the Java keyword void.
		final JType voidType = outline.getCodeModel().VOID;
		
		// Process every POJO class generated by JAXB.
		for (ClassOutline classOutline : outline.getClasses())
		{
			if (!getIgnoring().isIgnored(classOutline))
				processClassOutline(classOutline, voidType);
		}
		
		return !hadError(outline.getErrorReceiver());
	}

	private void processClassOutline(ClassOutline classOutline, final JType voidType)
	{
		final JDefinedClass targetClass = classOutline.implClass;
		
		Map<String, FieldOutline> filteredFieldMap = new HashMap<>();
		putFieldOutlines(filteredFieldMap, getSuperClassFilteredFields(classOutline));
		putFieldOutlines(filteredFieldMap, filter(classOutline.getDeclaredFields(), getIgnoring()));
		
		// Traverse the class hierarchy
		Collection<FluentMethodInfo> fluentMethodInfoList = new ArrayList<FluentMethodInfo>();
		Set<String> originalMethodNames = new HashSet<String>();
		boolean isOverride = false;
		while ( classOutline != null )
		{
			JDefinedClass implClass = classOutline.implClass;
			String implClassName = implClass.name();
			String implPackageName = implClass.getPackage().name();
			String implFullClassName = implPackageName + "." +implClassName;
			
			// Collect the methods we are interested in but defer the respective fluent
			// methods creation to avoid ConcurrentModificationException
			for (JMethod originalMethod : implClass.methods())
			{
				// Skip processed original method name.
				String originalMethodName = originalMethod.name();
				if ( !originalMethodNames.contains(originalMethodName) )
				{
					// Process filtered (non-ignored) field outline.
					FieldOutline fieldOutline = filteredFieldMap.get(implFullClassName + "." + originalMethodName);
					if ( fieldOutline != null )
					{
						if (isSetterMethod(originalMethod, voidType))
							fluentMethodInfoList.add(new FluentMethodInfo(originalMethod, FLUENT_SETTER, isOverride));
						else if (isListGetterMethod(originalMethod))
						{
							fluentMethodInfoList.add(new FluentMethodInfo(originalMethod, FLUENT_LIST_SETTER, isOverride));
							// Originally proposed by Alex Wei ozgwei@dev.java.net:
							// https://jaxb2-commons.dev.java.net/issues/show_bug.cgi?id=12
							fluentMethodInfoList.add(new FluentMethodInfo(originalMethod, FLUENT_COLLECTION_SETTER, isOverride));
						}
						
						CPropertyInfo fieldInfo = fieldOutline.getPropertyInfo();
						trace("{}, processClassOutline; Class={}, Field={}",
							toLocation(fieldInfo.getLocator()), targetClass.name(), fieldInfo.getName(false));
					}
					// Cache processed original method name.
					originalMethodNames.add(originalMethodName);
				}
			}
			
			// Let's climb up the class hierarchy; from here,
			// any additional FluentMethodInfo(s) are overrides.
			classOutline = classOutline.getSuperClass();
			if (classOutline != null)
				isOverride = true;
		}
		
		// Generate a respective fluent method for each setter method
		// For each FluentMethodInfo, invoke a FluentMethodType on the target class.
		for (FluentMethodInfo fluentMethodInfo : fluentMethodInfoList)
			fluentMethodInfo.createFluentMethod(targetClass);
		
		debug("{}, processClassOutline; Class={}", toLocation(targetClass.metadata), targetClass.name());
	}

	private void putFieldOutlines(Map<String, FieldOutline> fieldMethodMap, FieldOutline[] classFields)
	{
		for ( FieldOutline fieldOutline : classFields)
		{
			CPropertyInfo fieldInfo = fieldOutline.getPropertyInfo();
			CClassInfo fieldParent = (CClassInfo) fieldInfo.parent();
			String fullClassName = fieldParent.getName();
			String fieldPublicName = fieldInfo.getName(true);
			String fieldMethodName = null;
			if ( fieldInfo.isCollection() )
				fieldMethodName = GETTER_METHOD_PREFIX + fieldPublicName;
			else
				fieldMethodName = SETTER_METHOD_PREFIX + fieldPublicName;
			fieldMethodMap.put(fullClassName + "." + fieldMethodName, fieldOutline);
		}
	}

	/**
	 * Returns true if the given method is a public, non-static, setter method
	 * that follows the JavaBean convention; false otherwise. The setter method
	 * can either be a simple property setter method or an indexed property
	 * setter method.
	 * 
	 * @param method The method is examine.
	 * @param setterType The return type of a setter method is expected to be void.
	 * 
	 * @return True if the given method is a public, non-static, setter method
	 *         that follows the JavaBean convention; otherwise, false.
	 */
	private boolean isSetterMethod(JMethod method, final JType setterType)
	{
		// Return type of a setter method is expected to be void.
		if (method.type() == setterType)
		{
			JVar[] jvars = method.listParams();
			switch (jvars.length)
			{
				case 2:
					// could be an indexed property setter method.
					// if so, the first argument must be the index (a primitive int).
					if (!isInt(jvars[0].type()))
						return false;
					// drop thru.
				case 1:
					// or could be a simple property setter method
					int mods = method.mods().getValue();
					if ((mods & JMod.STATIC) == 0 && (mods & JMod.PUBLIC) == 1)
					{
						String methodName = method.name();
						return methodName.length() > SETTER_METHOD_PREFIX_LEN
							&& methodName.startsWith(SETTER_METHOD_PREFIX);
					}
					break;
			}
		}
		return false;
	}

	/**
	 * Returns true if the given method is a public, non-static, getter method
	 * that returns a List<T>; otherwise, false.
	 * 
	 * @param method The given method to examine.
	 * 
	 * @return True if the given method is a public, non-static, getter method
	 *         that returns a List<T>; otherwise, false.
	 */
	private boolean isListGetterMethod(JMethod method)
	{
		int mods = method.mods().getValue();
		
		// check if it is a non-static public method
		if ((mods & JMod.STATIC) == 1 || (mods & JMod.PUBLIC) == 0)
			return false;
		
		// See if the method name looks like a getter method
		String methodName = method.name();
		if (methodName.length() <= GETTER_METHOD_PREFIX_LEN || !methodName.startsWith(GETTER_METHOD_PREFIX))
			return false;
		
		// A list getter method will have no argument.
		if (method.listParams().length > 0)
			return false;
		
		// See if the return type of the method is a List<T>
		JType jtype = method.type();
		if (jtype instanceof JClass)
		{
			JClass jclass = JClass.class.cast(jtype);
			List<JClass> typeParams = jclass.getTypeParameters();
			
			if (typeParams.size() != 1)
				return false;
			
			return jclass.fullName().startsWith(PARAMETERIZED_LIST_PREFIX);
		}
		
		return false;
	}

	/**
	 * Is the given type a primitive integer.
	 * 
	 * @param type The type to examine.
	 * 
	 * @return True if the given type is a primitive integer; otherwise, false.
	 */
	private boolean isInt(JType type)
	{
		JCodeModel codeModel = type.owner();
		return type.isPrimitive() && codeModel.INT.equals(JType.parse(codeModel, type.name()));
	}


	/**
	 * Retrieve a List of the fields of each ancestor class. I walk up the class
	 * hierarchy until I reach a class that isn't being generated by JAXB. The filtered
	 * fields from each class in the hierarchy is added to the beginning of the result
	 * list in order to preserve the <code>super(...)</code> method's parameter order.
	 */
	protected FieldOutline[] getSuperClassFilteredFields(final ClassOutline classOutline)
	{
		List<FieldOutline> fieldOutlineList = new LinkedList<FieldOutline>();

		// Walk up the class hierarchy.
		for ( ClassOutline sco = classOutline.getSuperClass(); sco != null; sco = sco.getSuperClass() )
		{
			FieldOutline[] superFieldOutlines = filter(sco.getDeclaredFields(), getIgnoring());
			// Accumulate the super-field outline array into a list.
			List<FieldOutline> superFieldOutlineList = new ArrayList<FieldOutline>();
			for ( FieldOutline superFieldOutline : superFieldOutlines )
				superFieldOutlineList.add(superFieldOutline);
			// Add the list of super-field outlines to the start of
			// the linked cumulative field outline list.
			fieldOutlineList.addAll(0, superFieldOutlineList);
		}

		// Return an array of the cumulative field outline list from the super hierarchy.
		return fieldOutlineList.toArray(new FieldOutline[fieldOutlineList.size()]);
	}
}
