/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.picketlink.trust.jbossws.handler;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.security.auth.Subject;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.jboss.logging.Logger;
import org.jboss.security.SecurityContext;
import org.jboss.wsf.common.handler.GenericSOAPHandler;
import org.picketlink.identity.federation.bindings.jboss.subject.PicketLinkPrincipal;
import org.picketlink.identity.federation.core.wstrust.SamlCredential;
import org.picketlink.trust.jbossws.Constants;
import org.picketlink.trust.jbossws.SAML2Constants;
import org.picketlink.trust.jbossws.Util;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * A SAMLv2 WS handler.
 * 
 * @author <a href="mmoyses@redhat.com">Marcus Moyses</a>
 * @author <a href="alessio.soldano@jboss.com">Alessio Soldano</a>
 * @version $Revision: 1 $
 */
public class SAML2Handler extends GenericSOAPHandler
{

   protected Logger log = Logger.getLogger(this.getClass());
   
   private static Set<QName> headers;

   static
   {
      HashSet<QName> set = new HashSet<QName>();
      set.add(Constants.WSSE_HEADER_QNAME);
      headers = Collections.unmodifiableSet(set);
   }

   public Set<QName> getHeaders()
   {
      //return a collection with just the wsse:Security header to pass the MustUnderstand check on it
      return headers;
   }
   
   /**
    * Retrieves the SAML assertion from the SOAP payload and lets invocation go to JAAS for validation.
    */
   protected boolean handleInbound(MessageContext msgContext)
   {
      SOAPMessageContext ctx = (SOAPMessageContext) msgContext;
      SOAPMessage soapMessage = ctx.getMessage();
      
      // retrieve the assertion
      Document document = soapMessage.getSOAPPart();
      Element soapHeader = Util.findOrCreateSoapHeader(document.getDocumentElement());
      Element assertion = Util.findElement(soapHeader, new QName(SAML2Constants.SAML2_ASSERTION_URI, "Assertion"));
      if (assertion != null)
      {
         SamlCredential credential = new SamlCredential(assertion);
         if (log.isTraceEnabled())
         {
            log.trace("Assertion included in SOAP payload:");
            log.trace(credential.getAssertionAsString());
         }
         Element subject = Util.findElement(assertion, new QName(SAML2Constants.SAML2_ASSERTION_URI, "Subject"));
         Element nameID = Util.findElement(subject, new QName(SAML2Constants.SAML2_ASSERTION_URI, "NameID"));
         String username = nameID.getTextContent();
         // set SecurityContext
         Subject s = new Subject();
         SecurityContext sc = SecurityActions.createSecurityContext(new PicketLinkPrincipal(username), credential, s);
         SecurityActions.setSecurityContext(sc);
      }
      
      return true;
   }

   /**
    * It expects a {@link Element} assertion as the value of the {@link SAML2Constants#SAML2_ASSERTION_PROPERTY} property.
    * This assertion is then included in the SOAP payload.
    */
   protected boolean handleOutbound(MessageContext msgContext)
   {
      SOAPMessageContext ctx = (SOAPMessageContext) msgContext;
      SOAPMessage soapMessage = ctx.getMessage();
      
      // retrieve assertion
      Element assertion = (Element) ctx.get(SAML2Constants.SAML2_ASSERTION_PROPERTY);
      
      // add wsse header
      Document document = soapMessage.getSOAPPart();
      Element soapHeader = Util.findOrCreateSoapHeader(document.getDocumentElement());
      try
      {
         Element wsse = getSecurityHeaderElement(document);
         wsse.setAttributeNS(soapHeader.getNamespaceURI(), soapHeader.getPrefix() + ":mustUnderstand", "1");
         if (assertion != null)
         {
            // add the assertion as a child of the wsse header
            // check if the assertion element comes from the same document, otherwise import the node
            if (document != assertion.getOwnerDocument())
            {
               wsse.appendChild(document.importNode(assertion, true));
            }
            else
            {
               wsse.appendChild(assertion);
            }
         }
         soapHeader.insertBefore(wsse, soapHeader.getFirstChild());
      }
      catch (Exception e)
      {
         log.error(e);
         return false;
      }
      
      return true;
   }
   
   private Element getSecurityHeaderElement(Document document)
   {
      Element element = document.createElementNS(Constants.WSSE_NS, Constants.WSSE_HEADER);
      Util.addNamespace(element, Constants.WSSE_PREFIX, Constants.WSSE_NS);
      Util.addNamespace(element, Constants.WSU_PREFIX, Constants.WSU_NS);
      Util.addNamespace(element, Constants.XML_ENCRYPTION_PREFIX, Constants.XML_SIGNATURE_NS);
      return element;
   }
   
}
