/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.picketlink.identity.federation.bindings.wildfly.sp;

import io.undertow.server.HttpServerExchange;
import io.undertow.server.handlers.form.FormData;
import io.undertow.servlet.handlers.ServletRequestContext;
import io.undertow.util.AttachmentKey;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.util.function.Function;

/**
 * Helper class. It makes easier to use of {@see SPFormAuthenticationRequestWrapper}.
 * If different FormData for request has to be used, new FormData is saved as an attachment to exchange.
 * SPFormAuthenticationRequestWrapper then returns parameters based also on this new FormData.
 *
 * @Author Jiri Ondrusek
 */
public class SPFormAuthrenticationRequestUtil {

    public static final AttachmentKey<SPFAMFormDataWrapper> FORM_DATA_ATTACHMENT_KEY = AttachmentKey.create(SPFAMFormDataWrapper.class);

    /**
     *
     * @param buffer
     * @param formData
     * @param exchange
     * @param  contentLength
     */
    public static void store(byte[] buffer, FormData formData, HttpServerExchange exchange, int contentLength) {
        exchange.putAttachment(SPFormAuthrenticationRequestUtil.FORM_DATA_ATTACHMENT_KEY, new SPFAMFormDataWrapper(formData, buffer, contentLength));
    }


    /**
     *
     * @param exchange
     */
    public static byte[] getStoredBuffer(HttpServerExchange exchange) {
        SPFAMFormDataWrapper formData = exchange.getAttachment(FORM_DATA_ATTACHMENT_KEY);
        if(formData != null) {
            return formData.getBuffer();
        }
        return null;
    }

    /**
     *
     * @param exchange
     */
    public static int getStoredBufferLength(HttpServerExchange exchange) {
        SPFAMFormDataWrapper formData = exchange.getAttachment(FORM_DATA_ATTACHMENT_KEY);
        if(formData != null) {
            return formData.getContentLength();
        }
        return -1;
    }

    /**
     * Wrap request into {@see SPFormAuthenticationRequestWrapper}.
     *
     * @param exchange
     * @return Either SPFormAuthenticationRequestWrapper or originial unmodified request.
     */
    public static HttpServletRequest wrapRequest(HttpServerExchange exchange) {
        ServletRequestContext servletRequestContext = exchange.getAttachment(ServletRequestContext.ATTACHMENT_KEY);
        HttpServletRequest request = (HttpServletRequest) servletRequestContext.getServletRequest();

        if(request instanceof SPFormAuthenticationRequestWrapper) {
            return request;
        }

        //try to get parsed dara from SPFormAuthenticationMechanism
        SPFAMFormDataWrapper formData = exchange.getAttachment(FORM_DATA_ATTACHMENT_KEY);
        if (formData != null) {
            return new SPFormAuthenticationRequestWrapper(request, formData.getFormData());
        }

        return request;
    }

    /**
     * Holder for {@see FormData}, to be add as attachment into exchange.
     * (Different class then FormData has to be used, because attachment with FormData may already exist)
     */
    public static class SPFAMFormDataWrapper {
        private final FormData formData;
        private final byte[] buffer;
        //length has to be saved too, because exchangfe length is in long, and casting long to int could create more errors in future
        private final int contentLength;

        public SPFAMFormDataWrapper(FormData formData, byte[] buffer, int contentLength) {
            this.formData = formData;
            this.buffer = buffer;
            this.contentLength = contentLength;
        }

        public FormData getFormData() {
            return formData;
        }

        public byte[] getBuffer() {
            return buffer;
        }

        public int getContentLength() {
            return contentLength;
        }
    }


}
