/*
 * Copyright (c) 2012 The ANTLR Project. All rights reserved.
 * Use of this file is governed by the BSD-3-Clause license that
 * can be found in the LICENSE.txt file in the project root.
 */

package org.pkl.thirdparty.antlr.v4.runtime;

import org.pkl.thirdparty.antlr.v4.runtime.misc.NotNull;
import org.pkl.thirdparty.antlr.v4.runtime.misc.Nullable;

/** How to emit recognition errors. */
public interface ANTLRErrorListener<Symbol> {
	/**
	 * Upon syntax error, notify any interested parties. This is not how to
	 * recover from errors or compute error messages. {@link ANTLRErrorStrategy}
	 * specifies how to recover from syntax errors and how to compute error
	 * messages. This listener's job is simply to emit a computed message,
	 * though it has enough information to create its own message in many cases.
	 *
	 * <p>The {@link RecognitionException} is non-null for all syntax errors except
	 * when we discover mismatched token errors that we can recover from
	 * in-line, without returning from the surrounding rule (via the single
	 * token insertion and deletion mechanism).</p>
	 *
	 * @param recognizer
     *        What parser got the error. From this
	 * 		  object, you can access the context as well
	 * 		  as the input stream.
	 * @param offendingSymbol
	 *        The offending token in the input token
	 * 		  stream, unless recognizer is a lexer (then it's null). If
	 * 		  no viable alternative error, {@code e} has token at which we
	 * 		  started production for the decision.
	 * @param line
	 * 		  The line number in the input where the error occurred.
	 * @param charPositionInLine
	 * 		  The character position within that line where the error occurred.
	 * @param msg
	 * 		  The message to emit.
	 * @param e
	 *        The exception generated by the parser that led to
	 *        the reporting of an error. It is null in the case where
	 *        the parser was able to recover in line without exiting the
	 *        surrounding rule.
	 */
	public <T extends Symbol> void syntaxError(@NotNull Recognizer<T, ?> recognizer,
											   @Nullable T offendingSymbol,
											   int line,
											   int charPositionInLine,
											   @NotNull String msg,
											   @Nullable RecognitionException e);
}
