/*
 * Copyright (c) 2012 The ANTLR Project. All rights reserved.
 * Use of this file is governed by the BSD-3-Clause license that
 * can be found in the LICENSE.txt file in the project root.
 */

package org.pkl.thirdparty.antlr.v4.runtime.atn;

import org.pkl.thirdparty.antlr.v4.runtime.Lexer;
import org.pkl.thirdparty.antlr.v4.runtime.misc.MurmurHash;
import org.pkl.thirdparty.antlr.v4.runtime.misc.NotNull;

/**
 * Implements the {@code skip} lexer action by calling {@link Lexer#skip}.
 *
 * <p>The {@code skip} command does not have any parameters, so this action is
 * implemented as a singleton instance exposed by {@link #INSTANCE}.</p>
 *
 * @author Sam Harwell
 * @since 4.2
 */
public final class LexerSkipAction implements LexerAction {
	/**
	 * Provides a singleton instance of this parameterless lexer action.
	 */
	public static final LexerSkipAction INSTANCE = new LexerSkipAction();

	/**
	 * Constructs the singleton instance of the lexer {@code skip} command.
	 */
	private LexerSkipAction() {
	}

	/**
	 * {@inheritDoc}
	 * @return This method returns {@link LexerActionType#SKIP}.
	 */
	@Override
	public LexerActionType getActionType() {
		return LexerActionType.SKIP;
	}

	/**
	 * {@inheritDoc}
	 * @return This method returns {@code false}.
	 */
	@Override
	public boolean isPositionDependent() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>This action is implemented by calling {@link Lexer#skip}.</p>
	 */
	@Override
	public void execute(@NotNull Lexer lexer) {
		lexer.skip();
	}

	@Override
	public int hashCode() {
		int hash = MurmurHash.initialize();
		hash = MurmurHash.update(hash, getActionType().ordinal());
		return MurmurHash.finish(hash, 1);
	}

	@Override
	@SuppressWarnings("EqualsWhichDoesntCheckParameterClass")
	public boolean equals(Object obj) {
		return obj == this;
	}

	@Override
	public String toString() {
		return "skip";
	}
}
