/*
 * Copyright 2002-2004 Jeremias Maerki.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.krysalis.barcode4j.impl.int2of5;

import org.krysalis.barcode4j.BarcodeDimension;
import org.krysalis.barcode4j.ChecksumMode;
import org.krysalis.barcode4j.ClassicBarcodeLogicHandler;
import org.krysalis.barcode4j.impl.AbstractBarcodeBean;
import org.krysalis.barcode4j.impl.DefaultCanvasLogicHandler;
import org.krysalis.barcode4j.output.Canvas;
import org.krysalis.barcode4j.output.CanvasProvider;

/**
 * This class is an implementation of the Interleaved 2 of 5 barcode.
 *
 * @author Jeremias Maerki
 * @version $Id: Interleaved2Of5Bean.java,v 1.5 2008-05-13 13:00:46 jmaerki Exp $
 */
public class Interleaved2Of5Bean extends AbstractBarcodeBean {

    /**
     * The default wide factor for Interleaved 2 of 5.
     */
    public static final double DEFAULT_WIDE_FACTOR = 3.0;
    /**
     * The default module width for Interleaved 2 of 5.
     */
    protected static final double DEFAULT_MODULE_WIDTH = 0.21f; //mm
    private ChecksumMode checksumMode = ChecksumMode.CP_AUTO;
    private double wideFactor = DEFAULT_WIDE_FACTOR; //Determines the width of wide bar
    private boolean displayChecksum = false;

    /**
     * Create a new instance.
     */
    public Interleaved2Of5Bean() {
        this.moduleWidth = DEFAULT_MODULE_WIDTH;
        setQuietZone(10 * this.moduleWidth);
        setVerticalQuietZone(0); //1D barcodes don't have vertical quiet zones
    }

    /**
     * Indicates whether the checksum will be displayed as
     * part of the human-readable message.
     *
     * @return true if checksum will be included in the human-readable message
     */
    public boolean isDisplayChecksum() {
        return this.displayChecksum;
    }

    /**
     * Enables or disables the use of the checksum in the
     * human-readable message.
     *
     * @param value true to include the checksum in the human-readable message,
     *              false to ignore
     */
    public void setDisplayChecksum(boolean value) {
        this.displayChecksum = value;
    }

    /**
     * @see org.krysalis.barcode4j.impl.AbstractBarcodeBean#getBarWidth(int)
     */
    public double getBarWidth(int width) {
        if (width == 1) {
            return moduleWidth;
        }
        else if (width == 2) {
            return moduleWidth * wideFactor;
        }
        else {
            throw new IllegalArgumentException("Only widths 1 and 2 allowed");
        }
    }

    /**
     * @see org.krysalis.barcode4j.BarcodeGenerator#generateBarcode(CanvasProvider, String)
     */
    public void generateBarcode(CanvasProvider canvas, String msg) {
        if ((msg == null)
                    || (msg.length() == 0)) {
            throw new NullPointerException("Parameter msg must not be empty");
        }

        ClassicBarcodeLogicHandler handler =
                    new DefaultCanvasLogicHandler(this, new Canvas(canvas));
        //handler = new LoggingLogicHandlerProxy(handler);

        Interleaved2Of5LogicImpl impl = new Interleaved2Of5LogicImpl(
                    getChecksumMode(), isDisplayChecksum());
        impl.generateBarcodeLogic(handler, msg);
    }

    /**
     * @see org.krysalis.barcode4j.BarcodeGenerator#calcDimensions(String)
     */
    public BarcodeDimension calcDimensions(String msg) {
        int msgLen = msg.length();
        if (getChecksumMode() == ChecksumMode.CP_ADD) {
            msgLen++;
        }
        if ((msgLen % 2) != 0) {
            msgLen++; //Compensate for odd number of characters
        }
        final double charwidth = 2 * wideFactor + 3;
        final double width = ((msgLen * charwidth) + 6 + wideFactor) * moduleWidth;
        final double qz = (hasQuietZone() ? quietZone : 0);
        return new BarcodeDimension(width, getHeight(),
                    width + (2 * qz), getHeight(),
                    quietZone, 0.0);
    }

    /**
     * Returns the current checksum mode.
     *
     * @return ChecksumMode the checksum mode
     */
    public ChecksumMode getChecksumMode() {
        return this.checksumMode;
    }

    /**
     * Sets the checksum mode
     *
     * @param mode the checksum mode
     */
    public void setChecksumMode(ChecksumMode mode) {
        this.checksumMode = mode;
    }

    /**
     * Returns the factor by which wide bars are broader than narrow bars.
     *
     * @return the wide factor
     */
    public double getWideFactor() {
        return this.wideFactor;
    }

    /**
     * Sets the factor by which wide bars are broader than narrow bars.
     *
     * @param value the wide factory (should be > 1.0)
     */
    public void setWideFactor(double value) {
        if (value <= 1.0) {
            throw new IllegalArgumentException("wide factor must be > 1.0");
        }
        this.wideFactor = value;
    }

}