/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.versioned.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.MoreCollectors;
import com.google.common.collect.Streams;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.projectnessie.versioned.BranchName;
import org.projectnessie.versioned.Delete;
import org.projectnessie.versioned.Diff;
import org.projectnessie.versioned.Hash;
import org.projectnessie.versioned.Key;
import org.projectnessie.versioned.NamedRef;
import org.projectnessie.versioned.Operation;
import org.projectnessie.versioned.Put;
import org.projectnessie.versioned.Ref;
import org.projectnessie.versioned.ReferenceAlreadyExistsException;
import org.projectnessie.versioned.ReferenceConflictException;
import org.projectnessie.versioned.ReferenceNotFoundException;
import org.projectnessie.versioned.Serializer;
import org.projectnessie.versioned.TagName;
import org.projectnessie.versioned.Unchanged;
import org.projectnessie.versioned.VersionStore;
import org.projectnessie.versioned.VersionStoreException;
import org.projectnessie.versioned.WithHash;
import org.projectnessie.versioned.memory.Commit;
import org.projectnessie.versioned.memory.CommitsIterator;
import org.projectnessie.versioned.memory.InactiveCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryVersionStore<ValueT, MetadataT>
implements VersionStore<ValueT, MetadataT> {
    private static final Logger LOGGER = LoggerFactory.getLogger(InMemoryVersionStore.class);
    private final ConcurrentMap<Hash, Commit<ValueT, MetadataT>> commits = new ConcurrentHashMap<Hash, Commit<ValueT, MetadataT>>();
    private final ConcurrentMap<NamedRef, Hash> namedReferences = new ConcurrentHashMap<NamedRef, Hash>();
    private final Serializer<ValueT> valueSerializer;
    private final Serializer<MetadataT> metadataSerializer;

    private InMemoryVersionStore(Builder<ValueT, MetadataT> builder) {
        this.valueSerializer = ((Builder)builder).valueSerializer;
        this.metadataSerializer = ((Builder)builder).metadataSerializer;
    }

    public static <ValueT, MetadataT> Builder<ValueT, MetadataT> builder() {
        return new Builder();
    }

    public Hash toHash(NamedRef ref) throws ReferenceNotFoundException {
        Hash hash = (Hash)this.namedReferences.get(Objects.requireNonNull(ref));
        if (hash == null) {
            throw ReferenceNotFoundException.forReference((Ref)ref);
        }
        return hash;
    }

    private Hash toHash(Ref ref) throws ReferenceNotFoundException {
        if (ref instanceof NamedRef) {
            return this.toHash((NamedRef)ref);
        }
        if (ref instanceof Hash) {
            Hash hash = (Hash)ref;
            if (!hash.equals((Object)Commit.NO_ANCESTOR) && !this.commits.containsKey(hash)) {
                throw ReferenceNotFoundException.forReference((Ref)hash);
            }
            return hash;
        }
        throw new IllegalArgumentException(String.format("Unsupported reference type for ref %s", ref));
    }

    private void checkValidReferenceHash(BranchName branch, Hash currentBranchHash, Hash referenceHash) throws ReferenceNotFoundException {
        if (referenceHash.equals((Object)Commit.NO_ANCESTOR)) {
            return;
        }
        Optional foundHash = (Optional)Streams.stream(new CommitsIterator(this.commits::get, currentBranchHash)).map(WithHash::getHash).filter(hash -> hash.equals((Object)referenceHash)).collect(MoreCollectors.toOptional());
        foundHash.orElseThrow(() -> new ReferenceNotFoundException(String.format("'%s' hash is not a valid commit from branch '%s'(%s)", referenceHash, branch, currentBranchHash)));
    }

    public WithHash<Ref> toRef(String refOfUnknownType) throws ReferenceNotFoundException {
        Objects.requireNonNull(refOfUnknownType);
        Optional<WithHash> result = Stream.of(TagName::of, BranchName::of, Hash::of).map(f -> {
            try {
                Ref ref = (Ref)f.apply(refOfUnknownType);
                return WithHash.of((Hash)this.toHash(ref), (Object)ref);
            }
            catch (IllegalArgumentException | ReferenceNotFoundException e) {
                return null;
            }
        }).filter(Objects::nonNull).findFirst();
        return result.orElseThrow(() -> ReferenceNotFoundException.forReference((String)refOfUnknownType));
    }

    public void commit(BranchName branch, Optional<Hash> referenceHash, MetadataT metadata, List<Operation<ValueT>> operations) throws ReferenceNotFoundException, ReferenceConflictException {
        Hash currentHash = this.toHash((NamedRef)branch);
        List<Key> keys = operations.stream().map(Operation::getKey).distinct().collect(Collectors.toList());
        this.checkConcurrentModification(branch, currentHash, referenceHash, keys);
        InMemoryVersionStore.compute(this.namedReferences, branch, (key, hash) -> {
            Commit<ValueT, Object> commit = Commit.of(this.valueSerializer, this.metadataSerializer, currentHash, metadata, operations);
            Hash previousHash = Optional.ofNullable(hash).orElse(Commit.NO_ANCESTOR);
            if (!previousHash.equals((Object)currentHash)) {
                throw ReferenceConflictException.forReference((NamedRef)branch, (Optional)referenceHash, Optional.of(previousHash));
            }
            Hash commitHash = commit.getHash();
            this.commits.putIfAbsent(commitHash, commit);
            return commitHash;
        });
    }

    public void transplant(BranchName targetBranch, Optional<Hash> referenceHash, List<Hash> sequenceToTransplant) throws ReferenceNotFoundException, ReferenceConflictException {
        Objects.requireNonNull(targetBranch);
        Objects.requireNonNull(sequenceToTransplant);
        Hash currentHash = this.toHash((NamedRef)targetBranch);
        if (sequenceToTransplant.isEmpty()) {
            return;
        }
        HashSet keys = new HashSet();
        ArrayList<Commit<ValueT, MetadataT>> toStore = new ArrayList<Commit<ValueT, MetadataT>>(sequenceToTransplant.size());
        Hash ancestor = null;
        Hash newAncestor = currentHash;
        for (Hash hash2 : sequenceToTransplant) {
            Commit commit = (Commit)this.commits.get(hash2);
            if (commit == null) {
                throw ReferenceNotFoundException.forReference((Ref)hash2);
            }
            if (ancestor != null && !ancestor.equals((Object)commit.getAncestor())) {
                throw new IllegalArgumentException(String.format("Hash %s is not the ancestor for commit %s", ancestor, hash2));
            }
            commit.getOperations().forEach(op -> keys.add(op.getKey()));
            Commit<ValueT, MetadataT> newCommit = Commit.of(this.valueSerializer, this.metadataSerializer, newAncestor, commit.getMetadata(), commit.getOperations());
            toStore.add(newCommit);
            ancestor = commit.getHash();
            newAncestor = newCommit.getHash();
        }
        this.checkConcurrentModification(targetBranch, currentHash, referenceHash, new ArrayList<Key>(keys));
        InMemoryVersionStore.compute(this.namedReferences, targetBranch, (key, hash) -> {
            Hash previousHash = Optional.ofNullable(hash).orElse(Commit.NO_ANCESTOR);
            if (!previousHash.equals((Object)currentHash)) {
                throw ReferenceConflictException.forReference((NamedRef)targetBranch, (Optional)referenceHash, Optional.of(previousHash));
            }
            toStore.forEach(commit -> this.commits.putIfAbsent(commit.getHash(), (Commit<ValueT, MetadataT>)commit));
            Commit lastCommit = (Commit)Iterables.getLast((Iterable)toStore);
            return lastCommit.getHash();
        });
    }

    private void checkConcurrentModification(BranchName targetBranch, Hash currentHash, Optional<Hash> referenceHash, List<Key> keyList) throws ReferenceNotFoundException, ReferenceConflictException {
        try {
            InMemoryVersionStore.ifPresent(referenceHash, hash -> {
                this.checkValidReferenceHash(targetBranch, currentHash, (Hash)hash);
                List<Optional<ValueT>> referenceValues = this.getValues((Ref)hash, keyList);
                List<Optional<ValueT>> currentValues = this.getValues((Ref)currentHash, keyList);
                if (!referenceValues.equals(currentValues)) {
                    throw ReferenceConflictException.forReference((NamedRef)targetBranch, (Optional)referenceHash, Optional.of(currentHash));
                }
            });
        }
        catch (ReferenceConflictException | ReferenceNotFoundException e) {
            throw e;
        }
        catch (VersionStoreException e) {
            throw new AssertionError((Object)e);
        }
    }

    public void merge(Hash fromHash, BranchName toBranch, Optional<Hash> expectedBranchHash) throws ReferenceNotFoundException, ReferenceConflictException {
        Objects.requireNonNull(fromHash);
        Objects.requireNonNull(toBranch);
        if (!this.commits.containsKey(fromHash)) {
            throw ReferenceNotFoundException.forReference((Ref)fromHash);
        }
        Hash currentHash = this.toHash((NamedRef)toBranch);
        Hash referenceHash = expectedBranchHash.orElse(currentHash);
        Set toBranchHashes = Streams.stream(new CommitsIterator(this.commits::get, referenceHash)).map(WithHash::getHash).collect(Collectors.toSet());
        HashSet keys = new HashSet();
        ArrayList<Commit> toMerge = new ArrayList<Commit>();
        Hash commonAncestor = null;
        CommitsIterator iterator = new CommitsIterator(this.commits::get, fromHash);
        while (iterator.hasNext()) {
            WithHash commit = (WithHash)iterator.next();
            if (toBranchHashes.contains(commit.getHash())) {
                commonAncestor = commit.getHash();
                break;
            }
            toMerge.add((Commit)commit.getValue());
            ((Commit)commit.getValue()).getOperations().forEach(op -> keys.add(op.getKey()));
        }
        this.checkConcurrentModification(toBranch, currentHash, expectedBranchHash, new ArrayList<Key>(keys));
        ArrayList<Commit<ValueT, MetadataT>> toStore = new ArrayList<Commit<ValueT, MetadataT>>(toMerge.size());
        Hash newAncestor = currentHash;
        for (Commit commit : Lists.reverse(toMerge)) {
            Commit<ValueT, MetadataT> newCommit = Commit.of(this.valueSerializer, this.metadataSerializer, newAncestor, commit.getMetadata(), commit.getOperations());
            toStore.add(newCommit);
            newAncestor = newCommit.getHash();
        }
        InMemoryVersionStore.compute(this.namedReferences, toBranch, (key, hash) -> {
            Hash previousHash = Optional.ofNullable(hash).orElse(Commit.NO_ANCESTOR);
            if (!previousHash.equals((Object)currentHash)) {
                throw ReferenceConflictException.forReference((NamedRef)toBranch, (Optional)expectedBranchHash, Optional.of(previousHash));
            }
            toStore.forEach(commit -> this.commits.putIfAbsent(commit.getHash(), (Commit<ValueT, MetadataT>)commit));
            Commit lastCommit = (Commit)Iterables.getLast((Iterable)toStore);
            return lastCommit.getHash();
        });
    }

    public void assign(NamedRef ref, Optional<Hash> expectedRefHash, Hash targetHash) throws ReferenceNotFoundException, ReferenceConflictException {
        Objects.requireNonNull(ref);
        Objects.requireNonNull(targetHash);
        Hash currentHash = this.toHash(ref);
        InMemoryVersionStore.ifPresent(expectedRefHash, hash -> {
            if (!hash.equals((Object)currentHash)) {
                throw ReferenceConflictException.forReference((NamedRef)ref, (Optional)expectedRefHash, Optional.of(currentHash));
            }
        });
        if (!this.commits.containsKey(targetHash)) {
            throw ReferenceNotFoundException.forReference((Ref)targetHash);
        }
        this.doAssign(ref, currentHash, targetHash);
    }

    private void doAssign(NamedRef ref, Hash expectedHash, Hash newHash) throws ReferenceNotFoundException, ReferenceConflictException {
        InMemoryVersionStore.compute(this.namedReferences, ref, (key, hash) -> {
            Hash previousHash = Optional.ofNullable(hash).orElse(expectedHash);
            if (!expectedHash.equals((Object)previousHash)) {
                throw ReferenceConflictException.forReference((NamedRef)ref, Optional.of(expectedHash), Optional.of(previousHash));
            }
            return newHash;
        });
    }

    public void create(NamedRef ref, Optional<Hash> targetHash) throws ReferenceNotFoundException, ReferenceAlreadyExistsException {
        Preconditions.checkArgument((ref instanceof BranchName || targetHash.isPresent() ? 1 : 0) != 0, (Object)"Cannot create an unassigned tag reference");
        InMemoryVersionStore.compute(this.namedReferences, ref, (key, currentHash) -> {
            if (currentHash != null) {
                throw ReferenceAlreadyExistsException.forReference((NamedRef)ref);
            }
            return targetHash.orElse(Commit.NO_ANCESTOR);
        });
    }

    public void delete(NamedRef ref, Optional<Hash> hash) throws ReferenceNotFoundException, ReferenceConflictException {
        try {
            InMemoryVersionStore.compute(this.namedReferences, ref, (key, currentHash) -> {
                if (currentHash == null) {
                    throw ReferenceNotFoundException.forReference((Ref)ref);
                }
                InMemoryVersionStore.ifPresent(hash, h -> {
                    if (!h.equals(currentHash)) {
                        throw ReferenceConflictException.forReference((NamedRef)ref, (Optional)hash, Optional.of(currentHash));
                    }
                });
                return null;
            });
        }
        catch (ReferenceConflictException | ReferenceNotFoundException e) {
            throw e;
        }
        catch (VersionStoreException e) {
            throw new AssertionError((Object)e);
        }
    }

    public Stream<WithHash<NamedRef>> getNamedRefs() {
        return this.namedReferences.entrySet().stream().map(entry -> WithHash.of((Hash)((Hash)entry.getValue()), (Object)((NamedRef)entry.getKey())));
    }

    public Stream<WithHash<MetadataT>> getCommits(Ref ref) throws ReferenceNotFoundException {
        Hash hash = this.toHash(ref);
        CommitsIterator iterator = new CommitsIterator(this.commits::get, hash);
        return Streams.stream(iterator).map(wh -> WithHash.of((Hash)wh.getHash(), ((Commit)wh.getValue()).getMetadata()));
    }

    public Stream<Key> getKeys(Ref ref) throws ReferenceNotFoundException {
        Hash hash = this.toHash(ref);
        CommitsIterator iterator = new CommitsIterator(this.commits::get, hash);
        HashSet deleted = new HashSet();
        return Streams.stream(iterator).flatMap(wh -> Lists.reverse(((Commit)wh.getValue()).getOperations()).stream()).filter(operation -> {
            Key key = operation.getKey();
            if (operation instanceof Delete) {
                deleted.add(key);
            }
            return !deleted.contains(key);
        }).map(Operation::getKey).distinct();
    }

    public ValueT getValue(Ref ref, Key key) throws ReferenceNotFoundException {
        return this.getValues(ref, Collections.singletonList(key)).get(0).orElse(null);
    }

    public List<Optional<ValueT>> getValues(Ref ref, List<Key> keys) throws ReferenceNotFoundException {
        Hash hash = this.toHash(ref);
        int size = keys.size();
        ArrayList<Optional<ValueT>> results = new ArrayList<Optional<ValueT>>(size);
        results.addAll(Collections.nCopies(size, Optional.empty()));
        HashSet<Key> toFind = new HashSet<Key>();
        toFind.addAll(keys);
        CommitsIterator iterator = new CommitsIterator(this.commits::get, hash);
        while (iterator.hasNext() && !toFind.isEmpty()) {
            Commit commit = (Commit)((WithHash)iterator.next()).getValue();
            for (Operation operation : Lists.reverse(commit.getOperations())) {
                Key operationKey = operation.getKey();
                if (!toFind.contains(operationKey)) continue;
                if (operation instanceof Put) {
                    Put put = (Put)operation;
                    int index = keys.indexOf(operationKey);
                    results.set(index, Optional.of(put.getValue()));
                    toFind.remove(operationKey);
                    continue;
                }
                if (operation instanceof Delete) {
                    toFind.remove(operationKey);
                    continue;
                }
                if (!(operation instanceof Unchanged)) {
                    throw new AssertionError((Object)("Unsupported operation type for " + operation));
                }
            }
        }
        return results;
    }

    public Stream<Diff<ValueT>> getDiffs(Ref from, Ref to) {
        throw new UnsupportedOperationException("Not yet implemented.");
    }

    public VersionStore.Collector collectGarbage() {
        return InactiveCollector.of();
    }

    @VisibleForTesting
    public void clearUnsafe() {
        this.commits.clear();
        this.namedReferences.clear();
        try {
            this.create((NamedRef)BranchName.of((String)"main"), Optional.empty());
        }
        catch (ReferenceAlreadyExistsException | ReferenceNotFoundException e) {
            throw new RuntimeException("Failed to reset the InMemoryVersionStore for tests");
        }
    }

    private static <K, V, E extends VersionStoreException> V compute(ConcurrentMap<K, V> map, K key, ComputeFunction<K, V, E> doCompute) throws E {
        try {
            return (V)map.compute(key, (k, v) -> {
                try {
                    return doCompute.apply(k, v);
                }
                catch (VersionStoreException e) {
                    throw new VersionStoreExecutionError(e);
                }
            });
        }
        catch (VersionStoreExecutionError e) {
            VersionStoreException cause = e.getCause();
            throw cause;
        }
    }

    private static <T, E extends VersionStoreException> void ifPresent(Optional<T> optional, IfPresentConsumer<? super T, E> consumer) throws E {
        try {
            optional.ifPresent(value -> {
                try {
                    consumer.accept((Object)value);
                }
                catch (VersionStoreException e) {
                    throw new VersionStoreExecutionError(e);
                }
            });
        }
        catch (VersionStoreExecutionError e) {
            VersionStoreException cause = e.getCause();
            throw cause;
        }
    }

    @FunctionalInterface
    private static interface IfPresentConsumer<V, E extends VersionStoreException> {
        public void accept(V var1) throws E;
    }

    @FunctionalInterface
    private static interface ComputeFunction<K, V, E extends VersionStoreException> {
        public V apply(K var1, V var2) throws E;
    }

    private static class VersionStoreExecutionError
    extends Error {
        private VersionStoreExecutionError(VersionStoreException cause) {
            super((Throwable)cause);
        }

        public synchronized VersionStoreException getCause() {
            return (VersionStoreException)super.getCause();
        }
    }

    public static final class Builder<ValueT, MetadataT> {
        private Serializer<ValueT> valueSerializer = null;
        private Serializer<MetadataT> metadataSerializer = null;

        public Builder<ValueT, MetadataT> valueSerializer(Serializer<ValueT> serializer) {
            this.valueSerializer = Objects.requireNonNull(serializer);
            return this;
        }

        public Builder<ValueT, MetadataT> metadataSerializer(Serializer<MetadataT> serializer) {
            this.metadataSerializer = Objects.requireNonNull(serializer);
            return this;
        }

        public InMemoryVersionStore<ValueT, MetadataT> build() {
            Preconditions.checkState((this.valueSerializer != null ? 1 : 0) != 0, (Object)"Value serializer hasn't been set");
            Preconditions.checkState((this.metadataSerializer != null ? 1 : 0) != 0, (Object)"Metadata serializer hasn't been set");
            return new InMemoryVersionStore(this);
        }
    }
}

