/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.projectnessie.client.auth.oauth2.AccessToken;
import org.projectnessie.client.auth.oauth2.ClientCredentialsTokensResponse;
import org.projectnessie.client.auth.oauth2.ErrorResponse;
import org.projectnessie.client.auth.oauth2.ImmutableClientCredentialsTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutablePasswordTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableRefreshTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableTokensExchangeRequest;
import org.projectnessie.client.auth.oauth2.JwtToken;
import org.projectnessie.client.auth.oauth2.OAuth2Authenticator;
import org.projectnessie.client.auth.oauth2.OAuth2ClientParams;
import org.projectnessie.client.auth.oauth2.OAuth2Exception;
import org.projectnessie.client.auth.oauth2.OAuth2TokenRefreshExecutor;
import org.projectnessie.client.auth.oauth2.PasswordTokensResponse;
import org.projectnessie.client.auth.oauth2.RefreshTokensResponse;
import org.projectnessie.client.auth.oauth2.Token;
import org.projectnessie.client.auth.oauth2.TokenTypeIdentifiers;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.auth.oauth2.TokensExchangeResponse;
import org.projectnessie.client.http.HttpClient;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.client.http.HttpResponse;
import org.projectnessie.client.http.ResponseContext;
import org.projectnessie.client.http.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OAuth2Client
implements OAuth2Authenticator,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(OAuth2Client.class);
    private static final Duration MIN_WARN_INTERVAL = Duration.ofSeconds(10L);
    private final String grantType;
    private final String username;
    private final byte[] password;
    private final String scope;
    private final Duration defaultAccessTokenLifespan;
    private final Duration defaultRefreshTokenLifespan;
    private final Duration refreshSafetyWindow;
    private final Duration idleInterval;
    private final boolean tokenExchangeEnabled;
    private final HttpClient httpClient;
    private final ScheduledExecutorService executor;
    private final boolean shouldCloseExecutor;
    private final ObjectMapper objectMapper;
    private final CompletableFuture<Void> started = new CompletableFuture();
    final AtomicBoolean sleeping = new AtomicBoolean();
    private final AtomicBoolean closing = new AtomicBoolean();
    private final Supplier<Instant> clock;
    private volatile CompletionStage<Tokens> currentTokensStage;
    private volatile ScheduledFuture<?> tokenRefreshFuture;
    private volatile Instant lastAccess;
    private volatile Instant lastWarn;

    OAuth2Client(OAuth2ClientParams params) {
        this.grantType = params.getGrantType();
        this.username = params.getUsername().orElse(null);
        this.password = params.getPassword().map(s -> s.getBytes(StandardCharsets.UTF_8)).orElse(null);
        this.scope = params.getScope().orElse(null);
        this.defaultAccessTokenLifespan = params.getDefaultAccessTokenLifespan();
        this.defaultRefreshTokenLifespan = params.getDefaultRefreshTokenLifespan();
        this.refreshSafetyWindow = params.getRefreshSafetyWindow();
        this.idleInterval = params.getPreemptiveTokenRefreshIdleTimeout();
        this.tokenExchangeEnabled = params.getTokenExchangeEnabled();
        this.httpClient = params.getHttpClient().addResponseFilter(this::checkErrorResponse).build();
        this.executor = params.getExecutor();
        this.shouldCloseExecutor = this.executor instanceof OAuth2TokenRefreshExecutor;
        this.objectMapper = params.getObjectMapper();
        this.clock = params.getClock();
        this.lastAccess = this.clock.get();
        this.currentTokensStage = this.started.thenApplyAsync(v -> this.fetchNewTokens(), (Executor)this.executor);
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    @Override
    public AccessToken authenticate() {
        Instant now;
        this.lastAccess = now = this.clock.get();
        if (this.sleeping.compareAndSet(true, false)) {
            this.wakeUp(now);
        }
        return this.getCurrentTokens().getAccessToken();
    }

    Tokens getCurrentTokens() {
        try {
            return this.currentTokensStage.toCompletableFuture().get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof Error) {
                throw (Error)cause;
            }
            if (cause instanceof HttpClientException) {
                throw (HttpClientException)cause;
            }
            throw new RuntimeException("Cannot acquire a valid OAuth2 access token", cause);
        }
    }

    private Tokens getCurrentTokensIfAvailable() {
        try {
            return this.currentTokensStage.toCompletableFuture().getNow(null);
        }
        catch (CancellationException | CompletionException runtimeException) {
            return null;
        }
    }

    public void start() {
        this.started.complete(null);
    }

    @Override
    public void close() {
        if (this.closing.compareAndSet(false, true)) {
            LOGGER.debug("Closing...");
            try {
                this.currentTokensStage.toCompletableFuture().cancel(true);
                ScheduledFuture<?> tokenRefreshFuture = this.tokenRefreshFuture;
                if (tokenRefreshFuture != null) {
                    tokenRefreshFuture.cancel(true);
                }
                if (this.shouldCloseExecutor && !this.executor.isShutdown()) {
                    this.executor.shutdown();
                    if (!this.executor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        this.executor.shutdownNow();
                    }
                }
                if (this.password != null) {
                    Arrays.fill(this.password, (byte)0);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            finally {
                this.tokenRefreshFuture = null;
            }
            LOGGER.debug("Closed");
        }
    }

    private void wakeUp(Instant now) {
        LOGGER.debug("Waking up...");
        Tokens currentTokens = this.getCurrentTokensIfAvailable();
        Duration delay = this.nextTokenRefresh(currentTokens, now, Duration.ZERO);
        if (delay.compareTo(OAuth2ClientParams.MIN_REFRESH_DELAY) < 0) {
            LOGGER.debug("Refreshing tokens immediately");
            this.renewTokens();
        } else {
            LOGGER.debug("Tokens are still valid");
            this.scheduleTokensRenewal(delay);
        }
    }

    private void maybeScheduleTokensRenewal(Tokens currentTokens) {
        Instant now = this.clock.get();
        if (Duration.between(this.lastAccess, now).compareTo(this.idleInterval) > 0) {
            this.sleeping.set(true);
            LOGGER.debug("Sleeping...");
        } else {
            Duration delay = this.nextTokenRefresh(currentTokens, now, OAuth2ClientParams.MIN_REFRESH_DELAY);
            this.scheduleTokensRenewal(delay);
        }
    }

    private void scheduleTokensRenewal(Duration delay) {
        if (this.closing.get()) {
            return;
        }
        LOGGER.debug("Scheduling token refresh in {}", (Object)delay);
        try {
            this.tokenRefreshFuture = this.executor.schedule(this::renewTokens, delay.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            if (this.closing.get()) {
                return;
            }
            this.maybeWarn("Failed to schedule next token renewal, forcibly sleeping", null);
            this.sleeping.set(true);
        }
    }

    private void renewTokens() {
        CompletionStage<Tokens> oldTokensStage = this.currentTokensStage;
        this.currentTokensStage = oldTokensStage.thenApply(this::refreshTokens).exceptionally(error -> this.fetchNewTokens());
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    private void log(Throwable error) {
        if (error != null) {
            boolean tokensStageCancelled;
            boolean bl = tokensStageCancelled = error instanceof CancellationException && this.closing.get();
            if (tokensStageCancelled) {
                return;
            }
            if (error instanceof CompletionException) {
                error = error.getCause();
            }
            this.maybeWarn("Failed to renew tokens", error);
        } else {
            LOGGER.debug("Successfully renewed tokens");
        }
    }

    Tokens fetchNewTokens() {
        LOGGER.debug("Fetching new tokens");
        if (this.grantType.equals("client_credentials")) {
            ImmutableClientCredentialsTokensRequest body = ImmutableClientCredentialsTokensRequest.builder().scope(this.scope).build();
            HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
            return httpResponse.readEntity(ClientCredentialsTokensResponse.class);
        }
        ImmutablePasswordTokensRequest body = ImmutablePasswordTokensRequest.builder().username(this.username).password(new String(this.password, StandardCharsets.UTF_8)).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(PasswordTokensResponse.class);
    }

    Tokens refreshTokens(Tokens currentTokens) {
        if (currentTokens.getRefreshToken() == null) {
            return this.exchangeTokens(currentTokens);
        }
        if (this.isAboutToExpire(currentTokens.getRefreshToken())) {
            throw new MustFetchNewTokensException("Refresh token is about to expire");
        }
        LOGGER.debug("Refreshing tokens");
        ImmutableRefreshTokensRequest body = ImmutableRefreshTokensRequest.builder().refreshToken(currentTokens.getRefreshToken().getPayload()).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(RefreshTokensResponse.class);
    }

    Tokens exchangeTokens(Tokens currentToken) {
        if (!this.tokenExchangeEnabled) {
            throw new MustFetchNewTokensException("Token exchange is disabled");
        }
        LOGGER.debug("Exchanging tokens");
        ImmutableTokensExchangeRequest body = ImmutableTokensExchangeRequest.builder().subjectToken(currentToken.getAccessToken().getPayload()).subjectTokenType(TokenTypeIdentifiers.ACCESS_TOKEN).requestedTokenType(TokenTypeIdentifiers.REFRESH_TOKEN).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(TokensExchangeResponse.class);
    }

    private boolean isAboutToExpire(Token token) {
        Instant now = this.clock.get();
        return OAuth2Client.tokenExpirationTime(now, token, this.defaultRefreshTokenLifespan).isBefore(now.plus(this.refreshSafetyWindow));
    }

    private Duration nextTokenRefresh(Tokens currentTokens, Instant now, Duration minRefreshDelay) {
        if (currentTokens == null) {
            return minRefreshDelay;
        }
        Instant accessExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getAccessToken(), this.defaultAccessTokenLifespan);
        Instant refreshExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getRefreshToken(), this.defaultRefreshTokenLifespan);
        return OAuth2Client.shortestDelay(now, accessExpirationTime, refreshExpirationTime, this.refreshSafetyWindow, minRefreshDelay);
    }

    static Duration shortestDelay(Instant now, Instant accessExpirationTime, Instant refreshExpirationTime, Duration refreshSafetyWindow, Duration minRefreshDelay) {
        Instant expirationTime = accessExpirationTime.isBefore(refreshExpirationTime) ? accessExpirationTime : refreshExpirationTime;
        Duration delay = Duration.between(now, expirationTime).minus(refreshSafetyWindow);
        if (delay.compareTo(minRefreshDelay) < 0) {
            delay = minRefreshDelay;
        }
        return delay;
    }

    static Instant tokenExpirationTime(Instant now, Token token, Duration defaultLifespan) {
        Instant expirationTime = null;
        if (token != null && (expirationTime = token.getExpirationTime()) == null) {
            try {
                JwtToken jwtToken = JwtToken.parse(token.getPayload());
                expirationTime = jwtToken.getExpirationTime();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (expirationTime == null) {
            expirationTime = now.plus(defaultLifespan);
        }
        return expirationTime;
    }

    private void checkErrorResponse(ResponseContext responseContext) {
        try {
            Status status = responseContext.getResponseCode();
            if (status.getCode() >= 400) {
                if (!responseContext.isJsonCompatibleResponse()) {
                    throw OAuth2Client.genericError(status);
                }
                InputStream is = responseContext.getErrorStream();
                if (is != null) {
                    try {
                        ErrorResponse errorResponse = (ErrorResponse)this.objectMapper.readValue(is, ErrorResponse.class);
                        throw new OAuth2Exception(status, errorResponse);
                    }
                    catch (IOException ignored) {
                        throw OAuth2Client.genericError(status);
                    }
                }
            }
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new HttpClientException(e);
        }
    }

    private void maybeWarn(String message, Throwable error) {
        boolean shouldWarn;
        Instant now = this.clock.get();
        boolean bl = shouldWarn = this.lastWarn == null || Duration.between(this.lastWarn, now).compareTo(MIN_WARN_INTERVAL) > 0;
        if (shouldWarn) {
            if (error instanceof HttpClientException) {
                LOGGER.warn("{}: {}", (Object)message, (Object)error.toString());
            } else {
                LOGGER.warn(message, error);
            }
            this.lastWarn = now;
        } else {
            LOGGER.debug(message, error);
        }
    }

    private static HttpClientException genericError(Status status) {
        return new HttpClientException("OAuth2 server replied with HTTP status code: " + status.getCode());
    }

    static class MustFetchNewTokensException
    extends RuntimeException {
        public MustFetchNewTokensException(String message) {
            super(message);
        }
    }
}

