/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import java.io.Closeable;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.projectnessie.client.auth.oauth2.AccessToken;
import org.projectnessie.client.auth.oauth2.AuthorizationCodeFlow;
import org.projectnessie.client.auth.oauth2.ClientCredentialsTokensResponse;
import org.projectnessie.client.auth.oauth2.DeviceCodeFlow;
import org.projectnessie.client.auth.oauth2.GrantType;
import org.projectnessie.client.auth.oauth2.ImmutableClientCredentialsTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutablePasswordTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableRefreshTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableTokensExchangeRequest;
import org.projectnessie.client.auth.oauth2.JwtToken;
import org.projectnessie.client.auth.oauth2.OAuth2Authenticator;
import org.projectnessie.client.auth.oauth2.OAuth2ClientConfig;
import org.projectnessie.client.auth.oauth2.OAuth2TokenRefreshExecutor;
import org.projectnessie.client.auth.oauth2.PasswordTokensResponse;
import org.projectnessie.client.auth.oauth2.RefreshTokensResponse;
import org.projectnessie.client.auth.oauth2.Token;
import org.projectnessie.client.auth.oauth2.TokenTypeIdentifiers;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.auth.oauth2.TokensExchangeResponse;
import org.projectnessie.client.http.HttpClient;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.client.http.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OAuth2Client
implements OAuth2Authenticator,
Closeable {
    static final Logger LOGGER = LoggerFactory.getLogger(OAuth2Client.class);
    private static final Duration MIN_WARN_INTERVAL = Duration.ofSeconds(10L);
    private final OAuth2ClientConfig config;
    private final String username;
    private final byte[] password;
    private final String scope;
    private final HttpClient tokenEndpointClient;
    private final ScheduledExecutorService executor;
    private final CompletableFuture<Void> started = new CompletableFuture();
    final AtomicBoolean sleeping = new AtomicBoolean();
    private final AtomicBoolean closing = new AtomicBoolean();
    private volatile CompletionStage<Tokens> currentTokensStage;
    private volatile ScheduledFuture<?> tokenRefreshFuture;
    private volatile Instant lastAccess;
    private volatile Instant lastWarn;

    OAuth2Client(OAuth2ClientConfig config) {
        this.config = config;
        this.username = config.getUsername().orElse(null);
        this.password = config.getPassword().map(s -> s.getBytesAndClear(StandardCharsets.UTF_8)).orElse(null);
        this.scope = config.getScope().orElse(null);
        this.tokenEndpointClient = config.newHttpClientBuilder().setBaseUri(config.getResolvedTokenEndpoint()).setAuthentication(config.getBasicAuthentication()).build();
        this.executor = config.getExecutor();
        this.lastAccess = config.getClock().get();
        this.currentTokensStage = this.started.thenApplyAsync(v -> this.fetchNewTokens(), (Executor)this.executor);
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    @Override
    public AccessToken authenticate() {
        Instant now;
        this.lastAccess = now = this.config.getClock().get();
        if (this.sleeping.compareAndSet(true, false)) {
            this.wakeUp(now);
        }
        return this.getCurrentTokens().getAccessToken();
    }

    Tokens getCurrentTokens() {
        try {
            return this.currentTokensStage.toCompletableFuture().get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof Error) {
                throw (Error)cause;
            }
            if (cause instanceof HttpClientException) {
                throw (HttpClientException)cause;
            }
            throw new RuntimeException("Cannot acquire a valid OAuth2 access token", cause);
        }
    }

    private Tokens getCurrentTokensIfAvailable() {
        try {
            return this.currentTokensStage.toCompletableFuture().getNow(null);
        }
        catch (CancellationException | CompletionException runtimeException) {
            return null;
        }
    }

    @Override
    public void start() {
        this.started.complete(null);
    }

    @Override
    public void close() {
        if (this.closing.compareAndSet(false, true)) {
            LOGGER.debug("Closing...");
            try {
                boolean shouldClose;
                this.currentTokensStage.toCompletableFuture().cancel(true);
                ScheduledFuture<?> tokenRefreshFuture = this.tokenRefreshFuture;
                if (tokenRefreshFuture != null) {
                    tokenRefreshFuture.cancel(true);
                }
                if ((shouldClose = this.executor instanceof OAuth2TokenRefreshExecutor) && !this.executor.isShutdown()) {
                    this.executor.shutdown();
                    if (!this.executor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        this.executor.shutdownNow();
                    }
                }
                this.tokenEndpointClient.close();
                if (this.password != null) {
                    Arrays.fill(this.password, (byte)0);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            finally {
                this.tokenRefreshFuture = null;
            }
            LOGGER.debug("Closed");
        }
    }

    private void wakeUp(Instant now) {
        LOGGER.debug("Waking up...");
        Tokens currentTokens = this.getCurrentTokensIfAvailable();
        Duration delay = this.nextTokenRefresh(currentTokens, now, Duration.ZERO);
        if (delay.compareTo(this.config.getMinRefreshSafetyWindow()) < 0) {
            LOGGER.debug("Refreshing tokens immediately");
            this.renewTokens();
        } else {
            LOGGER.debug("Tokens are still valid");
            this.scheduleTokensRenewal(delay);
        }
    }

    private void maybeScheduleTokensRenewal(Tokens currentTokens) {
        Instant now = this.config.getClock().get();
        if (Duration.between(this.lastAccess, now).compareTo(this.config.getPreemptiveTokenRefreshIdleTimeout()) > 0) {
            this.sleeping.set(true);
            LOGGER.debug("Sleeping...");
        } else {
            Duration delay = this.nextTokenRefresh(currentTokens, now, this.config.getMinRefreshSafetyWindow());
            this.scheduleTokensRenewal(delay);
        }
    }

    private void scheduleTokensRenewal(Duration delay) {
        if (this.closing.get()) {
            return;
        }
        LOGGER.debug("Scheduling token refresh in {}", (Object)delay);
        try {
            this.tokenRefreshFuture = this.executor.schedule(this::renewTokens, delay.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            if (this.closing.get()) {
                return;
            }
            this.maybeWarn("Failed to schedule next token renewal, forcibly sleeping", null);
            this.sleeping.set(true);
        }
    }

    private void renewTokens() {
        CompletionStage<Tokens> oldTokensStage = this.currentTokensStage;
        this.currentTokensStage = oldTokensStage.thenApply(this::refreshTokens).exceptionally(error -> this.fetchNewTokens());
        this.currentTokensStage.whenComplete((tokens, error) -> this.log((Throwable)error)).whenComplete((tokens, error) -> this.maybeScheduleTokensRenewal((Tokens)tokens));
    }

    private void log(Throwable error) {
        if (error != null) {
            boolean tokensStageCancelled;
            boolean bl = tokensStageCancelled = error instanceof CancellationException && this.closing.get();
            if (tokensStageCancelled) {
                return;
            }
            if (error instanceof CompletionException) {
                error = error.getCause();
            }
            this.maybeWarn("Failed to renew tokens", error);
        } else {
            LOGGER.debug("Successfully renewed tokens");
        }
    }

    Tokens fetchNewTokens() {
        LOGGER.debug("Fetching new tokens");
        if (this.config.getGrantType() == GrantType.CLIENT_CREDENTIALS) {
            ImmutableClientCredentialsTokensRequest body = ImmutableClientCredentialsTokensRequest.builder().scope(this.scope).build();
            HttpResponse httpResponse = this.tokenEndpointClient.newRequest().postForm(body);
            return httpResponse.readEntity(ClientCredentialsTokensResponse.class);
        }
        if (this.config.getGrantType() == GrantType.PASSWORD) {
            ImmutablePasswordTokensRequest body = ImmutablePasswordTokensRequest.builder().username(this.username).password(new String(this.password, StandardCharsets.UTF_8)).scope(this.scope).build();
            HttpResponse httpResponse = this.tokenEndpointClient.newRequest().postForm(body);
            return httpResponse.readEntity(PasswordTokensResponse.class);
        }
        if (this.config.getGrantType() == GrantType.AUTHORIZATION_CODE) {
            try (AuthorizationCodeFlow flow = new AuthorizationCodeFlow(this.config, this.tokenEndpointClient);){
                Tokens tokens = flow.fetchNewTokens();
                return tokens;
            }
        }
        if (this.config.getGrantType() == GrantType.DEVICE_CODE) {
            try (DeviceCodeFlow flow = new DeviceCodeFlow(this.config, this.tokenEndpointClient);){
                Tokens tokens = flow.fetchNewTokens();
                return tokens;
            }
        }
        throw new IllegalStateException("Unsupported grant type: " + (Object)((Object)this.config.getGrantType()));
    }

    Tokens refreshTokens(Tokens currentTokens) {
        if (currentTokens.getRefreshToken() == null) {
            return this.exchangeTokens(currentTokens);
        }
        if (this.isAboutToExpire(currentTokens.getRefreshToken())) {
            throw new MustFetchNewTokensException("Refresh token is about to expire");
        }
        LOGGER.debug("Refreshing tokens");
        ImmutableRefreshTokensRequest body = ImmutableRefreshTokensRequest.builder().refreshToken(currentTokens.getRefreshToken().getPayload()).scope(this.scope).build();
        HttpResponse httpResponse = this.tokenEndpointClient.newRequest().postForm(body);
        return httpResponse.readEntity(RefreshTokensResponse.class);
    }

    Tokens exchangeTokens(Tokens currentToken) {
        if (!this.config.getTokenExchangeEnabled()) {
            throw new MustFetchNewTokensException("Token exchange is disabled");
        }
        LOGGER.debug("Exchanging tokens");
        ImmutableTokensExchangeRequest body = ImmutableTokensExchangeRequest.builder().subjectToken(currentToken.getAccessToken().getPayload()).subjectTokenType(TokenTypeIdentifiers.ACCESS_TOKEN).requestedTokenType(TokenTypeIdentifiers.REFRESH_TOKEN).scope(this.scope).build();
        HttpResponse httpResponse = this.tokenEndpointClient.newRequest().postForm(body);
        return httpResponse.readEntity(TokensExchangeResponse.class);
    }

    private boolean isAboutToExpire(Token token) {
        Instant now = this.config.getClock().get();
        return OAuth2Client.tokenExpirationTime(now, token, this.config.getDefaultRefreshTokenLifespan()).isBefore(now.plus(this.config.getRefreshSafetyWindow()));
    }

    private Duration nextTokenRefresh(Tokens currentTokens, Instant now, Duration minRefreshDelay) {
        if (currentTokens == null) {
            return minRefreshDelay;
        }
        Instant accessExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getAccessToken(), this.config.getDefaultAccessTokenLifespan());
        Instant refreshExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getRefreshToken(), this.config.getDefaultRefreshTokenLifespan());
        return OAuth2Client.shortestDelay(now, accessExpirationTime, refreshExpirationTime, this.config.getRefreshSafetyWindow(), minRefreshDelay);
    }

    static Duration shortestDelay(Instant now, Instant accessExpirationTime, Instant refreshExpirationTime, Duration refreshSafetyWindow, Duration minRefreshDelay) {
        Instant expirationTime = accessExpirationTime.isBefore(refreshExpirationTime) ? accessExpirationTime : refreshExpirationTime;
        Duration delay = Duration.between(now, expirationTime).minus(refreshSafetyWindow);
        if (delay.compareTo(minRefreshDelay) < 0) {
            delay = minRefreshDelay;
        }
        return delay;
    }

    static Instant tokenExpirationTime(Instant now, Token token, Duration defaultLifespan) {
        Instant expirationTime = null;
        if (token != null && (expirationTime = token.getExpirationTime()) == null) {
            try {
                JwtToken jwtToken = JwtToken.parse(token.getPayload());
                expirationTime = jwtToken.getExpirationTime();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (expirationTime == null) {
            expirationTime = now.plus(defaultLifespan);
        }
        return expirationTime;
    }

    private void maybeWarn(String message, Throwable error) {
        boolean shouldWarn;
        Instant now = this.config.getClock().get();
        boolean bl = shouldWarn = this.lastWarn == null || Duration.between(this.lastWarn, now).compareTo(MIN_WARN_INTERVAL) > 0;
        if (shouldWarn) {
            if (error instanceof HttpClientException) {
                LOGGER.warn("{}: {}", (Object)message, (Object)error.toString());
            } else {
                LOGGER.warn(message, error);
            }
            this.lastWarn = now;
        } else {
            LOGGER.debug(message, error);
        }
    }

    static class MustFetchNewTokensException
    extends RuntimeException {
        public MustFetchNewTokensException(String message) {
            super(message);
        }
    }
}

