/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import java.io.PrintStream;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.projectnessie.client.auth.oauth2.DeviceCodeResponse;
import org.projectnessie.client.auth.oauth2.DeviceCodeTokensResponse;
import org.projectnessie.client.auth.oauth2.ImmutableDeviceCodeRequest;
import org.projectnessie.client.auth.oauth2.ImmutableDeviceCodeTokensRequest;
import org.projectnessie.client.auth.oauth2.OAuth2ClientConfig;
import org.projectnessie.client.auth.oauth2.OAuth2Exception;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.http.HttpClient;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.client.http.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DeviceCodeFlow
implements AutoCloseable {
    static final String MSG_PREFIX = "[nessie-oauth2-client] ";
    private static final Logger LOGGER = LoggerFactory.getLogger(DeviceCodeFlow.class);
    private final OAuth2ClientConfig config;
    private final HttpClient httpClient;
    private final PrintStream console;
    private final Duration flowTimeout;
    private final CompletableFuture<Tokens> tokensFuture = new CompletableFuture();
    private final CompletableFuture<Void> closeFuture = new CompletableFuture();
    private final ScheduledExecutorService executor;
    private volatile Duration pollInterval;
    private volatile Future<?> pollFuture;

    DeviceCodeFlow(OAuth2ClientConfig config, HttpClient httpClient) {
        this(config, httpClient, System.out);
    }

    DeviceCodeFlow(OAuth2ClientConfig config, HttpClient httpClient, PrintStream console) {
        this.config = config;
        this.httpClient = httpClient;
        this.console = console;
        this.flowTimeout = config.getDeviceCodeFlowTimeout();
        this.pollInterval = config.getDeviceCodeFlowPollInterval();
        this.closeFuture.thenRun(this::doClose);
        LOGGER.debug("Device Code Flow: started");
        this.executor = Executors.newSingleThreadScheduledExecutor();
    }

    @Override
    public void close() {
        this.closeFuture.complete(null);
    }

    private void doClose() {
        LOGGER.debug("Device Code Flow: closing");
        this.executor.shutdownNow();
        this.pollFuture = null;
    }

    private void abort() {
        Future<?> pollFuture = this.pollFuture;
        if (pollFuture != null) {
            pollFuture.cancel(true);
        }
        this.tokensFuture.cancel(true);
    }

    public Tokens fetchNewTokens() {
        DeviceCodeResponse response = this.requestDeviceCode();
        this.checkPollInterval(response.getInterval());
        this.console.println();
        this.console.println("[nessie-oauth2-client] ======= Nessie authentication required =======");
        this.console.println("[nessie-oauth2-client] Browse to the following URL:");
        this.console.println(MSG_PREFIX + response.getVerificationUri());
        this.console.println("[nessie-oauth2-client] And enter the code:");
        this.console.println(MSG_PREFIX + response.getUserCode());
        this.printExpirationNotice(response.getExpiresIn());
        this.console.println();
        this.console.flush();
        this.pollFuture = this.executor.submit(() -> this.pollForNewTokens(response.getDeviceCode()));
        try {
            return this.tokensFuture.get(this.flowTimeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            LOGGER.error("Timed out waiting for user to authorize device.");
            this.abort();
            throw new RuntimeException("Timed out waiting for user to authorize device", e);
        }
        catch (InterruptedException e) {
            this.abort();
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            this.abort();
            Throwable cause = e.getCause();
            LOGGER.error("Authentication failed: " + cause.getMessage());
            if (cause instanceof HttpClientException) {
                throw (HttpClientException)cause;
            }
            throw new RuntimeException(cause);
        }
    }

    private DeviceCodeResponse requestDeviceCode() {
        ImmutableDeviceCodeRequest body = ImmutableDeviceCodeRequest.builder().scope(this.config.getScope().orElse(null)).build();
        return this.httpClient.newRequest(this.config.getResolvedDeviceAuthEndpoint()).postForm(body).readEntity(DeviceCodeResponse.class);
    }

    private void checkPollInterval(Duration serverPollInterval) {
        if (!this.config.ignoreDeviceCodeFlowServerPollInterval() && serverPollInterval != null && serverPollInterval.compareTo(this.pollInterval) > 0) {
            LOGGER.debug("Device Code Flow: server requested minimum poll interval of {} seconds", (Object)serverPollInterval.getSeconds());
            this.pollInterval = serverPollInterval;
        }
    }

    private void printExpirationNotice(Duration expiresIn) {
        long seconds = expiresIn.getSeconds();
        String exp = seconds < 60L ? seconds + " seconds" : (seconds % 60L == 0L ? seconds / 60L + " minutes" : seconds / 60L + " minutes and " + seconds % 60L + " seconds");
        this.console.println("[nessie-oauth2-client] (The code will expire in " + exp + ")");
    }

    private void pollForNewTokens(String deviceCode) {
        try {
            LOGGER.debug("Device Code Flow: polling for new tokens");
            ImmutableDeviceCodeTokensRequest body = ImmutableDeviceCodeTokensRequest.builder().deviceCode(deviceCode).scope(this.config.getScope().orElse(null)).build();
            HttpResponse response = this.httpClient.newRequest(this.config.getResolvedTokenEndpoint()).postForm(body);
            Tokens tokens = response.readEntity(DeviceCodeTokensResponse.class);
            LOGGER.debug("Device Code Flow: new tokens received");
            this.tokensFuture.complete(tokens);
        }
        catch (OAuth2Exception e) {
            switch (e.getErrorCode()) {
                case "authorization_pending": {
                    LOGGER.debug("Device Code Flow: waiting for authorization to complete");
                    this.pollFuture = this.executor.schedule(() -> this.pollForNewTokens(deviceCode), this.pollInterval.toMillis(), TimeUnit.MILLISECONDS);
                    return;
                }
                case "slow_down": {
                    LOGGER.debug("Device Code Flow: server requested to slow down");
                    Duration pollInterval = this.pollInterval;
                    if (!this.config.ignoreDeviceCodeFlowServerPollInterval()) {
                        this.pollInterval = pollInterval = pollInterval.plus(pollInterval);
                    }
                    this.pollFuture = this.executor.schedule(() -> this.pollForNewTokens(deviceCode), pollInterval.toMillis(), TimeUnit.MILLISECONDS);
                    return;
                }
            }
            this.tokensFuture.completeExceptionally(e);
        }
        catch (Exception e) {
            this.tokensFuture.completeExceptionally(e);
        }
    }
}

