/*
 * Decompiled with CFR 0.152.
 */
package org.radarbase.auth.authentication;

import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.exceptions.SignatureVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.net.URI;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.radarbase.auth.authentication.AlgorithmLoader;
import org.radarbase.auth.config.TokenValidatorConfig;
import org.radarbase.auth.config.TokenVerifierPublicKeyConfig;
import org.radarbase.auth.exception.TokenValidationException;
import org.radarbase.auth.security.jwk.JavaWebKeySet;
import org.radarbase.auth.token.JwtRadarToken;
import org.radarbase.auth.token.RadarToken;
import org.radarbase.auth.token.validation.TokenValidationAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(TokenValidator.class);
    private static final Duration FETCH_TIMEOUT_DEFAULT = Duration.ofMinutes(1L);
    private final TokenValidatorConfig config;
    private final Duration fetchTimeout;
    private final OkHttpClient client;
    private final ObjectMapper mapper;
    private final AlgorithmLoader algorithmLoader;
    private List<JWTVerifier> verifiers;
    private Instant lastFetch = Instant.MIN;

    private TokenValidator(Builder builder) {
        this.mapper = builder.mapper;
        this.fetchTimeout = builder.fetchTimeout;
        this.client = builder.httpClient;
        this.config = builder.config;
        this.algorithmLoader = new AlgorithmLoader(builder.algorithms);
        this.verifiers = builder.verifiers;
    }

    public TokenValidator() {
        this(new Builder().verify());
    }

    public TokenValidator(TokenValidatorConfig config) {
        this(new Builder().config(config).verify());
    }

    public RadarToken validateAccessToken(String token) throws TokenValidationException {
        return this.validateAccessToken(token, true);
    }

    private RadarToken validateAccessToken(String token, boolean tryRefresh) {
        List<JWTVerifier> localVerifiers = this.getVerifiers();
        boolean signatureFailed = false;
        for (JWTVerifier verifier : localVerifiers) {
            try {
                DecodedJWT jwt = verifier.verify(token);
                Map claims = jwt.getClaims();
                LOGGER.debug("Verified JWT header {} and payload {}", (Object)jwt.getHeader(), (Object)jwt.getPayload());
                if (!claims.containsKey("scope")) {
                    throw new TokenValidationException("The required claim scopeis missing from the token");
                }
                return new JwtRadarToken(jwt);
            }
            catch (SignatureVerificationException sve) {
                LOGGER.debug("Client presented a token with an incorrect signature.");
                signatureFailed = true;
            }
            catch (JWTVerificationException ex) {
                LOGGER.debug("Verifier {} with implementation {} did not accept token", (Object)verifier, verifier.getClass());
            }
        }
        if (signatureFailed && tryRefresh) {
            LOGGER.info("Trying to fetch public keys again...");
            try {
                this.refresh();
            }
            catch (TokenValidationException ex) {
                LOGGER.warn("Could not fetch public keys.", (Throwable)ex);
            }
            return this.validateAccessToken(token, false);
        }
        throw new TokenValidationException("No registered validator could authenticate this token");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<JWTVerifier> getVerifiers() {
        TokenValidator tokenValidator = this;
        synchronized (tokenValidator) {
            if (!this.verifiers.isEmpty()) {
                return this.verifiers;
            }
        }
        List<JWTVerifier> localVerifiers = this.loadVerifiers();
        TokenValidator tokenValidator2 = this;
        synchronized (tokenValidator2) {
            this.verifiers = localVerifiers;
            return this.verifiers;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refresh() throws TokenValidationException {
        List<JWTVerifier> localVerifiers = this.loadVerifiers();
        if (!localVerifiers.isEmpty()) {
            TokenValidator tokenValidator = this;
            synchronized (tokenValidator) {
                this.verifiers = localVerifiers;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<JWTVerifier> loadVerifiers() throws TokenValidationException {
        TokenValidator tokenValidator = this;
        synchronized (tokenValidator) {
            if (Instant.now().isBefore(this.lastFetch.plus(this.fetchTimeout))) {
                LOGGER.warn("Fetched public key less than {} ago, denied access.", (Object)this.fetchTimeout);
                throw new TokenValidationException("Not fetching public key more than once every " + this.fetchTimeout);
            }
            this.lastFetch = Instant.now();
        }
        return TokenValidator.streamEmptyIfNull(this.config.getPublicKeyEndpoints()).map(this::algorithmFromServerPublicKeyEndpoint).filter(Objects::nonNull).flatMap(Collection::stream).map(alg -> AlgorithmLoader.buildVerifier(alg, this.config.getResourceName())).collect(Collectors.toList());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<Algorithm> algorithmFromServerPublicKeyEndpoint(URI serverUri) throws TokenValidationException {
        LOGGER.info("Getting the JWT public key at " + serverUri);
        try {
            Request request = new Request.Builder().url(serverUri.toURL()).header("Accept", "application/json").build();
            try (Response response = this.client.newCall(request).execute();){
                if (response.isSuccessful() && response.body() != null) {
                    JavaWebKeySet publicKeyInfo = (JavaWebKeySet)this.mapper.readValue(response.body().string(), JavaWebKeySet.class);
                    LOGGER.debug("Processing {} public keys from public-key endpoint {}", (Object)publicKeyInfo.getKeys().size(), (Object)serverUri.toURL());
                    List<Algorithm> list2 = this.algorithmLoader.loadAlgorithmsFromJavaWebKeys(publicKeyInfo);
                    return list2;
                }
                LOGGER.warn("Could not load newer public keys from {}", (Object)serverUri.toURL());
                List<Algorithm> list = null;
                return list;
            }
        }
        catch (Exception ex) {
            throw new TokenValidationException(ex);
        }
    }

    private static <T> Stream<T> streamEmptyIfNull(Collection<T> collection) {
        return collection != null ? collection.stream() : Stream.empty();
    }

    public static class Builder {
        private static final int DEFAULT_HTTP_TIMEOUT = 30;
        public TokenValidatorConfig config;
        private OkHttpClient httpClient;
        private ObjectMapper mapper;
        private Duration fetchTimeout;
        private List<JWTVerifier> verifiers;
        private List<TokenValidationAlgorithm> algorithms;

        public Builder httpClient(OkHttpClient client) {
            this.httpClient = client;
            return this;
        }

        public Builder objectMapper(ObjectMapper mapper) {
            this.mapper = mapper;
            return this;
        }

        public Builder fetchTimeout(Duration timeout) {
            this.fetchTimeout = timeout;
            return this;
        }

        public Builder config(TokenValidatorConfig config) {
            this.config = config;
            return this;
        }

        public Builder verifiers(List<JWTVerifier> verifiers) {
            this.verifiers = verifiers;
            return this;
        }

        public Builder validators(List<TokenValidationAlgorithm> validators) {
            this.algorithms = validators;
            return this;
        }

        private Builder verify() {
            this.httpClient = this.httpClient == null ? new OkHttpClient.Builder().connectTimeout(30L, TimeUnit.SECONDS).readTimeout(30L, TimeUnit.SECONDS).writeTimeout(30L, TimeUnit.SECONDS).build() : this.httpClient.newBuilder().build();
            if (this.mapper == null) {
                this.mapper = new ObjectMapper();
            }
            if (this.fetchTimeout == null) {
                this.fetchTimeout = FETCH_TIMEOUT_DEFAULT;
            }
            if (this.config == null) {
                this.config = TokenVerifierPublicKeyConfig.readFromFileOrClasspath();
            }
            if (this.algorithms == null) {
                this.algorithms = AlgorithmLoader.defaultAlgorithms();
            }
            if (this.verifiers == null) {
                this.verifiers = List.of();
            }
            return this;
        }

        public TokenValidator build() {
            this.verify();
            return new TokenValidator(this);
        }
    }
}

