/*
 * Decompiled with CFR 0.152.
 */
package org.rhq.server.metrics;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.Row;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.joda.time.DateTime;
import org.joda.time.DateTimeComparator;
import org.joda.time.Duration;
import org.joda.time.ReadableDuration;
import org.rhq.core.domain.measurement.MeasurementDataNumeric;
import org.rhq.core.domain.measurement.composite.MeasurementDataNumericHighLowComposite;
import org.rhq.core.util.StringUtil;
import org.rhq.server.metrics.ArithmeticMeanCalculator;
import org.rhq.server.metrics.Buckets;
import org.rhq.server.metrics.DateTimeService;
import org.rhq.server.metrics.MetricsConfiguration;
import org.rhq.server.metrics.MetricsDAO;
import org.rhq.server.metrics.RawDataInsertedCallback;
import org.rhq.server.metrics.domain.AggregateNumericMetric;
import org.rhq.server.metrics.domain.AggregateType;
import org.rhq.server.metrics.domain.MetricsIndexEntry;
import org.rhq.server.metrics.domain.MetricsTable;
import org.rhq.server.metrics.domain.RawNumericMetric;

public class MetricsServer {
    private final Log log = LogFactory.getLog(MetricsServer.class);
    private DateTimeService dateTimeService = new DateTimeService();
    private MetricsDAO dao;
    private MetricsConfiguration configuration;
    private Semaphore semaphore = new Semaphore(100);
    private boolean shutdown = false;
    private boolean pastAggregationMissed;
    private Long mostRecentRawDataPriorToStartup;

    public void setDAO(MetricsDAO dao) {
        this.dao = dao;
    }

    public void setConfiguration(MetricsConfiguration configuration) {
        this.configuration = configuration;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void init(boolean isNewServerInstall, long serverInstallTime) {
        if (!isNewServerInstall) {
            this.determineMostRecentRawDataSinceLastShutdown(serverInstallTime);
        }
    }

    private void determineMostRecentRawDataSinceLastShutdown(long serverInstallTime) {
        DateTime previousHour = this.currentHour().minusHours(1);
        ResultSet resultSet = this.dao.setFindTimeSliceForIndex(MetricsTable.ONE_HOUR, previousHour.getMillis());
        Row row = resultSet.one();
        while (row == null && previousHour.getMillis() >= serverInstallTime) {
            previousHour = previousHour.minusHours(1);
            resultSet = this.dao.setFindTimeSliceForIndex(MetricsTable.ONE_HOUR, previousHour.getMillis());
            row = resultSet.one();
        }
        if (row == null) {
            this.log.info((Object)"Did not find any raw data in the storage database since the last server shutdown. Raw data aggregate computations are up to date.");
        } else {
            this.mostRecentRawDataPriorToStartup = row.getDate(0).getTime();
            if (this.roundDownToHour(this.mostRecentRawDataPriorToStartup).equals((Object)this.currentHour())) {
                this.log.info((Object)"Raw data aggregate computations are up to date");
            } else {
                this.pastAggregationMissed = true;
                this.log.info((Object)("Found the most recently inserted raw data prior to this server start up with a timestamp of [" + this.mostRecentRawDataPriorToStartup + "]. Aggregates for this data will be computed the " + "next time the aggregation job runs."));
            }
        }
    }

    protected DateTime currentHour() {
        DateTime dt = new DateTime(System.currentTimeMillis());
        return this.dateTimeService.getTimeSlice(dt, Duration.standardHours((long)1L));
    }

    protected DateTime roundDownToHour(long timestamp) {
        return this.dateTimeService.getTimeSlice(new DateTime(timestamp), Duration.standardHours((long)1L));
    }

    public void shutdown() {
        this.shutdown = true;
    }

    public RawNumericMetric findLatestValueForResource(int scheduleId) {
        return this.dao.findLatestRawMetric(scheduleId);
    }

    public Iterable<MeasurementDataNumericHighLowComposite> findDataForResource(int scheduleId, long beginTime, long endTime, int numberOfBuckets) {
        DateTime begin = new DateTime(beginTime);
        if (this.dateTimeService.isInRawDataRange(begin)) {
            Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleId, beginTime, endTime);
            return this.createRawComposites(metrics, beginTime, endTime, numberOfBuckets);
        }
        Iterable<AggregateNumericMetric> metrics = null;
        if (this.dateTimeService.isIn1HourDataRange(begin)) {
            metrics = this.dao.findOneHourMetrics(scheduleId, beginTime, endTime);
        } else if (this.dateTimeService.isIn6HourDataRnage(begin)) {
            metrics = this.dao.findSixHourMetrics(scheduleId, beginTime, endTime);
        } else if (this.dateTimeService.isIn24HourDataRnage(begin)) {
            metrics = this.dao.findTwentyFourHourMetrics(scheduleId, beginTime, endTime);
        } else {
            throw new IllegalArgumentException("beginTime[" + beginTime + "] is outside the accepted range.");
        }
        return this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
    }

    public List<MeasurementDataNumericHighLowComposite> findDataForGroup(List<Integer> scheduleIds, long beginTime, long endTime, int numberOfBuckets) {
        DateTime begin = new DateTime(beginTime);
        if (this.dateTimeService.isInRawDataRange(begin)) {
            Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleIds, beginTime, endTime);
            return this.createRawComposites(metrics, beginTime, endTime, numberOfBuckets);
        }
        Iterable<AggregateNumericMetric> metrics = null;
        if (this.dateTimeService.isIn1HourDataRange(begin)) {
            metrics = this.dao.findOneHourMetrics(scheduleIds, beginTime, endTime);
        } else if (this.dateTimeService.isIn6HourDataRnage(begin)) {
            metrics = this.dao.findSixHourMetrics(scheduleIds, beginTime, endTime);
        } else if (this.dateTimeService.isIn24HourDataRnage(begin)) {
            metrics = this.dao.findTwentyFourHourMetrics(scheduleIds, beginTime, endTime);
        } else {
            throw new IllegalArgumentException("beginTime[" + beginTime + "] is outside the accepted range.");
        }
        return this.createComposites(metrics, beginTime, endTime, numberOfBuckets);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AggregateNumericMetric getSummaryAggregate(int scheduleId, long beginTime, long endTime) {
        long start = System.currentTimeMillis();
        try {
            DateTime begin;
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Calculating resource summary aggregate for [scheduleId: " + scheduleId + ", beginTime: " + beginTime + ", endTime: " + endTime + "]"));
            }
            if (this.dateTimeService.isInRawDataRange(begin = new DateTime(beginTime))) {
                Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleId, beginTime, endTime);
                AggregateNumericMetric aggregateNumericMetric = this.calculateAggregatedRaw(metrics, beginTime);
                return aggregateNumericMetric;
            }
            Iterable<AggregateNumericMetric> metrics = null;
            if (this.dateTimeService.isIn1HourDataRange(begin)) {
                metrics = this.dao.findOneHourMetrics(scheduleId, beginTime, endTime);
            } else if (this.dateTimeService.isIn6HourDataRnage(begin)) {
                metrics = this.dao.findSixHourMetrics(scheduleId, beginTime, endTime);
            } else if (this.dateTimeService.isIn24HourDataRnage(begin)) {
                metrics = this.dao.findTwentyFourHourMetrics(scheduleId, beginTime, endTime);
            } else {
                throw new IllegalArgumentException("beginTime[" + beginTime + "] is outside the accepted range.");
            }
            AggregateNumericMetric aggregateNumericMetric = this.calculateAggregate(metrics, beginTime);
            return aggregateNumericMetric;
        }
        finally {
            long end = System.currentTimeMillis();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating resource summary aggregate in " + (end - start) + " ms"));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AggregateNumericMetric getSummaryAggregate(List<Integer> scheduleIds, long beginTime, long endTime) {
        long start = System.currentTimeMillis();
        try {
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Calculating group summary aggregate for [scheduleIds: [" + StringUtil.listToString(scheduleIds) + "], beginTime: " + beginTime + ", endTime: " + endTime + "]"));
            }
            DateTime begin = new DateTime(beginTime);
            if (this.dateTimeService.isInRawDataRange(new DateTime(beginTime))) {
                Iterable<RawNumericMetric> metrics = this.dao.findRawMetrics(scheduleIds, beginTime, endTime);
                AggregateNumericMetric aggregateNumericMetric = this.calculateAggregatedRaw(metrics, beginTime);
                return aggregateNumericMetric;
            }
            Iterable<AggregateNumericMetric> metrics = null;
            if (this.dateTimeService.isIn1HourDataRange(begin)) {
                metrics = this.dao.findOneHourMetrics(scheduleIds, beginTime, endTime);
            } else if (this.dateTimeService.isIn6HourDataRnage(begin)) {
                metrics = this.dao.findSixHourMetrics(scheduleIds, beginTime, endTime);
            } else if (this.dateTimeService.isIn24HourDataRnage(begin)) {
                metrics = this.dao.findTwentyFourHourMetrics(scheduleIds, beginTime, endTime);
            } else {
                throw new IllegalArgumentException("beginTime[" + beginTime + "] is outside the accepted range.");
            }
            AggregateNumericMetric aggregateNumericMetric = this.calculateAggregate(metrics, beginTime);
            return aggregateNumericMetric;
        }
        finally {
            long end = System.currentTimeMillis();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Finished calculating group summary aggregate in " + (end - start) + " ms"));
            }
        }
    }

    private List<MeasurementDataNumericHighLowComposite> createRawComposites(Iterable<RawNumericMetric> metrics, long beginTime, long endTime, int numberOfBuckets) {
        Buckets buckets = new Buckets(beginTime, endTime, numberOfBuckets);
        for (RawNumericMetric metric : metrics) {
            buckets.insert(metric.getTimestamp(), metric.getValue(), metric.getValue(), metric.getValue());
        }
        ArrayList<MeasurementDataNumericHighLowComposite> data = new ArrayList<MeasurementDataNumericHighLowComposite>();
        for (int i = 0; i < buckets.getNumDataPoints(); ++i) {
            Buckets.Bucket bucket = buckets.get(i);
            data.add(new MeasurementDataNumericHighLowComposite(bucket.getStartTime(), bucket.getAvg(), bucket.getMax(), bucket.getMin()));
        }
        return data;
    }

    private List<MeasurementDataNumericHighLowComposite> createComposites(Iterable<AggregateNumericMetric> metrics, long beginTime, long endTime, int numberOfBuckets) {
        Buckets buckets = new Buckets(beginTime, endTime, numberOfBuckets);
        for (AggregateNumericMetric metric : metrics) {
            buckets.insert(metric.getTimestamp(), metric.getAvg(), metric.getMin(), metric.getMax());
        }
        ArrayList<MeasurementDataNumericHighLowComposite> data = new ArrayList<MeasurementDataNumericHighLowComposite>();
        for (int i = 0; i < buckets.getNumDataPoints(); ++i) {
            Buckets.Bucket bucket = buckets.get(i);
            data.add(new MeasurementDataNumericHighLowComposite(bucket.getStartTime(), bucket.getAvg(), bucket.getMax(), bucket.getMin()));
        }
        return data;
    }

    public void addNumericData(final Set<MeasurementDataNumeric> dataSet, final RawDataInsertedCallback callback) {
        try {
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Inserting " + dataSet.size() + " raw metrics"));
            }
            final long startTime = System.currentTimeMillis();
            final AtomicInteger remainingInserts = new AtomicInteger(dataSet.size());
            for (final MeasurementDataNumeric data : dataSet) {
                this.semaphore.acquire();
                ResultSetFuture resultSetFuture = this.dao.insertRawData(data);
                Futures.addCallback((ListenableFuture)resultSetFuture, (FutureCallback)new FutureCallback<ResultSet>(){

                    public void onSuccess(ResultSet rows) {
                        MetricsServer.this.updateMetricsIndex(data, dataSet.size(), remainingInserts, startTime, callback);
                    }

                    public void onFailure(Throwable throwable) {
                        MetricsServer.this.log.error((Object)("An error occurred while inserting raw data " + data), throwable);
                        callback.onFailure(throwable);
                        MetricsServer.this.semaphore.release();
                    }
                });
            }
        }
        catch (Exception e) {
            this.log.error((Object)"An error occurred while inserting raw numeric data ", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    void updateMetricsIndex(final MeasurementDataNumeric rawData, final int total, final AtomicInteger remainingInserts, final long startTime, final RawDataInsertedCallback callback) {
        long timeSlice = this.dateTimeService.getTimeSlice(new DateTime(rawData.getTimestamp()), this.configuration.getRawTimeSliceDuration()).getMillis();
        ResultSetFuture resultSetFuture = this.dao.updateMetricsIndex(MetricsTable.ONE_HOUR, rawData.getScheduleId(), timeSlice);
        Futures.addCallback((ListenableFuture)resultSetFuture, (FutureCallback)new FutureCallback<ResultSet>(){

            public void onSuccess(ResultSet rows) {
                callback.onSuccess(rawData);
                if (remainingInserts.decrementAndGet() == 0) {
                    long endTime = System.currentTimeMillis();
                    if (MetricsServer.this.log.isDebugEnabled()) {
                        MetricsServer.this.log.debug((Object)("Finished inserting " + total + " raw metrics in " + (endTime - startTime) + " ms"));
                    }
                    callback.onFinish();
                }
                MetricsServer.this.semaphore.release();
            }

            public void onFailure(Throwable throwable) {
                MetricsServer.this.log.error((Object)("An error occurred while trying to update " + (Object)((Object)MetricsTable.INDEX) + " for raw data " + rawData));
                callback.onFailure(throwable);
                MetricsServer.this.semaphore.release();
            }
        });
    }

    public Iterable<AggregateNumericMetric> calculateAggregates() {
        DateTime theHour = this.currentHour();
        if (this.pastAggregationMissed) {
            this.calculateAggregates(this.roundDownToHour(this.mostRecentRawDataPriorToStartup).plusHours(1).getMillis());
            this.pastAggregationMissed = false;
            return this.calculateAggregates(theHour.getMillis());
        }
        return this.calculateAggregates(theHour.getMillis());
    }

    private Iterable<AggregateNumericMetric> calculateAggregates(long startTime) {
        List<AggregateNumericMetric> updatedSchedules;
        DateTime dt = new DateTime(startTime);
        DateTime currentHour = this.dateTimeService.getTimeSlice(dt, this.configuration.getRawTimeSliceDuration());
        DateTime lastHour = currentHour.minusHours(1);
        long hourTimeSlice = lastHour.getMillis();
        long sixHourTimeSlice = this.dateTimeService.getTimeSlice(lastHour, this.configuration.getOneHourTimeSliceDuration()).getMillis();
        long twentyFourHourTimeSlice = this.dateTimeService.getTimeSlice(lastHour, this.configuration.getSixHourTimeSliceDuration()).getMillis();
        List<AggregateNumericMetric> newOneHourAggregates = null;
        newOneHourAggregates = updatedSchedules = this.aggregateRawData(hourTimeSlice);
        if (!updatedSchedules.isEmpty()) {
            this.dao.deleteMetricsIndexEntries(MetricsTable.ONE_HOUR, hourTimeSlice);
            this.updateMetricsIndex(MetricsTable.SIX_HOUR, updatedSchedules, this.configuration.getOneHourTimeSliceDuration());
        }
        if (!(updatedSchedules = this.calculateAggregates(MetricsTable.ONE_HOUR, MetricsTable.SIX_HOUR, sixHourTimeSlice, this.configuration.getOneHourTimeSliceDuration())).isEmpty()) {
            this.dao.deleteMetricsIndexEntries(MetricsTable.SIX_HOUR, sixHourTimeSlice);
            this.updateMetricsIndex(MetricsTable.TWENTY_FOUR_HOUR, updatedSchedules, this.configuration.getSixHourTimeSliceDuration());
        }
        if (!(updatedSchedules = this.calculateAggregates(MetricsTable.SIX_HOUR, MetricsTable.TWENTY_FOUR_HOUR, twentyFourHourTimeSlice, this.configuration.getSixHourTimeSliceDuration())).isEmpty()) {
            this.dao.deleteMetricsIndexEntries(MetricsTable.TWENTY_FOUR_HOUR, twentyFourHourTimeSlice);
        }
        return newOneHourAggregates;
    }

    private void updateMetricsIndex(MetricsTable bucket, Iterable<AggregateNumericMetric> metrics, Duration duration) {
        TreeMap<Integer, Long> updates = new TreeMap<Integer, Long>();
        for (AggregateNumericMetric metric : metrics) {
            updates.put(metric.getScheduleId(), this.dateTimeService.getTimeSlice(new DateTime(metric.getTimestamp()), duration).getMillis());
        }
        this.dao.updateMetricsIndex(bucket, updates);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<AggregateNumericMetric> aggregateRawData(long theHour) {
        long start = System.currentTimeMillis();
        try {
            Iterable<MetricsIndexEntry> indexEntries = this.dao.findMetricsIndexEntries(MetricsTable.ONE_HOUR, theHour);
            ArrayList<AggregateNumericMetric> oneHourMetrics = new ArrayList<AggregateNumericMetric>();
            for (MetricsIndexEntry indexEntry : indexEntries) {
                DateTime startTime = indexEntry.getTime();
                DateTime endTime = startTime.plusMinutes(60);
                Iterable<RawNumericMetric> rawMetrics = this.dao.findRawMetrics(indexEntry.getScheduleId(), startTime.getMillis(), endTime.getMillis());
                AggregateNumericMetric aggregatedRaw = this.calculateAggregatedRaw(rawMetrics, startTime.getMillis());
                aggregatedRaw.setScheduleId(indexEntry.getScheduleId());
                oneHourMetrics.add(aggregatedRaw);
            }
            for (AggregateNumericMetric metric : oneHourMetrics) {
                this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MIN, metric.getMin());
                this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MAX, metric.getMax());
                this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.AVG, metric.getAvg());
            }
            ArrayList<AggregateNumericMetric> arrayList = oneHourMetrics;
            return arrayList;
        }
        finally {
            long end = System.currentTimeMillis();
            if (this.log.isInfoEnabled()) {
                this.log.info((Object)("Finished computing aggregates for table [" + (Object)((Object)MetricsTable.RAW) + "]" + (end - start) + " ms"));
            }
        }
    }

    private AggregateNumericMetric calculateAggregatedRaw(Iterable<RawNumericMetric> rawMetrics, long timestamp) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (RawNumericMetric metric : rawMetrics) {
            double value = metric.getValue();
            if (count == 0) {
                max = min = value;
            }
            if (value < min) {
                min = value;
            } else if (value > max) {
                max = value;
            }
            mean.add(value);
            ++count;
        }
        return new AggregateNumericMetric(0, mean.getArithmeticMean(), min, max, timestamp);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<AggregateNumericMetric> calculateAggregates(MetricsTable fromTable, MetricsTable toTable, long timeSlice, Duration nextDuration) {
        if (this.log.isDebugEnabled()) {
            this.log.debug((Object)("Preparing to compute aggregates for data in " + (Object)((Object)fromTable) + " table"));
        }
        long start = System.currentTimeMillis();
        try {
            DateTimeComparator dateTimeComparator;
            DateTime startTime = new DateTime(timeSlice);
            DateTime endTime = startTime.plus((ReadableDuration)nextDuration);
            DateTime currentHour = this.currentHour();
            if (this.log.isDebugEnabled()) {
                this.log.debug((Object)("Time slice start time is [" + startTime + "] and the end time is [" + endTime + "]."));
            }
            if ((dateTimeComparator = DateTimeComparator.getInstance()).compare((Object)currentHour, (Object)endTime) < 0) {
                if (this.log.isDebugEnabled()) {
                    this.log.debug((Object)("Skipping aggregation for " + (Object)((Object)fromTable) + " since the time slice has not yet completed"));
                }
                List<AggregateNumericMetric> list = Collections.emptyList();
                return list;
            }
            Iterable<MetricsIndexEntry> indexEntries = this.dao.findMetricsIndexEntries(toTable, timeSlice);
            ArrayList<AggregateNumericMetric> toMetrics = new ArrayList<AggregateNumericMetric>();
            for (MetricsIndexEntry indexEntry : indexEntries) {
                Iterable<AggregateNumericMetric> metrics = null;
                switch (fromTable) {
                    case ONE_HOUR: {
                        metrics = this.dao.findOneHourMetrics(indexEntry.getScheduleId(), startTime.getMillis(), endTime.getMillis());
                        break;
                    }
                    case SIX_HOUR: {
                        metrics = this.dao.findSixHourMetrics(indexEntry.getScheduleId(), startTime.getMillis(), endTime.getMillis());
                        break;
                    }
                    default: {
                        metrics = this.dao.findTwentyFourHourMetrics(indexEntry.getScheduleId(), startTime.getMillis(), endTime.getMillis());
                    }
                }
                AggregateNumericMetric aggregatedMetric = this.calculateAggregate(metrics, startTime.getMillis());
                aggregatedMetric.setScheduleId(indexEntry.getScheduleId());
                toMetrics.add(aggregatedMetric);
            }
            switch (toTable) {
                case ONE_HOUR: {
                    this.insertOneHourAggregates(toMetrics);
                    break;
                }
                case SIX_HOUR: {
                    this.insertSixHourAggregates(toMetrics);
                    break;
                }
                default: {
                    this.insertTwentyFourHourAggregates(toMetrics);
                }
            }
            ArrayList<AggregateNumericMetric> arrayList = toMetrics;
            return arrayList;
        }
        finally {
            long end = System.currentTimeMillis();
            if (this.log.isInfoEnabled()) {
                this.log.info((Object)("Finished computing aggregates for table [" + (Object)((Object)fromTable) + "] " + (end - start) + " ms"));
            }
        }
    }

    private void insertOneHourAggregates(List<AggregateNumericMetric> metrics) {
        for (AggregateNumericMetric metric : metrics) {
            this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MIN, metric.getMin());
            this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MAX, metric.getMax());
            this.dao.insertOneHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.AVG, metric.getAvg());
        }
    }

    private void insertSixHourAggregates(List<AggregateNumericMetric> metrics) {
        for (AggregateNumericMetric metric : metrics) {
            this.dao.insertSixHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MIN, metric.getMin());
            this.dao.insertSixHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MAX, metric.getMax());
            this.dao.insertSixHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.AVG, metric.getAvg());
        }
    }

    private void insertTwentyFourHourAggregates(List<AggregateNumericMetric> metrics) {
        for (AggregateNumericMetric metric : metrics) {
            this.dao.insertTwentyFourHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MIN, metric.getMin());
            this.dao.insertTwentyFourHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.MAX, metric.getMax());
            this.dao.insertTwentyFourHourData(metric.getScheduleId(), metric.getTimestamp(), AggregateType.AVG, metric.getAvg());
        }
    }

    private AggregateNumericMetric calculateAggregate(Iterable<AggregateNumericMetric> metrics, long timestamp) {
        double min;
        double max = min = Double.NaN;
        int count = 0;
        ArithmeticMeanCalculator mean = new ArithmeticMeanCalculator();
        for (AggregateNumericMetric metric : metrics) {
            if (count == 0) {
                min = metric.getMin();
                max = metric.getMax();
            }
            if (metric.getMin() < min) {
                min = metric.getMin();
            } else if (metric.getMax() > max) {
                max = metric.getMax();
            }
            mean.add(metric.getAvg());
            ++count;
        }
        return new AggregateNumericMetric(0, mean.getArithmeticMean(), min, max, timestamp);
    }
}

