package org.richfaces.component;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.faces.FacesException;
import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.component.UIComponentBase;
import javax.faces.component.UIInput;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;
import javax.faces.el.EvaluationException;
import javax.faces.el.MethodBinding;
import javax.faces.el.ValueBinding;
import javax.faces.event.AbortProcessingException;
import javax.faces.event.FacesEvent;
import javax.faces.event.ValueChangeEvent;
import javax.faces.event.ValueChangeListener;
import javax.faces.model.DataModel;
import javax.faces.validator.Validator;
import javax.faces.validator.ValidatorException;

import org.ajax4jsf.Messages;
import org.ajax4jsf.model.DataVisitor;
import org.ajax4jsf.model.ExtendedDataModel;
import org.ajax4jsf.model.SequenceDataModel;
import org.richfaces.component.util.MessageUtil;
import org.richfaces.model.OrderingListDataModel;

public abstract class UIOrderingList extends UIOrderingBaseComponent {

	protected void processDecodes(FacesContext faces, Object argument) {
		if (!this.isRendered())
			return;
		this.decode(faces);

		SubmittedValue submittedValue = UIOrderingList.this.submittedValueHolder;
		if (submittedValue != null) {
			Object modelValue = getValue();
			Iterator iterator = submittedValue.dataMap.entrySet().iterator();
			while (iterator.hasNext()) {
				Entry entry = (Entry) iterator.next();
				Object value = entry.getValue();
				
				if (!isSuitableValue(modelValue, value)) {
					String messageText = Messages.getMessage(
							Messages.INVALID_VALUE, MessageUtil.getLabel(faces, this), value);
					
					FacesMessage message = new FacesMessage(FacesMessage.SEVERITY_ERROR, messageText, null);
					faces.addMessage(this.getClientId(faces), message);

					setValid(false);
					break;
				}
			}
		}
		
		if (isImmediate()) {
			executeValidate(faces);
		}
		
        if (!isValid()) {
            faces.renderResponse();
        }

        this.iterate(faces, decodeVisitor, argument);
	}

	private transient Map dataMap;

	protected static final class SubmittedValue implements Serializable {
		/**
		 * 
		 */
		private static final long serialVersionUID = 5860506816451180551L;
		
		private Map dataMap;
		private Collection selection;
		private Object activeItem;
		
		public SubmittedValue(Map dataMap, Set selection, Object activeItem) {
			this.dataMap = dataMap;
			this.selection = selection;
			this.activeItem = activeItem;
		}
	}

	private final class ModelItemState implements ItemState {
		private Collection selectedItems;
		private Object activeItem;

		public ModelItemState(Collection selectedItems, Object activeItem) {
			super();
			this.selectedItems = selectedItems;
			this.activeItem = activeItem;
		}

		public boolean isSelected() {
			return selectedItems != null && selectedItems.contains(getRowData());
		}

		public boolean isActive() {
			return activeItem != null && activeItem.equals(getRowData());
		}
	}

	protected ExtendedDataModel createDataModel() {
		Map modelMap = null;

		if (submittedValueHolder != null) {
			modelMap = submittedValueHolder.dataMap;
		} else {
			modelMap = this.dataMap;
		}
		
		if (modelMap != null) {
			OrderingListDataModel dataModel = new OrderingListDataModel();
			dataModel.setWrappedData(modelMap);
			return dataModel;
		} else {
			DataModel dataModel = createDataModel(getValue());
			return new SequenceDataModel(dataModel);
		}
	}

	private transient SubmittedValue submittedValueHolder = null;
	
	protected static final class ValueHolder implements Serializable {

		/**
		 * 
		 */
		private static final long serialVersionUID = -4216115242421460529L;
		
		private Collection selection;
		private boolean selectionSet;
		
		private Map map;
	}

	private Collection selection;
	private boolean selectionSet;

	public void addValueChangeListener(ValueChangeListener listener) {
		addFacesListener(listener);
	}


	public abstract MethodBinding getValueChangeListener();

	public ValueChangeListener[] getValueChangeListeners() {
		return (ValueChangeListener[]) getFacesListeners(ValueChangeListener.class);
	}

	public abstract boolean isImmediate();

	public abstract boolean isRequired();

	public abstract boolean isValid();


	public void removeValueChangeListener(ValueChangeListener listener) {
		removeFacesListener(listener);
	}

	public void setSubmittedString(Map submittedString, Set selection, Object activeItem) {
		this.submittedValueHolder = new SubmittedValue(submittedString, selection, activeItem);
	}

	protected Object saveIterationSubmittedState() {
		return submittedValueHolder;
	}

	protected void restoreIterationSubmittedState(Object object) {
		this.submittedValueHolder = (SubmittedValue) object;
	}

	protected Object saveIterationState() {
		ValueHolder valueHolder = new ValueHolder();
		valueHolder.map = dataMap;
		valueHolder.selection = selection;
		valueHolder.selectionSet = selectionSet;
		return valueHolder;
	}

	protected void restoreIterationState(Object object) {
		ValueHolder valueHolder = (ValueHolder) object;
		dataMap = valueHolder.map;
		selection = valueHolder.selection;
		selectionSet = valueHolder.selectionSet;
	}

	public abstract void setImmediate(boolean immediate);

	public abstract void setRequired(boolean required);

	public abstract void setValid(boolean valid);

	public abstract void setValueChangeListener(MethodBinding valueChangeMethod);

	public abstract Converter getConverter();

	public abstract void setConverter(Converter converter);

	/**
	 * <p>Specialized decode behavior on top of that provided by the
	 * superclass.  In addition to the standard
	 * <code>processDecodes</code> behavior inherited from {@link
	 * UIComponentBase}, calls <code>validate()</code> if the the
	 * <code>immediate</code> property is true; if the component is
	 * invalid afterwards or a <code>RuntimeException</code> is thrown,
	 * calls {@link FacesContext#renderResponse}.  </p>
	 * @exception NullPointerException {@inheritDoc}     
	 */ 
	public void processDecodes(FacesContext context) {

		if (context == null) {
			throw new NullPointerException();
		}

		// Skip processing if our rendered flag is false
		if (!isRendered()) {
			return;
		}

		super.processDecodes(context);
	}

	/**
	 * <p>In addition to the standard <code>processValidators</code> behavior
	 * inherited from {@link UIComponentBase}, calls <code>validate()</code>
	 * if the <code>immediate</code> property is false (which is the 
	 * default);  if the component is invalid afterwards, calls
	 * {@link FacesContext#renderResponse}.
	 * If a <code>RuntimeException</code> is thrown during
	 * validation processing, calls {@link FacesContext#renderResponse}
	 * and re-throw the exception.
	 * </p>
	 * @exception NullPointerException {@inheritDoc}    
	 */ 
	public void processValidators(FacesContext context) {

		if (context == null) {
			throw new NullPointerException();
		}

		// Skip processing if our rendered flag is false
		if (!isRendered()) {
			return;
		}

		super.processValidators(context);
	}

	/**
	 * <p>In addition to the standard <code>processUpdates</code> behavior
	 * inherited from {@link UIComponentBase}, calls
	 * <code>updateModel()</code>.
	 * If the component is invalid afterwards, calls
	 * {@link FacesContext#renderResponse}.
	 * If a <code>RuntimeException</code> is thrown during
	 * update processing, calls {@link FacesContext#renderResponse}
	 * and re-throw the exception.
	 * </p>
	 * @exception NullPointerException {@inheritDoc}     
	 */ 
	public void processUpdates(FacesContext context) {

		if (context == null) {
			throw new NullPointerException();
		}

		// Skip processing if our rendered flag is false
		if (!isRendered()) {
			return;
		}

		super.processUpdates(context);
	}

	/**
	 * <p>In addition to to the default {@link UIComponent#broadcast}
	 * processing, pass the {@link ValueChangeEvent} being broadcast to the
	 * method referenced by <code>valueChangeListener</code> (if any).</p>
	 *
	 * @param event {@link FacesEvent} to be broadcast
	 *
	 * @exception AbortProcessingException Signal the JavaServer Faces
	 *  implementation that no further processing on the current event
	 *  should be performed
	 * @exception IllegalArgumentException if the implementation class
	 *  of this {@link FacesEvent} is not supported by this component
	 * @exception NullPointerException if <code>event</code> is
	 * <code>null</code>
	 */
	public void broadcast(FacesEvent event)
	throws AbortProcessingException {

		// Perform standard superclass processing
		super.broadcast(event);

		if (event instanceof ValueChangeEvent) {
			MethodBinding method = getValueChangeListener();
			if (method != null) {
				FacesContext context = getFacesContext();
				method.invoke(context, new Object[] { event });
			}
		}

	}

	protected final UpdateModelCommand updateSelectionCommand = new UpdateModelCommand() {

		public void execute(FacesContext context) {
			if (selectionSet) {
				ValueBinding vb = getValueBinding("selection");
				if (vb != null) {
					vb.setValue(context, selection);
					selection = null;
					selectionSet = false;
				}
			}
		}
		
	};
	
	/**
	 * <p>Perform the following algorithm to update the model dataMap
	 * associated with this {@link UIInput}, if any, as appropriate.</p>
	 * <ul>
	 * <li>If the <code>valid</code> property of this component is
	 *     <code>false</code>, take no further action.</li>
	 * <li>If the <code>localValueSet</code> property of this component is
	 *     <code>false</code>, take no further action.</li>
	 * <li>If no {@link ValueBinding} for <code>value</code> exists,
	 *     take no further action.</li>
	 * <li>Call <code>setValue()</code> method of the {@link ValueBinding}
	 *      to update the value that the {@link ValueBinding} points at.</li>
	 * <li>If the <code>setValue()</code> method returns successfully:
	 *     <ul>
	 *     <li>Clear the local value of this {@link UIInput}.</li>
	 *     <li>Set the <code>localValueSet</code> property of this
	 *         {@link UIInput} to false.</li>
	 *     </ul></li>
	 * <li>If the <code>setValue()</code> method call fails:
	 *     <ul>
	 *     <li>Enqueue an error message by calling <code>addMessage()</code>
	 *         on the specified {@link FacesContext} instance.</li>
	 *     <li>Set the <code>valid</code> property of this {@link UIInput}
	 *         to <code>false</code>.</li>
	 *     </ul></li>
	 * </ul>
	 *
	 * @param context {@link FacesContext} for the request we are processing
	 *
	 * @exception NullPointerException if <code>context</code>
	 *  is <code>null</code>
	 */
	public void updateModel(FacesContext context) {

		if (context == null) {
			throw new NullPointerException();
		}

		if (!isValid()) {
			return;
		}

		updateModel(context, updateValueCommand);
		updateModel(context, updateSelectionCommand);
		updateModel(context, updateActiveItemCommand);
	}


	// ------------------------------------------------------ Validation Methods


	/**
	 * <p>Perform the following algorithm to validate the local value of
	 * this {@link UIInput}.</p>
	 * <ul>
	 * <li>Retrieve the submitted value with <code>getSubmittedValue()</code>.
	 *   If this returns null, exit without further processing.  (This
	 *   indicates that no value was submitted for this component.)</li>
	 *
	 * <li> Convert the submitted value into a "local value" of the
	 * appropriate dataMap type by calling {@link #getConvertedValue}.</li>
	 *
	 * <li>Validate the property by calling {@link #validateValue}.</li>
	 *
	 * <li>If the <code>valid</code> property of this component is still
	 *     <code>true</code>, retrieve the previous value of the component
	 *     (with <code>getValue()</code>), store the new local value using
	 *     <code>setValue()</code>, and reset the submitted value to 
	 *     null.  If the local value is different from
	 *     the previous value of this component, fire a
	 *     {@link ValueChangeEvent} to be broadcast to all interested
	 *     listeners.</li>
	 * </ul>
	 *
	 * <p>Application components implementing {@link UIInput} that wish to
	 * perform validation with logic embedded in the component should perform
	 * their own correctness checks, and then call the
	 * <code>super.validate()</code> method to perform the standard
	 * processing described above.</p>
	 *
	 * @param context The {@link FacesContext} for the current request
	 *
	 * @exception NullPointerException if <code>context</code>
	 *  is null
	 */
	public void validate(FacesContext context) {

		if (context == null) {
			throw new NullPointerException();
		}

		// Submitted value == null means "the component was not submitted
		// at all";  validation should not continue
		if (submittedValueHolder == null) {
			return;
		}

		Object previousValue = getValue();
		Object newValue = null;

		try {
			if (previousValue == null) {
				previousValue = Collections.EMPTY_LIST;
			}

			OrderingListDataModel dataModel = (OrderingListDataModel) getExtendedDataModel();

			try {
				final ArrayList list = new ArrayList(dataModel.getRowCount());
				
				walk(context, new DataVisitor() {
					public void process(FacesContext context, Object rowKey,
							Object argument) throws IOException {

						setRowKey(context, rowKey);
						list.add(getRowData());
					}

				}, null);

				newValue = createContainer(list, previousValue.getClass());
			} catch (IOException e) {
				throw new ConverterException(e.getLocalizedMessage(), e);
			}
		}
		catch (ConverterException ce) {
			Object submittedValue = submittedValueHolder;
			addConversionErrorMessage(context, ce, submittedValue);
			setValid(false);
		}	

		validateValue(context, newValue);

		// If our value is valid, store the new value, erase the
		// "submitted" value, and emit a ValueChangeEvent if appropriate
		if (isValid()) {
			setSelection(submittedValueHolder.selection);

			setActiveItem(submittedValueHolder.activeItem);
			
			setValue(newValue);
			setTranslatedState();

			if (compareValues(previousValue, newValue)) {
				queueEvent(new ValueChangeEvent(this, previousValue, newValue));
			}

			this.dataMap = this.submittedValueHolder.dataMap;
			this.submittedValueHolder = null;
		}
	}

	protected void resetDataModel() {
		super.resetDataModel();

		this.dataMap = null;

		if (this.submittedValueHolder != null) {
			setTranslatedRenderingState();
		}
	}

	/**
	 *
	 * <p>Set the "valid" property according to the below algorithm.</p>
	 *
	 * <ul>
	 *
	 * <li>If the <code>valid</code> property on this component is still
	 *     <code>true</code>, and the <code>required</code> property is also
	 *     true, ensure that the local value is not empty (where "empty" is
	 *     defined as <code>null</code> or a zero-length String.  If the local
	 *     value is empty:
	 *     <ul>
	 *     <li>Enqueue an appropriate error message by calling the
	 *         <code>addMessage()</code> method on the <code>FacesContext</code>
	 *         instance for the current request.</li>
	 *     <li>Set the <code>valid</code> property on this component to
	 *         <code>false</code>.</li>
	 *     </ul></li>
	 * <li>If the <code>valid</code> property on this component is still
	 *     <code>true</code>, and the local value is not empty, call the
	 *     <code>validate()</code> method of each {@link Validator}
	 *     registered for this {@link UIInput}, followed by the method
	 *     pointed at by the <code>validatorBinding</code> property (if any).
	 *     If any of these validators or the method throws a
	 *     {@link ValidatorException}, catch the exception, add
	 *     its message (if any) to the {@link FacesContext}, and set
	 *     the <code>valid</code> property of this component to false.</li>
	 *
	 * </ul>
	 *
	 */

	protected void validateValue(FacesContext context, Object newValue) {
		// If our value is valid, enforce the required property if present
		if (isValid() && isRequired() && isEmpty(newValue)) {
			FacesMessage message =
				MessageFactory.getMessage(context, UIInput.REQUIRED_MESSAGE_ID);
			message.setSeverity(FacesMessage.SEVERITY_ERROR);
			context.addMessage(getClientId(context), message);
			setValid(false);
		}

		// If our value is valid and not empty, call all validators
		if (isValid() && !isEmpty(newValue)) {
			Validator[] validators = getValidators();
			for (int i = 0; i < validators.length; i++) {
				Validator validator = (Validator) validators[i];
				try { 
					validator.validate(context, this, newValue);
				}
				catch (ValidatorException ve) {
					// If the validator throws an exception, we're
					// invalid, and we need to add a message
					setValid(false);
					FacesMessage message = ve.getFacesMessage();
					if (message != null) {
						message.setSeverity(FacesMessage.SEVERITY_ERROR);
						context.addMessage(getClientId(context), message);
					}
				}
			}

			MethodBinding validator = getValidator();
			if (validator != null) {
				try {
					validator.invoke(context,
							new Object[] { context, this, newValue});
				}
				catch (EvaluationException ee) {
					if (ee.getCause() instanceof ValidatorException) {
						ValidatorException ve =
							(ValidatorException) ee.getCause();

						// If the validator throws an exception, we're
						// invalid, and we need to add a message
						setValid(false);
						FacesMessage message = ve.getFacesMessage();
						if (message != null) {
							message.setSeverity(FacesMessage.SEVERITY_ERROR);
							context.addMessage(getClientId(context), message);
						}
					} else {
						// Otherwise, rethrow the EvaluationException
						throw ee;
					}
				}
			}
		}
	}

	public ItemState getItemState() {
		if (submittedValueHolder != null) {
			return new ModelItemState(submittedValueHolder.selection, 
					submittedValueHolder.activeItem);
		} else {
			return new ModelItemState(getSelection(), getActiveItem());
		}
	}

	public abstract String getControlsType();
	public abstract void setControlsType(String type);

	public Collection getSelection() {
		if (this.selection != null) {
			return this.selection;
		} else {
			ValueBinding vb = getValueBinding("selection");
			if (vb != null) {
				return (Collection) vb.getValue(FacesContext.getCurrentInstance());
			}
		}
		
		return null;
	}
	
	public void setSelection(Collection collection) {
		this.selection = collection;
		this.selectionSet = true;
	}
	
	public Object saveState(FacesContext faces) {
		Object[] state = new Object[5];
		state[0] = super.saveState(faces);
		
		Object rowKey = getRowKey();

		final HashSet selectionKeySet = new HashSet();
		final HashSet activeItemSet = new HashSet(1);
		try {
			walk(faces, new DataVisitor() {

				public void process(FacesContext context, Object rowKey,
						Object argument) throws IOException {

					setRowKey(context, rowKey);
					Object data = getRowData();
					
					if (data != null) {
						if (data.equals(activeItem)) {
							activeItemSet.add(rowKey);
						}
						
						if (selection != null && selection.contains(data)) {
							selectionKeySet.add(rowKey);
						}
					}
				}
				
			}, null);
		} catch (IOException e) {
			throw new FacesException(e.getLocalizedMessage(), e);
		}

		state[1] = selectionKeySet;
		state[2] = this.selectionSet ? Boolean.TRUE : Boolean.FALSE;
		
		state[3] = activeItemSet.isEmpty() ? null : activeItemSet.iterator().next();
		state[4] = this.activeItemSet ? Boolean.TRUE : Boolean.FALSE;

		return state;
	}
	
	public void restoreState(FacesContext faces, Object object) {
		Object[] state = (Object[]) object;
		
		super.restoreState(faces, state[0]);
		
		this.selection = (Collection) state[1];
		this.selectionSet = ((Boolean) state[2]).booleanValue();
		
		this.activeItem = state[3];
		this.activeItemSet = ((Boolean) state[4]).booleanValue();
	}

}
