/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.connectivity;

import android.net.LinkAddress;
import android.net.LinkProperties;
import android.net.Network;
import android.net.NetworkUtils;
import android.net.RouteInfo;
import android.os.SystemClock;
import android.system.ErrnoException;
import android.system.Os;
import android.system.OsConstants;
import android.system.StructTimeval;
import android.text.TextUtils;
import android.util.Pair;
import com.android.internal.util.IndentingPrintWriter;
import java.io.Closeable;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.SocketAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import libcore.io.IoUtils;

public class NetworkDiagnostics {
    private static final String TAG = "NetworkDiagnostics";
    private static final InetAddress TEST_DNS4 = NetworkUtils.numericToInetAddress("8.8.8.8");
    private static final InetAddress TEST_DNS6 = NetworkUtils.numericToInetAddress("2001:4860:4860::8888");
    private final Network mNetwork;
    private final LinkProperties mLinkProperties;
    private final Integer mInterfaceIndex;
    private final long mTimeoutMs;
    private final long mStartTime;
    private final long mDeadlineTime;
    private final CountDownLatch mCountDownLatch;
    private final Map<InetAddress, Measurement> mIcmpChecks = new HashMap<InetAddress, Measurement>();
    private final Map<Pair<InetAddress, InetAddress>, Measurement> mExplicitSourceIcmpChecks = new HashMap<Pair<InetAddress, InetAddress>, Measurement>();
    private final Map<InetAddress, Measurement> mDnsUdpChecks = new HashMap<InetAddress, Measurement>();
    private final String mDescription;

    private static final long now() {
        return SystemClock.elapsedRealtime();
    }

    public NetworkDiagnostics(Network network, LinkProperties lp, long timeoutMs) {
        this.mNetwork = network;
        this.mLinkProperties = lp;
        this.mInterfaceIndex = NetworkDiagnostics.getInterfaceIndex(this.mLinkProperties.getInterfaceName());
        this.mTimeoutMs = timeoutMs;
        this.mStartTime = NetworkDiagnostics.now();
        this.mDeadlineTime = this.mStartTime + this.mTimeoutMs;
        if (this.mLinkProperties.isReachable(TEST_DNS4)) {
            this.mLinkProperties.addDnsServer(TEST_DNS4);
        }
        if (this.mLinkProperties.hasGlobalIPv6Address() || this.mLinkProperties.hasIPv6DefaultRoute()) {
            this.mLinkProperties.addDnsServer(TEST_DNS6);
        }
        for (RouteInfo route : this.mLinkProperties.getRoutes()) {
            if (!route.hasGateway()) continue;
            InetAddress gateway = route.getGateway();
            this.prepareIcmpMeasurement(gateway);
            if (!route.isIPv6Default()) continue;
            this.prepareExplicitSourceIcmpMeasurements(gateway);
        }
        for (InetAddress nameserver : this.mLinkProperties.getDnsServers()) {
            this.prepareIcmpMeasurement(nameserver);
            this.prepareDnsMeasurement(nameserver);
        }
        this.mCountDownLatch = new CountDownLatch(this.totalMeasurementCount());
        this.startMeasurements();
        this.mDescription = "ifaces{" + TextUtils.join((CharSequence)",", this.mLinkProperties.getAllInterfaceNames()) + "}" + " index{" + this.mInterfaceIndex + "}" + " network{" + this.mNetwork + "}" + " nethandle{" + this.mNetwork.getNetworkHandle() + "}";
    }

    private static Integer getInterfaceIndex(String ifname) {
        try {
            NetworkInterface ni = NetworkInterface.getByName(ifname);
            return ni.getIndex();
        }
        catch (NullPointerException | SocketException e) {
            return null;
        }
    }

    private void prepareIcmpMeasurement(InetAddress target) {
        if (!this.mIcmpChecks.containsKey(target)) {
            Measurement measurement = new Measurement();
            measurement.thread = new Thread(new IcmpCheck(target, measurement));
            this.mIcmpChecks.put(target, measurement);
        }
    }

    private void prepareExplicitSourceIcmpMeasurements(InetAddress target) {
        for (LinkAddress l : this.mLinkProperties.getLinkAddresses()) {
            Pair<InetAddress, InetAddress> srcTarget;
            InetAddress source = l.getAddress();
            if (!(source instanceof Inet6Address) || !l.isGlobalPreferred() || this.mExplicitSourceIcmpChecks.containsKey(srcTarget = new Pair<InetAddress, InetAddress>(source, target))) continue;
            Measurement measurement = new Measurement();
            measurement.thread = new Thread(new IcmpCheck(source, target, measurement));
            this.mExplicitSourceIcmpChecks.put(srcTarget, measurement);
        }
    }

    private void prepareDnsMeasurement(InetAddress target) {
        if (!this.mDnsUdpChecks.containsKey(target)) {
            Measurement measurement = new Measurement();
            measurement.thread = new Thread(new DnsUdpCheck(target, measurement));
            this.mDnsUdpChecks.put(target, measurement);
        }
    }

    private int totalMeasurementCount() {
        return this.mIcmpChecks.size() + this.mExplicitSourceIcmpChecks.size() + this.mDnsUdpChecks.size();
    }

    private void startMeasurements() {
        for (Measurement measurement : this.mIcmpChecks.values()) {
            measurement.thread.start();
        }
        for (Measurement measurement : this.mExplicitSourceIcmpChecks.values()) {
            measurement.thread.start();
        }
        for (Measurement measurement : this.mDnsUdpChecks.values()) {
            measurement.thread.start();
        }
    }

    public void waitForMeasurements() {
        try {
            this.mCountDownLatch.await(this.mDeadlineTime - NetworkDiagnostics.now(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public List<Measurement> getMeasurements() {
        ArrayList<Measurement> measurements = new ArrayList<Measurement>(this.totalMeasurementCount());
        for (Map.Entry<InetAddress, Measurement> entry : this.mIcmpChecks.entrySet()) {
            if (!(entry.getKey() instanceof Inet4Address)) continue;
            measurements.add(entry.getValue());
        }
        for (Map.Entry<Object, Measurement> entry : this.mExplicitSourceIcmpChecks.entrySet()) {
            if (!(((Pair)entry.getKey()).first instanceof Inet4Address)) continue;
            measurements.add(entry.getValue());
        }
        for (Map.Entry<Object, Measurement> entry : this.mDnsUdpChecks.entrySet()) {
            if (!(entry.getKey() instanceof Inet4Address)) continue;
            measurements.add(entry.getValue());
        }
        for (Map.Entry<Object, Measurement> entry : this.mIcmpChecks.entrySet()) {
            if (!(entry.getKey() instanceof Inet6Address)) continue;
            measurements.add(entry.getValue());
        }
        for (Map.Entry<Object, Measurement> entry : this.mExplicitSourceIcmpChecks.entrySet()) {
            if (!(((Pair)entry.getKey()).first instanceof Inet6Address)) continue;
            measurements.add(entry.getValue());
        }
        for (Map.Entry<Object, Measurement> entry : this.mDnsUdpChecks.entrySet()) {
            if (!(entry.getKey() instanceof Inet6Address)) continue;
            measurements.add(entry.getValue());
        }
        return measurements;
    }

    public void dump(IndentingPrintWriter pw) {
        pw.println("NetworkDiagnostics:" + this.mDescription);
        long unfinished = this.mCountDownLatch.getCount();
        if (unfinished > 0L) {
            pw.println("WARNING: countdown wait incomplete: " + unfinished + " unfinished measurements");
        }
        pw.increaseIndent();
        for (Measurement m : this.getMeasurements()) {
            String prefix = m.checkSucceeded() ? "." : "F";
            pw.println(prefix + "  " + m.toString());
        }
        pw.decreaseIndent();
    }

    private class DnsUdpCheck
    extends SimpleSocketCheck
    implements Runnable {
        private static final int TIMEOUT_SEND = 100;
        private static final int TIMEOUT_RECV = 500;
        private static final int DNS_SERVER_PORT = 53;
        private static final int RR_TYPE_A = 1;
        private static final int RR_TYPE_AAAA = 28;
        private static final int PACKET_BUFSIZE = 512;
        private final Random mRandom;
        private final int mQueryType;

        private String responseCodeStr(int rcode) {
            try {
                return DnsResponseCode.values()[rcode].toString();
            }
            catch (IndexOutOfBoundsException e) {
                return String.valueOf(rcode);
            }
        }

        public DnsUdpCheck(InetAddress target, Measurement measurement) {
            super(target, measurement);
            this.mRandom = new Random();
            this.mQueryType = this.mAddressFamily == OsConstants.AF_INET6 ? 28 : 1;
            this.mMeasurement.description = "DNS UDP dst{" + this.mTarget.getHostAddress() + "}";
        }

        @Override
        public void run() {
            if (this.mMeasurement.finishTime > 0L) {
                NetworkDiagnostics.this.mCountDownLatch.countDown();
                return;
            }
            try {
                this.setupSocket(OsConstants.SOCK_DGRAM, OsConstants.IPPROTO_UDP, 100L, 500L, 53);
            }
            catch (ErrnoException | IOException e) {
                this.mMeasurement.recordFailure(e.toString());
                return;
            }
            this.mMeasurement.description = this.mMeasurement.description + " src{" + this.getSocketAddressString() + "}";
            String sixRandomDigits = String.valueOf(this.mRandom.nextInt(900000) + 100000);
            this.mMeasurement.description = this.mMeasurement.description + " qtype{" + this.mQueryType + "}" + " qname{" + sixRandomDigits + "-android-ds.metric.gstatic.com}";
            byte[] dnsPacket = this.getDnsQueryPacket(sixRandomDigits);
            int count = 0;
            this.mMeasurement.startTime = NetworkDiagnostics.now();
            while (NetworkDiagnostics.now() < NetworkDiagnostics.this.mDeadlineTime - 1000L) {
                ++count;
                try {
                    Os.write(this.mFileDescriptor, dnsPacket, 0, dnsPacket.length);
                }
                catch (ErrnoException | InterruptedIOException e) {
                    this.mMeasurement.recordFailure(e.toString());
                    break;
                }
                try {
                    ByteBuffer reply = ByteBuffer.allocate(512);
                    Os.read(this.mFileDescriptor, reply);
                    String rcodeStr = reply.limit() > 3 ? " " + this.responseCodeStr(reply.get(3) & 0xF) : "";
                    this.mMeasurement.recordSuccess("1/" + count + rcodeStr);
                    break;
                }
                catch (ErrnoException | InterruptedIOException e) {
                }
            }
            if (this.mMeasurement.finishTime == 0L) {
                this.mMeasurement.recordFailure("0/" + count);
            }
            this.close();
        }

        private byte[] getDnsQueryPacket(String sixRandomDigits) {
            byte[] rnd = sixRandomDigits.getBytes(StandardCharsets.US_ASCII);
            return new byte[]{(byte)this.mRandom.nextInt(), (byte)this.mRandom.nextInt(), 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 17, rnd[0], rnd[1], rnd[2], rnd[3], rnd[4], rnd[5], 45, 97, 110, 100, 114, 111, 105, 100, 45, 100, 115, 6, 109, 101, 116, 114, 105, 99, 7, 103, 115, 116, 97, 116, 105, 99, 3, 99, 111, 109, 0, 0, (byte)this.mQueryType, 0, 1};
        }
    }

    private class IcmpCheck
    extends SimpleSocketCheck
    implements Runnable {
        private static final int TIMEOUT_SEND = 100;
        private static final int TIMEOUT_RECV = 300;
        private static final int ICMPV4_ECHO_REQUEST = 8;
        private static final int ICMPV6_ECHO_REQUEST = 128;
        private static final int PACKET_BUFSIZE = 512;
        private final int mProtocol;
        private final int mIcmpType;

        public IcmpCheck(InetAddress source, InetAddress target, Measurement measurement) {
            super(source, target, measurement);
            if (this.mAddressFamily == OsConstants.AF_INET6) {
                this.mProtocol = OsConstants.IPPROTO_ICMPV6;
                this.mIcmpType = 128;
                this.mMeasurement.description = "ICMPv6";
            } else {
                this.mProtocol = OsConstants.IPPROTO_ICMP;
                this.mIcmpType = 8;
                this.mMeasurement.description = "ICMPv4";
            }
            this.mMeasurement.description = this.mMeasurement.description + " dst{" + this.mTarget.getHostAddress() + "}";
        }

        public IcmpCheck(InetAddress target, Measurement measurement) {
            this(null, target, measurement);
        }

        @Override
        public void run() {
            if (this.mMeasurement.finishTime > 0L) {
                NetworkDiagnostics.this.mCountDownLatch.countDown();
                return;
            }
            try {
                this.setupSocket(OsConstants.SOCK_DGRAM, this.mProtocol, 100L, 300L, 0);
            }
            catch (ErrnoException | IOException e) {
                this.mMeasurement.recordFailure(e.toString());
                return;
            }
            this.mMeasurement.description = this.mMeasurement.description + " src{" + this.getSocketAddressString() + "}";
            byte[] icmpPacket = new byte[]{(byte)this.mIcmpType, 0, 0, 0, 0, 0, 0, 0};
            int count = 0;
            this.mMeasurement.startTime = NetworkDiagnostics.now();
            while (NetworkDiagnostics.now() < NetworkDiagnostics.this.mDeadlineTime - 400L) {
                icmpPacket[icmpPacket.length - 1] = (byte)(++count);
                try {
                    Os.write(this.mFileDescriptor, icmpPacket, 0, icmpPacket.length);
                }
                catch (ErrnoException | InterruptedIOException e) {
                    this.mMeasurement.recordFailure(e.toString());
                    break;
                }
                try {
                    ByteBuffer reply = ByteBuffer.allocate(512);
                    Os.read(this.mFileDescriptor, reply);
                    this.mMeasurement.recordSuccess("1/" + count);
                    break;
                }
                catch (ErrnoException | InterruptedIOException e) {
                }
            }
            if (this.mMeasurement.finishTime == 0L) {
                this.mMeasurement.recordFailure("0/" + count);
            }
            this.close();
        }
    }

    private class SimpleSocketCheck
    implements Closeable {
        protected final InetAddress mSource;
        protected final InetAddress mTarget;
        protected final int mAddressFamily;
        protected final Measurement mMeasurement;
        protected FileDescriptor mFileDescriptor;
        protected SocketAddress mSocketAddress;

        protected SimpleSocketCheck(InetAddress source, InetAddress target, Measurement measurement) {
            this.mMeasurement = measurement;
            if (target instanceof Inet6Address) {
                InetAddress targetWithScopeId = null;
                if (target.isLinkLocalAddress() && NetworkDiagnostics.this.mInterfaceIndex != null) {
                    try {
                        targetWithScopeId = Inet6Address.getByAddress(null, target.getAddress(), NetworkDiagnostics.this.mInterfaceIndex);
                    }
                    catch (UnknownHostException e) {
                        this.mMeasurement.recordFailure(e.toString());
                    }
                }
                this.mTarget = targetWithScopeId != null ? targetWithScopeId : target;
                this.mAddressFamily = OsConstants.AF_INET6;
            } else {
                this.mTarget = target;
                this.mAddressFamily = OsConstants.AF_INET;
            }
            this.mSource = source;
        }

        protected SimpleSocketCheck(InetAddress target, Measurement measurement) {
            this(null, target, measurement);
        }

        protected void setupSocket(int sockType, int protocol, long writeTimeout, long readTimeout, int dstPort) throws ErrnoException, IOException {
            this.mFileDescriptor = Os.socket(this.mAddressFamily, sockType, protocol);
            Os.setsockoptTimeval(this.mFileDescriptor, OsConstants.SOL_SOCKET, OsConstants.SO_SNDTIMEO, StructTimeval.fromMillis(writeTimeout));
            Os.setsockoptTimeval(this.mFileDescriptor, OsConstants.SOL_SOCKET, OsConstants.SO_RCVTIMEO, StructTimeval.fromMillis(readTimeout));
            NetworkDiagnostics.this.mNetwork.bindSocket(this.mFileDescriptor);
            if (this.mSource != null) {
                Os.bind(this.mFileDescriptor, this.mSource, 0);
            }
            Os.connect(this.mFileDescriptor, this.mTarget, dstPort);
            this.mSocketAddress = Os.getsockname(this.mFileDescriptor);
        }

        protected String getSocketAddressString() {
            InetSocketAddress inetSockAddr = (InetSocketAddress)this.mSocketAddress;
            InetAddress localAddr = inetSockAddr.getAddress();
            return String.format(localAddr instanceof Inet6Address ? "[%s]:%d" : "%s:%d", localAddr.getHostAddress(), inetSockAddr.getPort());
        }

        @Override
        public void close() {
            IoUtils.closeQuietly(this.mFileDescriptor);
        }
    }

    public class Measurement {
        private static final String SUCCEEDED = "SUCCEEDED";
        private static final String FAILED = "FAILED";
        private boolean succeeded;
        String description = "";
        long startTime;
        long finishTime;
        String result = "";
        Thread thread;

        public boolean checkSucceeded() {
            return this.succeeded;
        }

        void recordSuccess(String msg) {
            this.maybeFixupTimes();
            this.succeeded = true;
            this.result = "SUCCEEDED: " + msg;
            if (NetworkDiagnostics.this.mCountDownLatch != null) {
                NetworkDiagnostics.this.mCountDownLatch.countDown();
            }
        }

        void recordFailure(String msg) {
            this.maybeFixupTimes();
            this.succeeded = false;
            this.result = "FAILED: " + msg;
            if (NetworkDiagnostics.this.mCountDownLatch != null) {
                NetworkDiagnostics.this.mCountDownLatch.countDown();
            }
        }

        private void maybeFixupTimes() {
            if (this.finishTime == 0L) {
                this.finishTime = NetworkDiagnostics.now();
            }
            if (this.startTime == 0L) {
                this.startTime = this.finishTime;
            }
        }

        public String toString() {
            return this.description + ": " + this.result + " (" + (this.finishTime - this.startTime) + "ms)";
        }
    }

    public static enum DnsResponseCode {
        NOERROR,
        FORMERR,
        SERVFAIL,
        NXDOMAIN,
        NOTIMP,
        REFUSED;

    }
}

