/*
 * Decompiled with CFR 0.152.
 */
package com.android.server.backup;

import android.content.Context;
import android.util.Slog;
import com.android.server.backup.utils.DataStreamCodec;
import com.android.server.backup.utils.DataStreamFileCodec;
import com.android.server.backup.utils.PasswordUtils;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.security.SecureRandom;

public final class BackupPasswordManager {
    private static final String TAG = "BackupPasswordManager";
    private static final boolean DEBUG = false;
    private static final int BACKUP_PW_FILE_VERSION = 2;
    private static final int DEFAULT_PW_FILE_VERSION = 1;
    private static final String PASSWORD_VERSION_FILE_NAME = "pwversion";
    private static final String PASSWORD_HASH_FILE_NAME = "pwhash";
    public static final String PBKDF_CURRENT = "PBKDF2WithHmacSHA1";
    public static final String PBKDF_FALLBACK = "PBKDF2WithHmacSHA1And8bit";
    private final SecureRandom mRng;
    private final Context mContext;
    private final File mBaseStateDir;
    private String mPasswordHash;
    private int mPasswordVersion;
    private byte[] mPasswordSalt;

    BackupPasswordManager(Context context, File baseStateDir, SecureRandom secureRandom) {
        this.mContext = context;
        this.mRng = secureRandom;
        this.mBaseStateDir = baseStateDir;
        this.loadStateFromFilesystem();
    }

    boolean hasBackupPassword() {
        this.mContext.enforceCallingOrSelfPermission("android.permission.BACKUP", "hasBackupPassword");
        return this.mPasswordHash != null && this.mPasswordHash.length() > 0;
    }

    boolean backupPasswordMatches(String password) {
        return !this.hasBackupPassword() || this.passwordMatchesSaved(password);
    }

    boolean setBackupPassword(String currentPassword, String newPassword) {
        this.mContext.enforceCallingOrSelfPermission("android.permission.BACKUP", "setBackupPassword");
        if (!this.passwordMatchesSaved(currentPassword)) {
            return false;
        }
        try {
            this.getPasswordVersionFileCodec().serialize(2);
            this.mPasswordVersion = 2;
        }
        catch (IOException e) {
            Slog.e(TAG, "Unable to write backup pw version; password not changed");
            return false;
        }
        if (newPassword == null || newPassword.isEmpty()) {
            return this.clearPassword();
        }
        try {
            byte[] salt = this.randomSalt();
            String newPwHash = PasswordUtils.buildPasswordHash(PBKDF_CURRENT, newPassword, salt, 10000);
            this.getPasswordHashFileCodec().serialize(new BackupPasswordHash(newPwHash, salt));
            this.mPasswordHash = newPwHash;
            this.mPasswordSalt = salt;
            return true;
        }
        catch (IOException e) {
            Slog.e(TAG, "Unable to set backup password");
            return false;
        }
    }

    private boolean usePbkdf2Fallback() {
        return this.mPasswordVersion < 2;
    }

    private boolean clearPassword() {
        File passwordHashFile = this.getPasswordHashFile();
        if (passwordHashFile.exists() && !passwordHashFile.delete()) {
            Slog.e(TAG, "Unable to clear backup password");
            return false;
        }
        this.mPasswordHash = null;
        this.mPasswordSalt = null;
        return true;
    }

    private void loadStateFromFilesystem() {
        try {
            this.mPasswordVersion = this.getPasswordVersionFileCodec().deserialize();
        }
        catch (IOException e) {
            Slog.e(TAG, "Unable to read backup pw version");
            this.mPasswordVersion = 1;
        }
        try {
            BackupPasswordHash hash = this.getPasswordHashFileCodec().deserialize();
            this.mPasswordHash = hash.hash;
            this.mPasswordSalt = hash.salt;
        }
        catch (IOException e) {
            Slog.e(TAG, "Unable to read saved backup pw hash");
        }
    }

    private boolean passwordMatchesSaved(String candidatePassword) {
        return this.passwordMatchesSaved(PBKDF_CURRENT, candidatePassword) || this.usePbkdf2Fallback() && this.passwordMatchesSaved(PBKDF_FALLBACK, candidatePassword);
    }

    private boolean passwordMatchesSaved(String algorithm, String candidatePassword) {
        if (this.mPasswordHash == null) {
            return candidatePassword == null || candidatePassword.equals("");
        }
        if (candidatePassword == null || candidatePassword.length() == 0) {
            return false;
        }
        String candidatePasswordHash = PasswordUtils.buildPasswordHash(algorithm, candidatePassword, this.mPasswordSalt, 10000);
        return this.mPasswordHash.equalsIgnoreCase(candidatePasswordHash);
    }

    private byte[] randomSalt() {
        int bitsPerByte = 8;
        byte[] array2 = new byte[512 / bitsPerByte];
        this.mRng.nextBytes(array2);
        return array2;
    }

    private DataStreamFileCodec<Integer> getPasswordVersionFileCodec() {
        return new DataStreamFileCodec<Integer>(new File(this.mBaseStateDir, PASSWORD_VERSION_FILE_NAME), new PasswordVersionFileCodec());
    }

    private DataStreamFileCodec<BackupPasswordHash> getPasswordHashFileCodec() {
        return new DataStreamFileCodec<BackupPasswordHash>(this.getPasswordHashFile(), new PasswordHashFileCodec());
    }

    private File getPasswordHashFile() {
        return new File(this.mBaseStateDir, PASSWORD_HASH_FILE_NAME);
    }

    private static final class PasswordHashFileCodec
    implements DataStreamCodec<BackupPasswordHash> {
        private PasswordHashFileCodec() {
        }

        @Override
        public void serialize(BackupPasswordHash backupPasswordHash, DataOutputStream dataOutputStream) throws IOException {
            dataOutputStream.writeInt(backupPasswordHash.salt.length);
            dataOutputStream.write(backupPasswordHash.salt);
            dataOutputStream.writeUTF(backupPasswordHash.hash);
        }

        @Override
        public BackupPasswordHash deserialize(DataInputStream dataInputStream) throws IOException {
            int saltLen = dataInputStream.readInt();
            byte[] salt = new byte[saltLen];
            dataInputStream.readFully(salt);
            String hash = dataInputStream.readUTF();
            return new BackupPasswordHash(hash, salt);
        }
    }

    private static final class PasswordVersionFileCodec
    implements DataStreamCodec<Integer> {
        private PasswordVersionFileCodec() {
        }

        @Override
        public void serialize(Integer integer2, DataOutputStream dataOutputStream) throws IOException {
            dataOutputStream.write(integer2);
        }

        @Override
        public Integer deserialize(DataInputStream dataInputStream) throws IOException {
            return dataInputStream.readInt();
        }
    }

    private static final class BackupPasswordHash {
        public String hash;
        public byte[] salt;

        BackupPasswordHash(String hash, byte[] salt) {
            this.hash = hash;
            this.salt = salt;
        }
    }
}

