/*
 * Decompiled with CFR 0.152.
 */
package com.android.internal.telephony.uicc.asn1;

import com.android.internal.telephony.uicc.IccUtils;
import com.android.internal.telephony.uicc.asn1.Asn1Decoder;
import com.android.internal.telephony.uicc.asn1.InvalidAsn1DataException;
import com.android.internal.telephony.uicc.asn1.TagNotFoundException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public final class Asn1Node {
    private static final int INT_BYTES = 4;
    private static final List<Asn1Node> EMPTY_NODE_LIST = Collections.emptyList();
    private static final byte[] TRUE_BYTES = new byte[]{-1};
    private static final byte[] FALSE_BYTES = new byte[]{0};
    private final int mTag;
    private final boolean mConstructed;
    private final List<Asn1Node> mChildren;
    private byte[] mDataBytes;
    private int mDataOffset;
    private int mDataLength;
    private int mEncodedLength;

    public static Builder newBuilder(int tag) {
        return new Builder(tag);
    }

    private static boolean isConstructedTag(int tag) {
        byte[] tagBytes = IccUtils.unsignedIntToBytes(tag);
        return (tagBytes[0] & 0x20) != 0;
    }

    private static int calculateEncodedBytesNumForLength(int length) {
        int len = 1;
        if (length > 127) {
            len += IccUtils.byteNumForUnsignedInt(length);
        }
        return len;
    }

    Asn1Node(int tag, byte[] src, int offset, int length) {
        this.mTag = tag;
        this.mConstructed = Asn1Node.isConstructedTag(tag);
        this.mDataBytes = src;
        this.mDataOffset = offset;
        this.mDataLength = length;
        this.mChildren = this.mConstructed ? new ArrayList() : EMPTY_NODE_LIST;
        this.mEncodedLength = IccUtils.byteNumForUnsignedInt(this.mTag) + Asn1Node.calculateEncodedBytesNumForLength(this.mDataLength) + this.mDataLength;
    }

    private Asn1Node(int tag, List<Asn1Node> children) {
        this.mTag = tag;
        this.mConstructed = true;
        this.mChildren = children;
        this.mDataLength = 0;
        int size = children.size();
        for (int i = 0; i < size; ++i) {
            this.mDataLength += children.get((int)i).mEncodedLength;
        }
        this.mEncodedLength = IccUtils.byteNumForUnsignedInt(this.mTag) + Asn1Node.calculateEncodedBytesNumForLength(this.mDataLength) + this.mDataLength;
    }

    public int getTag() {
        return this.mTag;
    }

    public boolean isConstructed() {
        return this.mConstructed;
    }

    public boolean hasChild(int tag, int ... tags) throws InvalidAsn1DataException {
        try {
            this.getChild(tag, tags);
        }
        catch (TagNotFoundException e) {
            return false;
        }
        return true;
    }

    public Asn1Node getChild(int tag, int ... tags) throws TagNotFoundException, InvalidAsn1DataException {
        if (!this.mConstructed) {
            throw new TagNotFoundException(tag);
        }
        int index = 0;
        Asn1Node node = this;
        while (node != null) {
            List<Asn1Node> children = node.getChildren();
            int size = children.size();
            Asn1Node foundChild = null;
            for (int i = 0; i < size; ++i) {
                Asn1Node child = children.get(i);
                if (child.getTag() != tag) continue;
                foundChild = child;
                break;
            }
            node = foundChild;
            if (index >= tags.length) break;
            tag = tags[index++];
        }
        if (node == null) {
            throw new TagNotFoundException(tag);
        }
        return node;
    }

    public List<Asn1Node> getChildren(int tag) throws TagNotFoundException, InvalidAsn1DataException {
        if (!this.mConstructed) {
            return EMPTY_NODE_LIST;
        }
        List<Asn1Node> children = this.getChildren();
        if (children.isEmpty()) {
            return EMPTY_NODE_LIST;
        }
        ArrayList<Asn1Node> output = new ArrayList<Asn1Node>();
        int size = children.size();
        for (int i = 0; i < size; ++i) {
            Asn1Node child = children.get(i);
            if (child.getTag() != tag) continue;
            output.add(child);
        }
        return output.isEmpty() ? EMPTY_NODE_LIST : output;
    }

    public List<Asn1Node> getChildren() throws InvalidAsn1DataException {
        if (!this.mConstructed) {
            return EMPTY_NODE_LIST;
        }
        if (this.mDataBytes != null) {
            Asn1Decoder subDecoder = new Asn1Decoder(this.mDataBytes, this.mDataOffset, this.mDataLength);
            while (subDecoder.hasNextNode()) {
                this.mChildren.add(subDecoder.nextNode());
            }
            this.mDataBytes = null;
            this.mDataOffset = 0;
        }
        return this.mChildren;
    }

    public boolean hasValue() {
        return !this.mConstructed && this.mDataBytes != null;
    }

    public int asInteger() throws InvalidAsn1DataException {
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        try {
            return IccUtils.bytesToInt(this.mDataBytes, this.mDataOffset, this.mDataLength);
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", e);
        }
    }

    public long asRawLong() throws InvalidAsn1DataException {
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        try {
            return IccUtils.bytesToRawLong(this.mDataBytes, this.mDataOffset, this.mDataLength);
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", e);
        }
    }

    public String asString() throws InvalidAsn1DataException {
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        try {
            return new String(this.mDataBytes, this.mDataOffset, this.mDataLength, StandardCharsets.UTF_8);
        }
        catch (IndexOutOfBoundsException e) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", e);
        }
    }

    public byte[] asBytes() throws InvalidAsn1DataException {
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        byte[] output = new byte[this.mDataLength];
        try {
            System.arraycopy((byte[])this.mDataBytes, (int)this.mDataOffset, (byte[])output, (int)0, (int)this.mDataLength);
        }
        catch (IndexOutOfBoundsException e) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", e);
        }
        return output;
    }

    public int asBits() throws InvalidAsn1DataException {
        int bits;
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        try {
            bits = IccUtils.bytesToInt(this.mDataBytes, this.mDataOffset + 1, this.mDataLength - 1);
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", e);
        }
        for (int i = this.mDataLength - 1; i < 4; ++i) {
            bits <<= 8;
        }
        return Integer.reverse(bits);
    }

    public boolean asBoolean() throws InvalidAsn1DataException {
        if (this.mConstructed) {
            throw new IllegalStateException("Cannot get value of a constructed node.");
        }
        if (this.mDataBytes == null) {
            throw new InvalidAsn1DataException(this.mTag, "Data bytes cannot be null.");
        }
        if (this.mDataLength != 1) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes as boolean: length=" + this.mDataLength);
        }
        if (this.mDataOffset < 0 || this.mDataOffset >= this.mDataBytes.length) {
            throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes.", new ArrayIndexOutOfBoundsException(this.mDataOffset));
        }
        if (this.mDataBytes[this.mDataOffset] == -1) {
            return Boolean.TRUE;
        }
        if (this.mDataBytes[this.mDataOffset] == 0) {
            return Boolean.FALSE;
        }
        throw new InvalidAsn1DataException(this.mTag, "Cannot parse data bytes as boolean: " + this.mDataBytes[this.mDataOffset]);
    }

    public int getEncodedLength() {
        return this.mEncodedLength;
    }

    public int getDataLength() {
        return this.mDataLength;
    }

    public void writeToBytes(byte[] dest, int offset) {
        if (offset < 0 || offset + this.mEncodedLength > dest.length) {
            throw new IndexOutOfBoundsException("Not enough space to write. Required bytes: " + this.mEncodedLength);
        }
        this.write(dest, offset);
    }

    public byte[] toBytes() {
        byte[] dest = new byte[this.mEncodedLength];
        this.write(dest, 0);
        return dest;
    }

    public String toHex() {
        return IccUtils.bytesToHexString(this.toBytes());
    }

    public String getHeadAsHex() {
        String headHex = IccUtils.bytesToHexString(IccUtils.unsignedIntToBytes(this.mTag));
        if (this.mDataLength <= 127) {
            headHex = headHex + IccUtils.byteToHex((byte)this.mDataLength);
        } else {
            byte[] lenBytes = IccUtils.unsignedIntToBytes(this.mDataLength);
            headHex = headHex + IccUtils.byteToHex((byte)(lenBytes.length | 0x80));
            headHex = headHex + IccUtils.bytesToHexString(lenBytes);
        }
        return headHex;
    }

    private int write(byte[] dest, int offset) {
        offset += IccUtils.unsignedIntToBytes(this.mTag, dest, offset);
        if (this.mDataLength <= 127) {
            dest[offset++] = (byte)this.mDataLength;
        } else {
            int lenLen = IccUtils.unsignedIntToBytes(this.mDataLength, dest, ++offset);
            dest[offset - 1] = (byte)(lenLen | 0x80);
            offset += lenLen;
        }
        if (this.mConstructed && this.mDataBytes == null) {
            int size = this.mChildren.size();
            for (int i = 0; i < size; ++i) {
                Asn1Node child = this.mChildren.get(i);
                offset = child.write(dest, offset);
            }
        } else if (this.mDataBytes != null) {
            System.arraycopy((byte[])this.mDataBytes, (int)this.mDataOffset, (byte[])dest, (int)offset, (int)this.mDataLength);
            offset += this.mDataLength;
        }
        return offset;
    }

    public static final class Builder {
        private final int mTag;
        private final List<Asn1Node> mChildren;

        private Builder(int tag) {
            if (!Asn1Node.isConstructedTag(tag)) {
                throw new IllegalArgumentException("Builder should be created for a constructed tag: " + tag);
            }
            this.mTag = tag;
            this.mChildren = new ArrayList<Asn1Node>();
        }

        public Builder addChild(Asn1Node child) {
            this.mChildren.add(child);
            return this;
        }

        public Builder addChild(Builder child) {
            this.mChildren.add(child.build());
            return this;
        }

        public Builder addChildren(byte[] encodedBytes) throws InvalidAsn1DataException {
            Asn1Decoder subDecoder = new Asn1Decoder(encodedBytes, 0, encodedBytes.length);
            while (subDecoder.hasNextNode()) {
                this.mChildren.add(subDecoder.nextNode());
            }
            return this;
        }

        public Builder addChildAsInteger(int tag, int value) {
            if (Asn1Node.isConstructedTag(tag)) {
                throw new IllegalStateException("Cannot set value of a constructed tag: " + tag);
            }
            byte[] dataBytes = IccUtils.signedIntToBytes(value);
            this.addChild(new Asn1Node(tag, dataBytes, 0, dataBytes.length));
            return this;
        }

        public Builder addChildAsString(int tag, String value) {
            if (Asn1Node.isConstructedTag(tag)) {
                throw new IllegalStateException("Cannot set value of a constructed tag: " + tag);
            }
            byte[] dataBytes = value.getBytes(StandardCharsets.UTF_8);
            this.addChild(new Asn1Node(tag, dataBytes, 0, dataBytes.length));
            return this;
        }

        public Builder addChildAsBytes(int tag, byte[] value) {
            if (Asn1Node.isConstructedTag(tag)) {
                throw new IllegalStateException("Cannot set value of a constructed tag: " + tag);
            }
            this.addChild(new Asn1Node(tag, value, 0, value.length));
            return this;
        }

        public Builder addChildAsBytesFromHex(int tag, String hex) {
            return this.addChildAsBytes(tag, IccUtils.hexStringToBytes(hex));
        }

        public Builder addChildAsBits(int tag, int value) {
            if (Asn1Node.isConstructedTag(tag)) {
                throw new IllegalStateException("Cannot set value of a constructed tag: " + tag);
            }
            byte[] dataBytes = new byte[5];
            value = Integer.reverse(value);
            int dataLength = 0;
            for (int i = 1; i < dataBytes.length; ++i) {
                dataBytes[i] = (byte)(value >> (4 - i) * 8);
                if (dataBytes[i] == 0) continue;
                dataLength = i;
            }
            dataBytes[0] = IccUtils.countTrailingZeros(dataBytes[++dataLength - 1]);
            this.addChild(new Asn1Node(tag, dataBytes, 0, dataLength));
            return this;
        }

        public Builder addChildAsBoolean(int tag, boolean value) {
            if (Asn1Node.isConstructedTag(tag)) {
                throw new IllegalStateException("Cannot set value of a constructed tag: " + tag);
            }
            this.addChild(new Asn1Node(tag, value ? TRUE_BYTES : FALSE_BYTES, 0, 1));
            return this;
        }

        public Asn1Node build() {
            return new Asn1Node(this.mTag, this.mChildren);
        }
    }
}

