/*
 * Decompiled with CFR 0.152.
 */
package android.net.wifi.hotspot2;

import android.net.wifi.hotspot2.PasspointConfiguration;
import android.net.wifi.hotspot2.omadm.PpsMoParser;
import android.text.TextUtils;
import android.util.Base64;
import android.util.Log;
import android.util.Pair;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public final class ConfigParser {
    private static final String TAG = "ConfigParser";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String CONTENT_TRANSFER_ENCODING = "Content-Transfer-Encoding";
    private static final String TYPE_MULTIPART_MIXED = "multipart/mixed";
    private static final String TYPE_WIFI_CONFIG = "application/x-wifi-config";
    private static final String TYPE_PASSPOINT_PROFILE = "application/x-passpoint-profile";
    private static final String TYPE_CA_CERT = "application/x-x509-ca-cert";
    private static final String TYPE_PKCS12 = "application/x-pkcs12";
    private static final String ENCODING_BASE64 = "base64";
    private static final String BOUNDARY = "boundary=";

    public static PasspointConfiguration parsePasspointConfig(String mimeType, byte[] data) {
        if (!TextUtils.equals(mimeType, TYPE_WIFI_CONFIG)) {
            Log.e(TAG, "Unexpected MIME type: " + mimeType);
            return null;
        }
        try {
            byte[] decodedData = Base64.decode(new String(data, StandardCharsets.ISO_8859_1), 0);
            Map<String, byte[]> mimeParts = ConfigParser.parseMimeMultipartMessage(new LineNumberReader(new InputStreamReader((InputStream)new ByteArrayInputStream(decodedData), StandardCharsets.ISO_8859_1)));
            return ConfigParser.createPasspointConfig(mimeParts);
        }
        catch (IOException | IllegalArgumentException e) {
            Log.e(TAG, "Failed to parse installation file: " + e.getMessage());
            return null;
        }
    }

    private static PasspointConfiguration createPasspointConfig(Map<String, byte[]> mimeParts) throws IOException {
        byte[] pkcs12Data;
        byte[] profileData = mimeParts.get(TYPE_PASSPOINT_PROFILE);
        if (profileData == null) {
            throw new IOException("Missing Passpoint Profile");
        }
        PasspointConfiguration config = PpsMoParser.parseMoText(new String(profileData));
        if (config == null) {
            throw new IOException("Failed to parse Passpoint profile");
        }
        if (config.getCredential() == null) {
            throw new IOException("Passpoint profile missing credential");
        }
        byte[] caCertData = mimeParts.get(TYPE_CA_CERT);
        if (caCertData != null) {
            try {
                config.getCredential().setCaCertificate(ConfigParser.parseCACert(caCertData));
            }
            catch (CertificateException e) {
                throw new IOException("Failed to parse CA Certificate");
            }
        }
        if ((pkcs12Data = mimeParts.get(TYPE_PKCS12)) != null) {
            try {
                Pair<PrivateKey, List<X509Certificate>> clientKey = ConfigParser.parsePkcs12(pkcs12Data);
                config.getCredential().setClientPrivateKey((PrivateKey)clientKey.first);
                config.getCredential().setClientCertificateChain(((List)clientKey.second).toArray(new X509Certificate[((List)clientKey.second).size()]));
            }
            catch (IOException | GeneralSecurityException e) {
                throw new IOException("Failed to parse PCKS12 string");
            }
        }
        return config;
    }

    private static Map<String, byte[]> parseMimeMultipartMessage(LineNumberReader in) throws IOException {
        MimePart mimePart;
        String line;
        MimeHeader header = ConfigParser.parseHeaders(in);
        if (!TextUtils.equals(header.contentType, TYPE_MULTIPART_MIXED)) {
            throw new IOException("Invalid content type: " + header.contentType);
        }
        if (TextUtils.isEmpty(header.boundary)) {
            throw new IOException("Missing boundary string");
        }
        if (!TextUtils.equals(header.encodingType, ENCODING_BASE64)) {
            throw new IOException("Unexpected encoding: " + header.encodingType);
        }
        do {
            if ((line = in.readLine()) != null) continue;
            throw new IOException("Unexpected EOF before first boundary @ " + in.getLineNumber());
        } while (!line.equals("--" + header.boundary));
        HashMap<String, byte[]> mimeParts = new HashMap<String, byte[]>();
        boolean isLast = false;
        do {
            mimePart = ConfigParser.parseMimePart(in, header.boundary);
            mimeParts.put(mimePart.type, mimePart.data);
        } while (!(isLast = mimePart.isLast));
        return mimeParts;
    }

    private static MimePart parseMimePart(LineNumberReader in, String boundary) throws IOException {
        MimeHeader header = ConfigParser.parseHeaders(in);
        if (!TextUtils.equals(header.encodingType, ENCODING_BASE64)) {
            throw new IOException("Unexpected encoding type: " + header.encodingType);
        }
        if (!(TextUtils.equals(header.contentType, TYPE_PASSPOINT_PROFILE) || TextUtils.equals(header.contentType, TYPE_CA_CERT) || TextUtils.equals(header.contentType, TYPE_PKCS12))) {
            throw new IOException("Unexpected content type: " + header.contentType);
        }
        StringBuilder text = new StringBuilder();
        boolean isLast = false;
        String partBoundary = "--" + boundary;
        String endBoundary = partBoundary + "--";
        while (true) {
            String line;
            if ((line = in.readLine()) == null) {
                throw new IOException("Unexpected EOF file in body @ " + in.getLineNumber());
            }
            if (line.startsWith(partBoundary)) {
                if (!line.equals(endBoundary)) break;
                isLast = true;
                break;
            }
            text.append(line);
        }
        MimePart part = new MimePart();
        part.type = header.contentType;
        part.data = Base64.decode(text.toString(), 0);
        part.isLast = isLast;
        return part;
    }

    private static MimeHeader parseHeaders(LineNumberReader in) throws IOException {
        MimeHeader header = new MimeHeader();
        Map<String, String> headers = ConfigParser.readHeaders(in);
        block8: for (Map.Entry<String, String> entry : headers.entrySet()) {
            switch (entry.getKey()) {
                case "Content-Type": {
                    Pair<String, String> value = ConfigParser.parseContentType(entry.getValue());
                    header.contentType = (String)value.first;
                    header.boundary = (String)value.second;
                    continue block8;
                }
                case "Content-Transfer-Encoding": {
                    header.encodingType = entry.getValue();
                    continue block8;
                }
            }
            throw new IOException("Unexpected header: " + entry.getKey());
        }
        return header;
    }

    private static Pair<String, String> parseContentType(String contentType) throws IOException {
        String[] attributes = contentType.toString().split(";");
        String type = null;
        String boundary = null;
        if (attributes.length < 1 || attributes.length > 2) {
            throw new IOException("Invalid Content-Type: " + contentType);
        }
        type = attributes[0].trim();
        if (attributes.length == 2) {
            boundary = attributes[1].trim();
            if (!boundary.startsWith(BOUNDARY)) {
                throw new IOException("Invalid Content-Type: " + contentType);
            }
            if ((boundary = boundary.substring(BOUNDARY.length())).length() > 1 && boundary.startsWith("\"") && boundary.endsWith("\"")) {
                boundary = boundary.substring(1, boundary.length() - 1);
            }
        }
        return new Pair<String, String>(type, boundary);
    }

    private static Map<String, String> readHeaders(LineNumberReader in) throws IOException {
        HashMap<String, String> headers = new HashMap<String, String>();
        String name = null;
        StringBuilder value = null;
        while (true) {
            String line;
            if ((line = in.readLine()) == null) {
                throw new IOException("Missing line @ " + in.getLineNumber());
            }
            if (line.length() == 0 || line.trim().length() == 0) {
                if (name == null) break;
                headers.put(name, value.toString());
                break;
            }
            int nameEnd = line.indexOf(58);
            if (nameEnd < 0) {
                if (value != null) {
                    value.append(' ').append(line.trim());
                    continue;
                }
                throw new IOException("Bad header line: '" + line + "' @ " + in.getLineNumber());
            }
            if (Character.isWhitespace(line.charAt(0))) {
                throw new IOException("Illegal blank prefix in header line '" + line + "' @ " + in.getLineNumber());
            }
            if (name != null) {
                headers.put(name, value.toString());
            }
            name = line.substring(0, nameEnd).trim();
            value = new StringBuilder();
            value.append(line.substring(nameEnd + 1).trim());
        }
        return headers;
    }

    private static X509Certificate parseCACert(byte[] octets) throws CertificateException {
        CertificateFactory factory = CertificateFactory.getInstance("X.509");
        return (X509Certificate)factory.generateCertificate(new ByteArrayInputStream(octets));
    }

    private static Pair<PrivateKey, List<X509Certificate>> parsePkcs12(byte[] octets) throws GeneralSecurityException, IOException {
        KeyStore ks = KeyStore.getInstance("PKCS12");
        ByteArrayInputStream in = new ByteArrayInputStream(octets);
        ks.load(in, new char[0]);
        in.close();
        if (ks.size() != 1) {
            throw new IOException("Unexpected key size: " + ks.size());
        }
        String alias = ks.aliases().nextElement();
        if (alias == null) {
            throw new IOException("No alias found");
        }
        PrivateKey clientKey = (PrivateKey)ks.getKey(alias, null);
        ArrayList<X509Certificate> clientCertificateChain = null;
        Certificate[] chain = ks.getCertificateChain(alias);
        if (chain != null) {
            clientCertificateChain = new ArrayList<X509Certificate>();
            for (Certificate certificate : chain) {
                if (!(certificate instanceof X509Certificate)) {
                    throw new IOException("Unexpceted certificate type: " + certificate.getClass());
                }
                clientCertificateChain.add((X509Certificate)certificate);
            }
        }
        return new Pair<PrivateKey, List<X509Certificate>>(clientKey, clientCertificateChain);
    }

    private static class MimeHeader {
        public String contentType = null;
        public String boundary = null;
        public String encodingType = null;

        private MimeHeader() {
        }
    }

    private static class MimePart {
        public String type = null;
        public byte[] data = null;
        public boolean isLast = false;

        private MimePart() {
        }
    }
}

