/*
 * Decompiled with CFR 0.152.
 */
package org.robolectric.internal.dependency;

import com.google.auto.value.AutoValue;
import com.google.common.base.Strings;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import com.google.common.util.concurrent.AsyncCallable;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import javax.annotation.Nonnull;
import org.robolectric.internal.dependency.AutoValue_MavenArtifactFetcher_ValidationResult;
import org.robolectric.internal.dependency.MavenJarArtifact;
import org.robolectric.util.Logger;

public class MavenArtifactFetcher {
    private final String repositoryUrl;
    private final String repositoryUserName;
    private final String repositoryPassword;
    private final String proxyHost;
    private final int proxyPort;
    private final File localRepositoryDir;
    private final ExecutorService executorService;
    private File stagingRepositoryDir;

    public MavenArtifactFetcher(String repositoryUrl, String repositoryUserName, String repositoryPassword, String proxyHost, int proxyPort, File localRepositoryDir, ExecutorService executorService) {
        this.repositoryUrl = repositoryUrl;
        this.repositoryUserName = repositoryUserName;
        this.repositoryPassword = repositoryPassword;
        this.proxyHost = proxyHost;
        this.proxyPort = proxyPort;
        this.localRepositoryDir = localRepositoryDir;
        this.executorService = executorService;
    }

    public void fetchArtifact(MavenJarArtifact artifact) {
        if (new File(this.localRepositoryDir, artifact.jarPath()).exists()) {
            Logger.info((String)String.format("Found %s in local maven repository", artifact), (Object[])new Object[0]);
            return;
        }
        this.stagingRepositoryDir = Files.createTempDir();
        this.stagingRepositoryDir.deleteOnExit();
        try {
            this.createArtifactSubdirectory(artifact, this.stagingRepositoryDir);
            Futures.whenAllSucceed((ListenableFuture[])new ListenableFuture[]{this.fetchToStagingRepository(artifact.pomSha512Path()), this.fetchToStagingRepository(artifact.pomPath()), this.fetchToStagingRepository(artifact.jarSha512Path()), this.fetchToStagingRepository(artifact.jarPath())}).callAsync(() -> {
                if (new File(this.localRepositoryDir, artifact.jarPath()).exists()) {
                    this.removeArtifactFiles(this.stagingRepositoryDir, artifact);
                    return Futures.immediateFuture(null);
                }
                this.createArtifactSubdirectory(artifact, this.localRepositoryDir);
                ValidationResult pomResult = this.validateStagedFiles(artifact.pomPath(), artifact.pomSha512Path());
                if (!pomResult.isSuccess()) {
                    throw new AssertionError((Object)("SHA-512 mismatch for POM file for " + artifact + ", expected SHA-512=" + pomResult.expectedHashCode() + ", actual SHA-512=" + pomResult.calculatedHashCode()));
                }
                ValidationResult jarResult = this.validateStagedFiles(artifact.jarPath(), artifact.jarSha512Path());
                if (!jarResult.isSuccess()) {
                    throw new AssertionError((Object)("SHA-512 mismatch for POM file for " + artifact + ", expected SHA-512=" + jarResult.expectedHashCode() + ", actual SHA-512=" + jarResult.calculatedHashCode()));
                }
                Logger.info((String)String.format("Checksums validated, moving artifact %s to local maven directory", artifact), (Object[])new Object[0]);
                this.commitFromStaging(artifact.pomSha512Path());
                this.commitFromStaging(artifact.pomPath());
                this.commitFromStaging(artifact.jarSha512Path());
                this.commitFromStaging(artifact.jarPath());
                this.removeArtifactFiles(this.stagingRepositoryDir, artifact);
                return Futures.immediateFuture(null);
            }, (Executor)this.executorService).get();
        }
        catch (IOException | InterruptedException | ExecutionException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            this.removeArtifactFiles(this.stagingRepositoryDir, artifact);
            this.removeArtifactFiles(this.localRepositoryDir, artifact);
            Logger.error((String)("Failed to fetch maven artifact " + artifact), (Throwable)e);
            throw new AssertionError("Failed to fetch maven artifact " + artifact, e);
        }
    }

    private void removeArtifactFiles(File repositoryDir, MavenJarArtifact artifact) {
        new File(repositoryDir, artifact.jarPath()).delete();
        new File(repositoryDir, artifact.jarSha512Path()).delete();
        new File(repositoryDir, artifact.pomPath()).delete();
        new File(repositoryDir, artifact.pomSha512Path()).delete();
    }

    private ValidationResult validateStagedFiles(String filePath, String sha512Path) throws IOException {
        File tempFile = new File(this.stagingRepositoryDir, filePath);
        File sha512File = new File(this.stagingRepositoryDir, sha512Path);
        HashCode expected = HashCode.fromString((String)new String(Files.asByteSource((File)sha512File).read(), StandardCharsets.UTF_8));
        HashCode actual = Files.asByteSource((File)tempFile).hash(Hashing.sha512());
        return ValidationResult.create(expected.equals((Object)actual), expected.toString(), actual.toString());
    }

    private void createArtifactSubdirectory(MavenJarArtifact artifact, File repositoryDir) throws IOException {
        File jarPath = new File(repositoryDir, artifact.jarPath());
        Files.createParentDirs((File)jarPath);
    }

    private URL getRemoteUrl(String path) {
        String url = this.repositoryUrl;
        if (!url.endsWith("/")) {
            url = url + "/";
        }
        try {
            return new URI(url + path).toURL();
        }
        catch (MalformedURLException | URISyntaxException e) {
            throw new AssertionError((Object)e);
        }
    }

    private ListenableFuture<Void> fetchToStagingRepository(String path) {
        URL remoteUrl = this.getRemoteUrl(path);
        File destination = new File(this.stagingRepositoryDir, path);
        return this.createFetchToFileTask(remoteUrl, destination);
    }

    protected ListenableFuture<Void> createFetchToFileTask(URL remoteUrl, File tempFile) {
        return Futures.submitAsync((AsyncCallable)new FetchToFileTask(remoteUrl, tempFile, this.repositoryUserName, this.repositoryPassword, this.proxyHost, this.proxyPort), (Executor)this.executorService);
    }

    private void commitFromStaging(String path) throws IOException {
        File source = new File(this.stagingRepositoryDir, path);
        File destination = new File(this.localRepositoryDir, path);
        Files.move((File)source, (File)destination);
    }

    static class FetchToFileTask
    implements AsyncCallable<Void> {
        private final URL remoteURL;
        private final File localFile;
        private String repositoryUserName;
        private String repositoryPassword;
        private String proxyHost;
        private int proxyPort;

        public FetchToFileTask(URL remoteURL, File localFile, String repositoryUserName, String repositoryPassword, String proxyHost, int proxyPort) {
            this.remoteURL = remoteURL;
            this.localFile = localFile;
            this.repositoryUserName = repositoryUserName;
            this.repositoryPassword = repositoryPassword;
            this.proxyHost = proxyHost;
            this.proxyPort = proxyPort;
        }

        public ListenableFuture<Void> call() throws Exception {
            URLConnection connection;
            if (this.proxyHost != null && !this.proxyHost.isEmpty() && this.proxyPort > 0) {
                Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(this.proxyHost, this.proxyPort));
                connection = this.remoteURL.openConnection(proxy);
            } else {
                connection = this.remoteURL.openConnection();
            }
            if (!Strings.isNullOrEmpty((String)this.repositoryUserName)) {
                String encoded = Base64.getEncoder().encodeToString((this.repositoryUserName + ":" + this.repositoryPassword).getBytes(StandardCharsets.UTF_8));
                connection.setRequestProperty("Authorization", "Basic " + encoded);
            }
            Logger.info((String)("Transferring " + this.remoteURL), (Object[])new Object[0]);
            try (InputStream inputStream = connection.getInputStream();
                 FileOutputStream outputStream = new FileOutputStream(this.localFile);){
                ByteStreams.copy((InputStream)inputStream, (OutputStream)outputStream);
                outputStream.flush();
                outputStream.getFD().sync();
            }
            return Futures.immediateFuture(null);
        }
    }

    @AutoValue
    static abstract class ValidationResult {
        ValidationResult() {
        }

        abstract boolean isSuccess();

        @Nonnull
        abstract String expectedHashCode();

        @Nonnull
        abstract String calculatedHashCode();

        static ValidationResult create(boolean isSuccess, String expectedHashCode, String calculatedHashCode) {
            return new AutoValue_MavenArtifactFetcher_ValidationResult(isSuccess, expectedHashCode, calculatedHashCode);
        }
    }
}

