package org.sapia.util.license;

import java.io.IOException;
import java.io.Serializable;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;

/**
 * An instance of this class holds a <code>License</code> object, a digital signature (kept as a byte array), and
 * a public key. The digital signature was created upong signing the license's serialized bytes.
 * <p>
 * An instance of this class is normally given to a licensee (the beholder of a license, in legal terms). The public
 * key held within an instance of this class is used on the licensee's side to ensure that the license is properly signed
 * (more concretely: to ensure that a given license cannot be generated by another party than the licensing one).
 * <p>
 * A <code>License</code> record should be kept at the licensee (it is typically serialized in a file). Application code can then
 * use the license object to ensure that the licensee has the proper rights (concretely: that in can use the application). 
 * 
 * @author Yanick Duchesne
 *
 * <dl>
 * <dt><b>Copyright:</b><dd>Copyright &#169; 2002-2004 <a href="http://www.sapia-oss.org">Sapia Open Source Software</a>. All Rights Reserved.</dd></dt>
 * <dt><b>License:</b><dd>Read the license.txt file of the jar or visit the
 *        <a href="http://www.sapia-oss.org/license.html">license page</a> at the Sapia OSS web site</dd></dt>
 * </dl>
 */
public class LicenseRecord implements Serializable{
  
  static final long serialVersionUID = 1L;
  
  private RecData _data;
  private Object _vendorId;
  private transient boolean _init, _valid;
  private transient PublicKey _key;  
  private boolean _activated;
  private long _id;
    
  LicenseRecord(Object vendorId, License license, byte[] sig, byte[] publicKey, long id){
    _vendorId = vendorId;
    _data = new RecData(sig, publicKey, license);
    _id = id;
  }
  
  /**
   * This method internally checks that the digital signature of the license
   * held by this instance corresponds to the internally kept public key, and
   * that the <code>isValid()</code> method of the encapsulated <code>License</code>
   * returns <code>true</code>.
   * <p>
   * If the digital signature does not match, or if the <code>License</code> is not
   * valid, the this method returns <code>false</code>.

   * @param context an arbitratry context <code>Object</code> that is 
   * eventually passed to the encapsulated <code>License</code> object.
   * @param fac a <code>SecurityFactory</code>. 
   * @return <code>true</code> if the license held by this instance is valid.
   * @throws IOException
   * @throws Exception
 
   */
  public boolean isValid(Object context, SecurityFactory fac) throws IOException, Exception{
    if(!_init){
      Signature algo = fac.newSignature("SHA/DSA");
      _key = fac.generatePublic(_data._key);
      algo.initVerify(_key);
      byte[] toVerify = _data._license.getBytes();
      algo.update(toVerify);
      _valid = algo.verify(_data._sig);
      _init = true;      
    }
    if(!_valid){
      return false;
    }
    return _valid = _data._license.isValid(context);
    
  }
  
  /**
   * Activates the license that this instance encapsulates. This
   * method should only be called once in this instance's lifecycle.
   * 
   * @param context an arbitrary <code>Context</code> object.
   */
  public void activate(Object context){
    if(!_activated){
      _data._license.activate(context);
      _activated = true;
    }
  }
  
  /**
   * Updates this instance with the <code>License</code> that the 
   * passed in record contains.
   * <p>
   * This method can be used on the client side to insure that a license update
   * is valid (a given license cannot be reused twice).
   * <p>
   * This method internally activates the license with which it is updated.
   * 
   * @param newRecord a <code>LicenseRecord</code>. 
   * @param context an arbitrary context <code>Object</code>.    
   * @return <code>true</code> if the new data is accepted.
   * @see License#activate(Object)
   */
  public synchronized boolean update(LicenseRecord newRecord, Object context){
    if(_id >= newRecord._id){
      return false;
    }
    if(newRecord._vendorId == null || !newRecord._vendorId.equals(_vendorId)){
      return false;
    }
    _data = newRecord._data;
    _id = newRecord._id;
    _init = false;
    _data._license.activate(context);
    return true;
  }
  
  /**
   * This method creates an instance of this class, given the provided license and key pair.
   * Internally, this method signs the bytes of the serialized license. The digital signature, as
   * well as the public key present in the key pair, is kept within the returned <code>LicenseRecord</code>.
   *  
   * @param license a <code>License</code>. 
   * @param pub the <code>PublicKey</code> corresponding to this instance.
   * @param priv the <code>PrivateKey</code> corresponding to this instance.
   * @param factory the <code>SecurityFactory</code> to use when internally creating
   * @param idFactory the <code>LicenseIdFactory</code> to use to internally create license identifiers.
   * @return a <code>LicenseRecord</code>
   * @throws IOException if an IO problem occurs.
   * @throws Exception if a low-level problem occurs.
   */
  public static LicenseRecord newInstance(Object vendorId, 
                                          License license, 
                                          PublicKey pub, 
                                          PrivateKey priv, 
                                          SecurityFactory factory, 
                                          LicenseIdFactory idFactory) throws IOException, Exception{
    Signature algo = factory.newSignature("SHA/DSA");
    algo.initSign(priv);
    byte[] toSign = license.getBytes();
    algo.update(toSign);
    byte[] sig = algo.sign();
    return new LicenseRecord(vendorId, license, sig, pub.getEncoded(), idFactory.newInstance());
  }
  
  synchronized void registerNewLicense(Object vendorId, 
                                       License license, 
                                       PublicKey pub, 
                                       PrivateKey priv, 
                                       SecurityFactory factory,
                                       LicenseIdFactory idFactory) throws IOException, Exception{
    LicenseRecord rec = LicenseRecord.newInstance(vendorId, license, pub, priv, factory, idFactory);
    if(_key == null){
      _key = factory.generatePublic(_data._key);
    }
    if(!_key.equals(pub)){
      throw new IllegalArgumentException("A license must be updated with existing public/private keys");
    }
    _data = rec._data;
    _id = rec._id;
    _init = false;
  }
  
  static final class RecData implements Serializable{
    
    static final long serialVersionUID = 1L;
    
    private byte[] _sig;
    private byte[] _key;
    private License _license;
    
    public RecData(byte[] sig, byte[] key, License license){
      _sig = sig;
      _key = key;
      _license = license;
    }

  }
}

