package org.sapia.util.xml.parser;


// Imports of David Meggison's SAX classes
// ---------------------------------------
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;


/**
 * The <CODE>HandlerContextIF</CODE> interface represents the context in which the
 * handlers of SAX events are stacked. It provides methods set the current and remove
 * a <CODE>HandlerStateIF</CODE> object. This context behaves like a stack where each
 * new handler added to this context becomes the "current" one. When you remove that
 * current state, the previous state now becomes the "current" one.<p>
 *
 * This context also provides a way to gather results generated by the different
 * handler that it contains. For more detail see the methods <CODE>setResultFor()</CODE>
 * and <CODE>getResultFor()</CODE>.
 *
 * @see HandlerStateIF
 * @see StatefullSAXHandler
 * @author Jean-Cedric Desrochers
 *
 * <dl>
 * <dt><b>Copyright:</b><dd>Copyright &#169; 2002-2003 <a href="http://www.sapia-oss.org">Sapia Open Source Software</a>. All Rights Reserved.</dd></dt>
 * <dt><b>License:</b><dd>Read the license.txt file of the jar or visit the
 *        <a href="http://www.sapia-oss.org/license.html">license page</a> at the Sapia OSS web site</dd></dt>
 * </dl>
 */
public interface HandlerContextIF {
  /**
   * Returns the current handler state of this context.
   *
   * @return The current handler state of this context.
   */
  public HandlerStateIF getCurrentState();

  /**
   * Changes the current handler state to the handler state passed in. The collection
   * of state objects needs to behave like a stack (LIFO - Last in, first out) in
   * order to go back to the previous state of a handler.<P>
   *
   * The handler context will notify the child handler state of the start of the element
   * by calling it's <CODE>startElement()</CODE> method passing the arguments received.
   *
   * @param aHandlerState The new current handler state.
   * @param anUri The namespace URI of the element that starts.
   * @param aLocalName The local name of the element that starts.
   * @param aQualifiedName The qualified name of the element that starts.
   * @param someAttributes The attributes of the element that starts.
   * @exception SAXException If an error occurs while setting the new handler state.
   */
  public void setCurrentState(HandlerStateIF aHandlerState, String anUri,
    String aLocalName, String aQualifiedName, Attributes someAttributes)
    throws SAXException;

  /**
   * Removes the current handler state from this handler context. The previous handler
   * state becomes the current state.<P>
   *
   * The handler context will notify the parent handler state of the end of the
   * element by calling it's <CODE>endElement()</CODE> method passing the arguments
   * received.
   *
   * @param anURI The namespace URI of the element that closes.
   * @param aLocalName The local name of the element that closes.
   * @param aQualifiedName The qualified name of the element that closes.
   * @exception SAXException If an error occurs while removing the new handler state.
   */
  public void removeCurrentState(String anUri, String aLocalName,
    String aQualifiedName) throws SAXException;

  /**
   * Sets the namespace URI passed in as the current mapping for the given prefix.
   *
   * @param aPrefix The Namespace prefix being declared.
   * @param anUri The Namespace URI mapped to the prefix.
   */
  public void startPrefixMapping(String aPrefix, String anUri);

  /**
   * Removes the current mapping of the prefix passed in.
   *
   * @param aPrefix The Namespace prefix being declared.
   */
  public void endPrefixMapping(String aPrefix);

  /**
   * Returns the current namespace URI for the passed in prefix.
   *
   * @return The current namespace URI or <CODE>null</CODE> if not mapping exists.
   */
  public String getNamespaceURIFor(String aPrefix);

  /**
   * associates the result object to the handler state passed in.
   *
   * @param aHandlerState The handler state for which a result is provided.
   * @param aResult The result object of the handler state.
   */
  public void setResultFor(HandlerStateIF aHandlerState, Object aResult);

  /**
   * Returns the result for the handler state passed in.
   *
   * @param aHandlerState The handler state for which to search for a result.
   * @return The result object found, or <CODE>null</CODE> if no handler state was foud
   *         or if no result was associated for the handler state passed in.
   */
  public Object getResultFor(HandlerStateIF aHandlerState);
}
