/*
 * Zinc - The incremental compiler for Scala.
 * Copyright Scala Center, Lightbend, and Mark Harrah
 *
 * Licensed under Apache License 2.0
 * SPDX-License-Identifier: Apache-2.0
 *
 * See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.
 */

package xsbti.compile.analysis;

import sbt.internal.inc.mappers.NaiveRelativeReadMapper;
import sbt.internal.inc.mappers.RelativeReadMapper;
import xsbti.VirtualFileRef;
import xsbti.compile.MiniSetup;

import java.nio.file.Path;

/**
 * Defines a reader-only mapper interface that is used by Zinc after reading
 * the analysis from an instance of an xsbti.compile.AnalysisStore.
 *
 * This interface is useful to make the analysis file machine-independent and
 * allow third parties to distribute them around.
 */
public interface ReadMapper extends GenericMapper {

    /**
     * Defines a mapper that reads from a machine-independent analysis file.
     *
     * An analysis file is machine independent if all the paths are relative to a path
     * and no information about the machine that produced it is stored -- only
     * information about the structure of the project from a given build tool is persisted.
     *
     * The mapper makes sure that the analysis file looks like if it was generated by
     * the local machine it's executed on.
     *
     * <b>Important note</b>: The assumption that all paths can be made relative to a concrete
     * position is not always correct. There is no guarantee that the paths of the caches, the
     * artifacts, the classpath entries, the compilation inputs and outputs, et cetera share
     * the same prefix. If this is not the case this reader will fail at runtime.
     *
     * Such assumption is broken in our test infrastructure: `lib_managed` does not share
     * the same prefix, and without redefining it, it fails. Note that this is a conscious
     * design decision of the relative read and write mappers. They are focused on simplicity.
     * Build tools that need more careful handling of paths should create their own read and
     * write mappers.
     *
     *
     * @param projectRootPath The path on which we want to "mount" all the relative paths in analysis.
     * @return A read mapper to pass in to {@link sbt.internal.inc.FileAnalysisStore}.
     */
    static ReadMapper getMachineIndependentMapper(Path projectRootPath) {
        return new NaiveRelativeReadMapper(projectRootPath);
    }

    /**
     * Defines a mapper that reads from a machine-independent analysis file.
     *
     * An analysis file is machine independent if all the paths are relative to a path
     * and no information about the machine that produced it is stored -- only
     * information about the structure of the project from a given build tool is persisted.
     *
     * The mapper makes sure that the analysis file looks like if it was generated by
     * the local machine it's executed on.
     *
     * This mapper will use the paths provided by {@link RootPaths} to make the file relative,
     * allowing build tools to pass independent common root paths for different kind of files.
     *
     * In case users of this machine-independent mapper need further customisation, they
     * can use the decorator pattern or roll their own.
     *
     * @param rootPaths The root paths to use internally by the mapper.
     * @return A machine-independent read mapper to pass in to {@link sbt.internal.inc.FileAnalysisStore}.
     */
    static ReadMapper getMachineIndependentMapper(RootPaths rootPaths) {
        return new RelativeReadMapper(rootPaths);
    }

    /**
     * Defines an no-op read mapper.
     *
     * This is useful when users are not interested in distributing the analysis files
     * and need to pass a read mapper to {@link sbt.internal.inc.FileAnalysisStore}.
     *
     * @return A no-op read mapper.
     */
    static ReadMapper getEmptyMapper() {
        return new ReadMapper() {
            @Override
            public VirtualFileRef mapSourceFile(VirtualFileRef sourceFile) {
                return sourceFile;
            }

            @Override
            public VirtualFileRef mapBinaryFile(VirtualFileRef binaryFile) {
                return binaryFile;
            }

            @Override
            public VirtualFileRef mapProductFile(VirtualFileRef productFile) {
                return productFile;
            }

            @Override
            public Path mapOutputDir(Path outputDir) {
                return outputDir;
            }

            @Override
            public Path mapSourceDir(Path sourceDir) {
                return sourceDir;
            }

            @Override
            public Path mapClasspathEntry(Path classpathEntry) {
                return classpathEntry;
            }

            @Override
            public String mapJavacOption(String javacOption) {
                return javacOption;
            }

            @Override
            public String mapScalacOption(String scalacOption) {
                return scalacOption;
            }

            @Override
            public Stamp mapBinaryStamp(VirtualFileRef file, Stamp binaryStamp) {
                return binaryStamp;
            }

            @Override
            public Stamp mapSourceStamp(VirtualFileRef file, Stamp sourceStamp) {
                return sourceStamp;
            }

            @Override
            public Stamp mapProductStamp(VirtualFileRef file, Stamp productStamp) {
                return productStamp;
            }

            @Override
            public MiniSetup mapMiniSetup(MiniSetup miniSetup) {
                return miniSetup;
            }
        };
    }
}
