/*
 * Copyright (c) 2013, Scodec
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors
 *    may be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package scodec

/** Provides immutable data types for working with bits and bytes.
  *
  * @see
  *   [[BitVector]] and [[ByteVector]]
  */
package object bits extends ScalaVersionSpecific {

  /** Provides the `bin` string interpolator, which returns `BitVector` instances from binary
    * strings.
    */
  final implicit class BinStringSyntax(val sc: StringContext) extends AnyVal {

    /** Converts this binary literal string to a `BitVector`. Whitespace characters are ignored.
      *
      * Named arguments are supported in the same manner as the standard `s` interpolator but they
      * must be of type `BitVector`.
      */
    def bin(args: BitVector*): BitVector = macro LiteralSyntaxMacros.binStringInterpolator
  }

  /** Provides the `hex` string interpolator, which returns `ByteVector` instances from hexadecimal
    * strings.
    */
  final implicit class HexStringSyntax(val sc: StringContext) extends AnyVal {

    /** Converts this hexadecimal literal string to a `ByteVector`. Whitespace characters are
      * ignored.
      *
      * Named arguments are supported in the same manner as the standard `s` interpolator but they
      * must be of type `ByteVector`.
      */
    def hex(args: ByteVector*): ByteVector = macro LiteralSyntaxMacros.hexStringInterpolator
  }

  /** Provides the `ascii` string interpolator, which returns `ByteVector` instances from ascii
    * strings.
    */
  final implicit class AsciiStringSyntax(val sc: StringContext) extends AnyVal {

    /** Converts this ascii literal string to a `ByteVector`.
      *
      * Named arguments are supported in the same manner as the standard `s` interpolator.
      */
    def asciiBytes(args: String*): ByteVector = macro LiteralSyntaxMacros.asciiStringInterpolator
  }

  /** Provides the `utf8` string interpolator, which returns `ByteVector` instances from UTF8
    * strings.
    */
  final implicit class Utf8StringSyntax(val sc: StringContext) extends AnyVal {

    /** Converts this UTF8 literal string to a `ByteVector`.
      *
      * Named arguments are supported in the same manner as the standard `s` interpolator.
      */
    def utf8Bytes(args: String*): ByteVector = macro LiteralSyntaxMacros.utf8StringInterpolator
  }

  private[bits] implicit class EitherOps[L, R](val self: Either[L, R]) extends AnyVal {
    def map[R2](f: R => R2): Either[L, R2] =
      self match {
        case Right(r) => Right(f(r))
        case l        => l.asInstanceOf[Either[L, R2]]
      }

    def flatMap[R2](f: R => Either[L, R2]): Either[L, R2] =
      self match {
        case Right(r) => f(r)
        case l        => l.asInstanceOf[Either[L, R2]]
      }

    def toOption: Option[R] =
      self match {
        case Right(r) => Some(r)
        case Left(_)  => None
      }
  }
}
