/*
 * Copyright 2009-10 www.scribble.org
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.scribble.protocol.export.text;

import java.text.MessageFormat;

import org.scribble.common.logging.Journal;
import org.scribble.protocol.export.ProtocolExporter;
import org.scribble.protocol.model.*;

public class TextProtocolExporter implements ProtocolExporter {

	public static final String TEXT_ID = "txt";

	/**
	 * This method returns the id of the exporter.
	 * 
	 * @return The exporter id
	 */
	public String getId() {
		return(TEXT_ID);
	}
	
	/**
	 * This method returns the name of the exporter for use in
	 * user based selectors.
	 * 
	 * @return The name of the exporter
	 */
	public String getName() {
		return("Text");
	}
	
	/**
	 * This method exports the supplied protocol model, in the implementation
	 * specific format, to the specified output stream. If any issues occur
	 * during the export process, they will be reported to the journal.
	 * 
	 * @param model The protocol model to be exported
	 * @param journal The journal
	 * @param os The output stream
	 */
	public void export(ProtocolModel model, Journal journal, java.io.OutputStream os) {
		TextExportVisitor visitor=new TextExportVisitor(os);
		
		model.visit(visitor);
		
		if (visitor.getException() != null) {
			journal.error(MessageFormat.format(
					java.util.PropertyResourceBundle.getBundle(
							"org.scribble.protocol.Messages").getString("_EXPORT_FAILED"),
								visitor.getException().getLocalizedMessage()), null);
		}
	}
	
	public static class TextExportVisitor extends DefaultVisitor {
		
		public TextExportVisitor(java.io.OutputStream os) {
			m_outputStream = os;
		}
		
		/**
		 * This method indicates the start of a
		 * block.
		 * 
		 * @param elem The block
		 * @return Whether to process the contents
		 */
		public boolean start(Block elem) {
			
			if (elem.getParent() instanceof When) {
				// Skip block
			} else {
				// Only add the 'and' keyword if the parent is a Parallel construct
				// and the block being processed is not the first block
				if (elem.getParent() instanceof Parallel &&
						((Parallel)elem.getParent()).getBlocks().indexOf(elem) > 0) {
					output(" and");
				}
				
				output(" {\r\n");
				
				m_indent++;
			}
			
			return(true);
		}
				
		/**
		 * This method indicates the end of a
		 * block.
		 * 
		 * @param elem The block
		 */
		public void end(Block elem) {
			
			if (elem.getParent() instanceof When) {
				// Skip block
			} else {
				m_indent--;
				
				indent();
				
				output("}");
				
				// Place newline after close bracket, unless the parent element is a
				// multipath construct AND the block is not the final block
				if (isEndOfBlock(elem)) {
					output("\r\n");
				}
			}
		}
		
		protected boolean isEndOfBlock(Block elem) {
			boolean ret=true;

			if (elem.getParent() instanceof Parallel) {
				ret = ((Parallel)elem.getParent()).getBlocks().indexOf(elem) ==
						((Parallel)elem.getParent()).getBlocks().size()-1;
			} else if (elem.getParent() instanceof Catch) {
				Catch c=(Catch)elem.getParent();
				
				if (c.getParent() instanceof Try) {
					Try te=(Try)c.getParent();
					
					ret = te.getCatches().indexOf(c) == te.getCatches().size()-1;
				}
			} else if (elem.getParent() instanceof Try) {
				ret = ((Try)elem.getParent()).getCatches().size() == 0;
			}
					
			return(ret);
		}
		/**
		 * This method visits an import component.
		 * 
		 * @param elem The import
		 */
		public void accept(ImportList elem) {
			output("import ");
			
			if (elem.getTypeImports().size() > 0) {
				boolean f_first=true;
				
				if (elem.getFormat() != null) {
					output(elem.getFormat()+" ");
				}
				
				for (TypeImport t : elem.getTypeImports()) {
					if (!f_first) {
						output(", ");
					}
					
					f_first = false;
					
					if (t.getDataType() != null && t.getDataType().getDetails() != null) {
						output("\""+t.getDataType().getDetails()+"\" as ");
					}
					
					output(t.getName());
				}
				
				if (elem.getLocation() != null) {
					output("from \""+elem.getLocation()+"\"");
				}
			} else {
				output("protocol ");
				
				boolean f_first=true;
				for (ProtocolImport t : elem.getProtocolImports()) {
					if (!f_first) {
						output(", ");
					}
					
					f_first = false;
					output(t.getName());
				}
			}
			
			output(";\r\n");
		}
		
		/**
		 * This method visits the role list.
		 * 
		 * @param elem The role list
		 */
		public void accept(RoleList elem) {
			indent();
			
			output("role ");
			
			for (int i=0; i < elem.getRoles().size(); i++) {
				if (i > 0) {
					output(", ");
				}
				output(elem.getRoles().get(i).getName());
			}
			
			output(";\r\n");
		}
		
		/**
		 * This method visits an interaction component.
		 * 
		 * @param elem The interaction
		 */
		public void accept(Interaction elem) {
			
			if ((elem.getParent() instanceof Catch) == false) {
				indent();
				
				outputInteraction(elem);
				
				output(";\r\n");
			}
		}
		
		protected void outputInteraction(Interaction elem) {
			outputMessageSignature(elem.getMessageSignature());
			
			if (elem.getFromRole() != null) {
				output(" from "+elem.getFromRole().getName());
			}
			
			if (elem.getToRoles().size() > 0) {
				output(" to ");
				
				for (int i=0; i < elem.getToRoles().size(); i++) {
					if (i > 0) {
						output(",");
					}
					output(elem.getToRoles().get(i).getName());
				}
			}

		}
		
		/**
		 * This method visits a recursion component.
		 * 
		 * @param elem The recursion
		 */
		public void accept(Recursion elem) {
			indent();

			if (elem.getLabel() != null) {
				output(elem.getLabel());
			}
			
			output(";\r\n");
		}
		
		/**
		 * This method indicates the start of a
		 * protocol.
		 * 
		 * @param elem The protocol
		 * @return Whether to process the contents
		 */
		public boolean start(Protocol elem) {
			indent();
			
			output("protocol "+elem.getName());
			
			if (elem.getRole() != null) {
				output(" @ "+elem.getRole().getName());
			}
			
			return(true);
		}
		
		/**
		 * This method indicates the start of a
		 * choice.
		 * 
		 * @param elem The choice
		 * @return Whether to process the contents
		 */
		public boolean start(Choice elem) {
			
			indent();
			
			output("choice");
			
			if (elem.getFromRole() != null) {
				output(" from "+elem.getFromRole().getName());
			}
			
			if (elem.getToRole() != null) {
				output(" to "+elem.getToRole().getName());
			}
			
			output(" {\r\n");
			
			m_indent++;
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * choice.
		 * 
		 * @param elem The choice
		 */
		public void end(Choice elem) {
			
			m_indent--;
			
			indent();
			
			output("}\r\n");
		}
		
		/**
		 * This method indicates the start of a
		 * when block.
		 * 
		 * @param elem The when block
		 * @return Whether to process the contents
		 */
		public boolean start(When elem) {
			
			indent();
			
			outputMessageSignature(elem.getMessageSignature());
			
			output(":\r\n");
			
			m_indent++;
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * block.
		 * 
		 * @param elem The when block
		 */
		public void end(When elem) {
			
			output("\r\n");
			
			m_indent--;
		}
		
		/**
		 * This method processes a when clause.
		 * 
		 * @param elem The when
		 */
		public boolean start(Catch elem) {
			
			indent();
			
			output(" catch (");
			
			for (int i=0; i < elem.getInteractions().size(); i++) {
				if (i > 0) {
					output(" | ");
				}
			
				outputInteraction(elem.getInteractions().get(i));
			}

			output(")");
			
			return(true);
		}
		
		private void outputMessageSignature(MessageSignature ms) {
			if (ms != null) {
				if (ms.getOperation() != null) {
					output(ms.getOperation()+"(");
					
					for (int i=0; i < ms.getTypeReferences().size(); i++) {
						if (i > 0) {
							output(", ");
						}	
						output(ms.getTypeReferences().get(i).getName());
					}
					
					output(")");
				} else if (ms.getTypeReferences().size() > 0) {
					output(ms.getTypeReferences().get(0).getName());					
				}
			}			
		}
		
		/**
		 * This method indicates the start of a
		 * run.
		 * 
		 * @param elem The run
		 * @return Whether to process the contents
		 */
		public boolean start(Run elem) {
			indent();
			
			output("run "+elem.getProtocolReference().getName());
			
			if (elem.getProtocolReference().getRole() != null) {
				output("@"+elem.getProtocolReference().getRole().getName());
			}
			
			if (elem.getParameters().size() > 0) {
				output("(");
				
				for (int i=0; i < elem.getParameters().size(); i++) {
					if (i > 0) {
						output(", ");
					}
					
					output(elem.getParameters().get(i).getBoundName()+":="+
							elem.getParameters().get(i).getName());
				}
				
				output(")");
			}
			
			output(";\r\n");
			
			return(true);
		}
		
		/**
		 * This method indicates the start of a
		 * parallel.
		 * 
		 * @param elem The parallel
		 * @return Whether to process the contents
		 */
		public boolean start(Parallel elem) {
			
			indent();
			
			output("par");
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * parallel.
		 * 
		 * @param elem The parallel
		 */
		public void end(Parallel elem) {
		}
		
		/**
		 * This method indicates the start of a
		 * try/escape.
		 * 
		 * @param elem The try escape
		 * @return Whether to process the contents
		 */
		public boolean start(Try elem) {
			
			indent();
			
			output("try");
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * try/escape.
		 * 
		 * @param elem The try escape
		 */
		public void end(Try elem) {
		}
		
		/**
		 * This method indicates the start of a
		 * repeat.
		 * 
		 * @param elem The repeat
		 * @return Whether to process the contents
		 */
		public boolean start(Repeat elem) {
			
			indent();
			
			output("repeat");
			
			if (elem.getRoles().size() > 0) {
				output(" @ ");
				
				for (int i=0; i < elem.getRoles().size(); i++) {
					if (i > 0) {
						output(",");
					}
					
					output(elem.getRoles().get(i).getName());
				}
			}
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * repeat.
		 * 
		 * @param elem The repeat
		 */
		public void end(Repeat elem) {
		}
		
		/**
		 * This method indicates the start of a
		 * unordered block.
		 * 
		 * @param elem The unordered block
		 * @return Whether to process the contents
		 */
		public boolean start(Unordered elem) {
			
			indent();
			
			output("unordered");
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * unordered block.
		 * 
		 * @param elem The unordered block
		 */
		public void end(Unordered elem) {
		}

		/**
		 * This method indicates the start of a
		 * labelled block.
		 * 
		 * @param elem The labelled block
		 * @return Whether to process the contents
		 */
		public boolean start(RecBlock elem) {
			
			indent();
			
			if (elem.getLabel() != null) {
				output("rec "+elem.getLabel());
			}
			
			return(true);
		}
		
		/**
		 * This method indicates the end of a
		 * labelled block.
		 * 
		 * @param elem The labelled block
		 */
		public void end(RecBlock elem) {
		}

		/**
		 * This method indicates the start of an
		 * Optional construct.
		 * 
		 * @param elem The Optional construct
		 * @return Whether to process the contents
		 */
		public boolean start(Optional elem) {
			
			indent();
			
			output("optional");
			
			if (elem.getRoles().size() > 0) {
				output(" @ ");
				
				for (int i=0; i < elem.getRoles().size(); i++) {
					if (i > 0) {
						output(",");
					}
					
					output(elem.getRoles().get(i).getName());
				}
			}
			
			return(true);
		}
		
		/**
		 * This method indicates the end of an
		 * Optional construct.
		 * 
		 * @param elem The Optional construct
		 */
		public void end(Optional elem) {
		}
		
		private void indent() {
			for (int i=0; i < m_indent; i++) {
				output("\t");
			}
		}
		
		private void output(String str) {
			try {
				m_outputStream.write(str.getBytes());
			} catch(Exception e) {
				m_exception = e;
			}
		}
		
		public Exception getException() {
			return(m_exception);
		}
		
		private java.io.OutputStream m_outputStream=null;
		private int m_indent=0;
		private Exception m_exception=null;
	}
}
