/*
 * Copyright 2009-10 www.scribble.org
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.scribble.protocol.model;

/**
 * This class represents the Recur construct.
 * 
 */
public class RecBlock extends Behaviour implements SinglePathBehaviour {

	private static final long serialVersionUID = 4349956896459473892L;

	/**
	 * This is the default constructor.
	 * 
	 */
	public RecBlock() {
		m_block.setParent(this);
	}
	
	/**
	 * This method returns the label associated with the labelled block construct.
	 * 
	 * @return The label
	 */
	public String getLabel() {
		return(m_label);
	}
	
	/**
	 * This method sets the label associated with the labelled block construct.
	 * 
	 * @param label The label
	 */
	public void setLabel(String label) {
		m_label = label;
	}
		
	/**
	 * This method returns whether the behaviour is a wait
	 * state.
	 * 
	 * @return Whether the behaviour is a wait state
	 */
	@Override
	public boolean isWaitState() {
		boolean ret=false;
		
		return(ret);
	}
	
	/**
	 * This method returns the activities.
	 * 
	 * @return The block of activities
	 */
	public Block getBlock() {
		return(m_block);
	}
	
	/**
	 * This method sets the block.
	 * 
	 * @param block The block
	 */
	public void setBlock(Block block) {
		if (m_block != null) {
			m_block.setParent(null);
		}
		
		m_block = block;
		
		if (m_block != null) {
			m_block.setParent(this);
		}
	}

	/**
	 * This method indicates whether the construct requires a
	 * strict scope to be maintained. If the scope does not
	 * need to be strictly maintained, then it is possible that
	 * activities could be contained by the construct or
	 * following the construct.
	 * 
	 * @return Whether a strict scope should be maintained
	 */
	public boolean isStrictScope() {
		return(true);
	}

	/**
	 * This method indicates whether the construct is
	 * repetitive.
	 * 
	 * @return Whether the construct supports repetition
	 */
	public boolean isRepetition() {
		return false;
	}
	
	/**
	 * This method returns the list of roles that are
	 * responsible for initiating the activity. This can
	 * be used to determine whether the model is
	 * consistent in terms of decision makers subsequently
	 * initiating actions.
	 * 
	 * @return The list of initiator roles
	 */
	@Override
	public java.util.List<Role> initiatorRoles() {
		java.util.List<Role> ret=super.initiatorRoles();
		
		// Get the initiator roles from the first contained activity
		if (getBlock().size() > 0) {
			return(getBlock().get(0).initiatorRoles());
		}

		return(ret);
	}

	/**
	 * This method returns the list of roles that are
	 * associated with the outcome of the activity.
	 * 
	 * @return The list of final roles
	 */
	@Override
	public java.util.List<Role> finalRoles() {
		java.util.List<Role> ret=super.finalRoles();
				
		// Get the final roles from the first contained activity
		if (getBlock().size() > 0) {
			return(getBlock().get(0).finalRoles());
		}

		return(ret);
	}

	/**
	 * This method visits the model object using the supplied
	 * visitor.
	 * 
	 * @param visitor The visitor
	 */
	public void visit(Visitor visitor) {
		visitor.start(this);
		
		if (getBlock() != null) {
			getBlock().visit(visitor);
		}
		
		visitor.end(this);
	}

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        RecBlock that = (RecBlock) o;

        return !(m_label != null
                ? !m_label.equals(that.m_label)
                : that.m_label != null)
            && !(m_block != null
                ? !m_block.equals(that.m_block)
                : that.m_block != null);
    }

    @Override
    public int hashCode() {
        int result = m_label != null ? m_label.hashCode() : 0;
        return 31 * result + (m_block != null ? m_block.hashCode() : 0);
    }

	@Override
	public String toString() {
		return "rec "+m_label+" "+m_block;
	}

	private String m_label=null;
	private Block m_block=new Block();
}
