import { useEffect, useRef, ComponentPropsWithoutRef } from 'react';
import { styled } from '../../stitches.config';

import { Icon } from '../Icon';
import { IconButton } from '../IconButton';

const StyledContainer = styled('div', {
    display: 'flex',
    alignItems: 'center',
    columnGap: '$2',
    justifyContent: 'space-between',
    marginTop: '$1',
    borderColor: `$grey-4`,
    borderWidth: '$default',
    borderRadius: '$default',
    padding: '$2 $4',
});

const StyledDownloadButton = styled('a', {
    alignItems: 'center',
    backgroundColor: `transparent`,
    borderRadius: '$default',
    color: `$grey-1`,
    cursor: 'pointer',
    display: 'inline-flex',
    justifyContent: 'center',
    height: '$space$4',
    width: '$space$4',
    fontSize: '$space$4',
    margin: '0',
    outline: 'none',
    textAlign: 'center',
    userSelect: 'none',
    whiteSpace: 'nowrap',
    '&:hover': {
        color: `$grey-2`,
    },
    '&:focus': {
        outline: '2px solid $primary-1',
        outlineOffset: '2px',
    },
});

const StyledDescription = styled('span', {
    flex: '1',
    fontSize: '$sm',
    color: '$grey-1',
    whiteSpace: 'nowrap',
    overflow: 'hidden',
    textOverflow: 'ellipsis',
});

const mdiDelete =
    'M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z';
const mdiDownload = 'M5,20H19V18H5M19,9H15V3H9V9H5L12,16L19,9Z';

interface FileDisplayProps
    extends Omit<ComponentPropsWithoutRef<'div'>, 'onChange'> {
    /** Displayed File */
    file: File;

    /** Name of the file */
    name?: string;

    /** Disable deleting of the file */
    disabled?: boolean;

    /** Callback triggered on deleting of the file */
    onDelete: () => void;
}

export const FileDisplay = (props: FileDisplayProps): JSX.Element => {
    const { disabled, file, onDelete, ...otherProps } = props;

    // refs
    const objectURLRef = useRef<string | undefined>(undefined);

    // create the url
    useEffect(() => {
        const url = URL.createObjectURL(file);

        // save the ref
        objectURLRef.current = url;

        return () => {
            URL.revokeObjectURL(url);
        };
    }, [file]);

    return (
        <StyledContainer {...otherProps}>
            <StyledDownloadButton
                href={objectURLRef.current}
                download={file.name}
                title={`Download ${file.name}`}
                aria-label={`Download ${file.name}`}
                role="button"
            >
                <Icon aria-hidden="true" path={mdiDownload} />
            </StyledDownloadButton>
            <StyledDescription title={file.name} aria-label={file.name}>
                {file.name}
            </StyledDescription>
            <IconButton
                size={'sm'}
                disabled={disabled}
                onClick={() => onDelete()}
                title={`Remove ${file.name} from list`}
                aria-label={`Remove ${file.name} from list`}
            >
                <Icon aria-hidden="true" path={mdiDelete} />
            </IconButton>
        </StyledContainer>
    );
};
