/*
 * Decompiled with CFR 0.152.
 */
package org.shredzone.acme4j.connector;

import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.security.KeyPair;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import org.shredzone.acme4j.Login;
import org.shredzone.acme4j.Problem;
import org.shredzone.acme4j.Session;
import org.shredzone.acme4j.connector.Connection;
import org.shredzone.acme4j.connector.HttpConnector;
import org.shredzone.acme4j.connector.Resource;
import org.shredzone.acme4j.connector.TrimmingInputStream;
import org.shredzone.acme4j.exception.AcmeException;
import org.shredzone.acme4j.exception.AcmeNetworkException;
import org.shredzone.acme4j.exception.AcmeProtocolException;
import org.shredzone.acme4j.exception.AcmeRateLimitedException;
import org.shredzone.acme4j.exception.AcmeRetryAfterException;
import org.shredzone.acme4j.exception.AcmeServerException;
import org.shredzone.acme4j.exception.AcmeUnauthorizedException;
import org.shredzone.acme4j.exception.AcmeUserActionRequiredException;
import org.shredzone.acme4j.toolbox.AcmeUtils;
import org.shredzone.acme4j.toolbox.JSON;
import org.shredzone.acme4j.toolbox.JSONBuilder;
import org.shredzone.acme4j.toolbox.JoseUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultConnection
implements Connection {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultConnection.class);
    private static final int HTTP_OK = 200;
    private static final int HTTP_CREATED = 201;
    private static final int HTTP_NO_CONTENT = 204;
    private static final int HTTP_NOT_MODIFIED = 304;
    private static final String ACCEPT_HEADER = "Accept";
    private static final String ACCEPT_CHARSET_HEADER = "Accept-Charset";
    private static final String ACCEPT_LANGUAGE_HEADER = "Accept-Language";
    private static final String ACCEPT_ENCODING_HEADER = "Accept-Encoding";
    private static final String CACHE_CONTROL_HEADER = "Cache-Control";
    private static final String CONTENT_TYPE_HEADER = "Content-Type";
    private static final String DATE_HEADER = "Date";
    private static final String EXPIRES_HEADER = "Expires";
    private static final String IF_MODIFIED_SINCE_HEADER = "If-Modified-Since";
    private static final String LAST_MODIFIED_HEADER = "Last-Modified";
    private static final String LINK_HEADER = "Link";
    private static final String LOCATION_HEADER = "Location";
    private static final String REPLAY_NONCE_HEADER = "Replay-Nonce";
    private static final String RETRY_AFTER_HEADER = "Retry-After";
    private static final String DEFAULT_CHARSET = "utf-8";
    private static final String MIME_JSON = "application/json";
    private static final String MIME_JSON_PROBLEM = "application/problem+json";
    private static final String MIME_CERTIFICATE_CHAIN = "application/pem-certificate-chain";
    private static final URI BAD_NONCE_ERROR = URI.create("urn:ietf:params:acme:error:badNonce");
    private static final int MAX_ATTEMPTS = 10;
    private static final Pattern NO_CACHE_PATTERN = Pattern.compile("(?:^|.*?,)\\s*no-(?:cache|store)\\s*(?:,.*|$)", 2);
    private static final Pattern MAX_AGE_PATTERN = Pattern.compile("(?:^|.*?,)\\s*max-age=(\\d+)\\s*(?:,.*|$)", 2);
    private static final Pattern DIGITS_ONLY_PATTERN = Pattern.compile("^\\d+$");
    protected final HttpConnector httpConnector;
    protected final HttpClient httpClient;
    @Nullable
    protected HttpResponse<InputStream> lastResponse;

    public DefaultConnection(HttpConnector httpConnector) {
        this.httpConnector = Objects.requireNonNull(httpConnector, "httpConnector");
        this.httpClient = httpConnector.createClientBuilder().build();
    }

    @Override
    public void resetNonce(Session session) throws AcmeException {
        this.assertConnectionIsClosed();
        try {
            session.setNonce(null);
            URL newNonceUrl = session.resourceUrl(Resource.NEW_NONCE);
            LOG.debug("HEAD {}", (Object)newNonceUrl);
            this.sendRequest(session, newNonceUrl, (HttpRequest.Builder b) -> b.method("HEAD", HttpRequest.BodyPublishers.noBody()));
            this.logHeaders();
            int rc = this.getResponse().statusCode();
            if (rc != 200 && rc != 204) {
                this.throwAcmeException();
            }
            session.setNonce(this.getNonce().orElseThrow(() -> new AcmeProtocolException("Server did not provide a nonce")));
        }
        catch (IOException ex) {
            throw new AcmeNetworkException(ex);
        }
        finally {
            this.close();
        }
    }

    @Override
    public int sendRequest(URL url, Session session, @Nullable ZonedDateTime ifModifiedSince) throws AcmeException {
        Objects.requireNonNull(url, "url");
        Objects.requireNonNull(session, "session");
        this.assertConnectionIsClosed();
        LOG.debug("GET {}", (Object)url);
        try {
            this.sendRequest(session, url, (HttpRequest.Builder builder) -> {
                builder.GET();
                builder.header(ACCEPT_HEADER, MIME_JSON);
                if (ifModifiedSince != null) {
                    builder.header(IF_MODIFIED_SINCE_HEADER, ifModifiedSince.format(DateTimeFormatter.RFC_1123_DATE_TIME));
                }
            });
            this.logHeaders();
            this.getNonce().ifPresent(session::setNonce);
            int rc = this.getResponse().statusCode();
            if (rc != 200 && rc != 201 && (rc != 304 || ifModifiedSince == null)) {
                this.throwAcmeException();
            }
            return rc;
        }
        catch (IOException ex) {
            throw new AcmeNetworkException(ex);
        }
    }

    @Override
    public int sendCertificateRequest(URL url, Login login) throws AcmeException {
        return this.sendSignedRequest(url, null, login.getSession(), login.getKeyPair(), login.getAccountLocation(), MIME_CERTIFICATE_CHAIN);
    }

    @Override
    public int sendSignedPostAsGetRequest(URL url, Login login) throws AcmeException {
        return this.sendSignedRequest(url, null, login.getSession(), login.getKeyPair(), login.getAccountLocation(), MIME_JSON);
    }

    @Override
    public int sendSignedRequest(URL url, JSONBuilder claims, Login login) throws AcmeException {
        return this.sendSignedRequest(url, claims, login.getSession(), login.getKeyPair(), login.getAccountLocation(), MIME_JSON);
    }

    @Override
    public int sendSignedRequest(URL url, JSONBuilder claims, Session session, KeyPair keypair) throws AcmeException {
        return this.sendSignedRequest(url, claims, session, keypair, null, MIME_JSON);
    }

    @Override
    public JSON readJsonResponse() throws AcmeException {
        JSON jSON;
        block8: {
            this.expectContentType(Set.of(MIME_JSON, MIME_JSON_PROBLEM));
            InputStream in = this.getResponseBody();
            try {
                JSON result = JSON.parse(in);
                LOG.debug("Result JSON: {}", (Object)result);
                jSON = result;
                if (in == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new AcmeNetworkException(ex);
                }
            }
            in.close();
        }
        return jSON;
    }

    @Override
    public List<X509Certificate> readCertificates() throws AcmeException {
        List<X509Certificate> list;
        this.expectContentType(Set.of(MIME_CERTIFICATE_CHAIN));
        TrimmingInputStream in = new TrimmingInputStream(this.getResponseBody());
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            list = cf.generateCertificates(in).stream().map(X509Certificate.class::cast).collect(Collectors.toUnmodifiableList());
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ex) {
                throw new AcmeNetworkException(ex);
            }
            catch (CertificateException ex) {
                throw new AcmeProtocolException("Failed to read certificate", ex);
            }
        }
        in.close();
        return list;
    }

    @Override
    public void handleRetryAfter(String message) throws AcmeException {
        Optional<Instant> retryAfter = this.getRetryAfter();
        if (retryAfter.isPresent()) {
            throw new AcmeRetryAfterException(message, retryAfter.get());
        }
    }

    @Override
    public Optional<String> getNonce() {
        Optional<String> nonceHeaderOpt = this.getResponse().headers().firstValue(REPLAY_NONCE_HEADER).map(String::trim).filter(Predicate.not(String::isEmpty));
        if (nonceHeaderOpt.isPresent()) {
            String nonceHeader = nonceHeaderOpt.get();
            if (!AcmeUtils.isValidBase64Url(nonceHeader)) {
                throw new AcmeProtocolException("Invalid replay nonce: " + nonceHeader);
            }
            LOG.debug("Replay Nonce: {}", (Object)nonceHeader);
        }
        return nonceHeaderOpt;
    }

    @Override
    public URL getLocation() {
        return this.getResponse().headers().firstValue(LOCATION_HEADER).map(l -> {
            LOG.debug("Location: {}", l);
            return l;
        }).map(this::resolveRelative).orElseThrow(() -> new AcmeProtocolException("location header is missing"));
    }

    @Override
    public Optional<ZonedDateTime> getLastModified() {
        return this.getResponse().headers().firstValue(LAST_MODIFIED_HEADER).map(lm -> {
            try {
                return ZonedDateTime.parse(lm, DateTimeFormatter.RFC_1123_DATE_TIME);
            }
            catch (DateTimeParseException ex) {
                LOG.debug("Ignored invalid Last-Modified date: {}", lm, (Object)ex);
                return null;
            }
        });
    }

    @Override
    public Optional<ZonedDateTime> getExpiration() {
        Optional<ZonedDateTime> cacheControlHeader = this.getResponse().headers().firstValue(CACHE_CONTROL_HEADER).filter(Predicate.not(h -> NO_CACHE_PATTERN.matcher((CharSequence)h).matches())).map(MAX_AGE_PATTERN::matcher).filter(Matcher::matches).map(m -> Integer.parseInt(m.group(1))).filter(maxAge -> maxAge != 0).map(maxAge -> ZonedDateTime.now(ZoneId.of("UTC")).plusSeconds(maxAge.intValue()));
        if (cacheControlHeader.isPresent()) {
            return cacheControlHeader;
        }
        return this.getResponse().headers().firstValue(EXPIRES_HEADER).flatMap(header -> {
            try {
                return Optional.of(ZonedDateTime.parse(header, DateTimeFormatter.RFC_1123_DATE_TIME));
            }
            catch (DateTimeParseException ex) {
                LOG.debug("Ignored invalid Expires date: {}", header, (Object)ex);
                return Optional.empty();
            }
        });
    }

    @Override
    public Collection<URL> getLinks(String relation) {
        return this.collectLinks(relation).stream().map(this::resolveRelative).collect(Collectors.toUnmodifiableList());
    }

    @Override
    public void close() {
        this.lastResponse = null;
    }

    protected void sendRequest(Session session, URL url, Consumer<HttpRequest.Builder> body) throws IOException {
        try {
            HttpRequest.Builder builder = this.httpConnector.createRequestBuilder(url).header(ACCEPT_CHARSET_HEADER, DEFAULT_CHARSET).header(ACCEPT_LANGUAGE_HEADER, session.getLanguageHeader());
            if (session.networkSettings().isCompressionEnabled()) {
                builder.header(ACCEPT_ENCODING_HEADER, "gzip");
            }
            body.accept(builder);
            this.lastResponse = this.httpClient.send(builder.build(), HttpResponse.BodyHandlers.ofInputStream());
        }
        catch (InterruptedException ex) {
            throw new IOException("Request was interrupted", ex);
        }
    }

    protected int sendSignedRequest(URL url, @Nullable JSONBuilder claims, Session session, KeyPair keypair, @Nullable URL accountLocation, String accept) throws AcmeException {
        Objects.requireNonNull(url, "url");
        Objects.requireNonNull(session, "session");
        Objects.requireNonNull(keypair, "keypair");
        Objects.requireNonNull(accept, "accept");
        this.assertConnectionIsClosed();
        int attempt = 1;
        while (true) {
            try {
                return this.performRequest(url, claims, session, keypair, accountLocation, accept);
            }
            catch (AcmeServerException ex) {
                if (!BAD_NONCE_ERROR.equals(ex.getType())) {
                    throw ex;
                }
                if (attempt == 10) {
                    throw ex;
                }
                LOG.info("Bad Replay Nonce, trying again (attempt {}/{})", (Object)attempt, (Object)10);
                ++attempt;
                continue;
            }
            break;
        }
    }

    private int performRequest(URL url, @Nullable JSONBuilder claims, Session session, KeyPair keypair, @Nullable URL accountLocation, String accept) throws AcmeException {
        try {
            if (session.getNonce() == null) {
                this.resetNonce(session);
            }
            JSONBuilder jose = JoseUtils.createJoseRequest(url, keypair, claims, session.getNonce(), accountLocation != null ? accountLocation.toString() : null);
            String outputData = jose.toString();
            this.sendRequest(session, url, (HttpRequest.Builder builder) -> {
                builder.POST(HttpRequest.BodyPublishers.ofString(outputData));
                builder.header(ACCEPT_HEADER, accept);
                builder.header(CONTENT_TYPE_HEADER, "application/jose+json");
            });
            this.logHeaders();
            session.setNonce(this.getNonce().orElse(null));
            int rc = this.getResponse().statusCode();
            if (rc != 200 && rc != 201) {
                this.throwAcmeException();
            }
            return rc;
        }
        catch (IOException ex) {
            throw new AcmeNetworkException(ex);
        }
    }

    @Override
    public Optional<Instant> getRetryAfter() {
        return this.getResponse().headers().firstValue(RETRY_AFTER_HEADER).map(this::parseRetryAfterHeader);
    }

    private Instant parseRetryAfterHeader(String header) {
        try {
            if (DIGITS_ONLY_PATTERN.matcher(header).matches()) {
                int delta = Integer.parseInt(header);
                Instant date = this.getResponse().headers().firstValue(DATE_HEADER).map(d -> ZonedDateTime.parse(d, DateTimeFormatter.RFC_1123_DATE_TIME).toInstant()).orElseGet(Instant::now);
                return date.plusSeconds(delta);
            }
            return ZonedDateTime.parse(header, DateTimeFormatter.RFC_1123_DATE_TIME).toInstant();
        }
        catch (RuntimeException ex) {
            throw new AcmeProtocolException("Bad retry-after header value: " + header, ex);
        }
    }

    private InputStream getResponseBody() throws IOException {
        InputStream stream = this.getResponse().body();
        if (stream == null) {
            throw new AcmeProtocolException("Unexpected empty response");
        }
        if (this.getResponse().headers().firstValue("Content-Encoding").filter("gzip"::equalsIgnoreCase).isPresent()) {
            stream = new GZIPInputStream(stream);
        }
        return stream;
    }

    private void throwAcmeException() throws AcmeException {
        try {
            if (this.getResponse().headers().firstValue(CONTENT_TYPE_HEADER).map(AcmeUtils::getContentType).filter(MIME_JSON_PROBLEM::equals).isEmpty()) {
                throw new AcmeException("HTTP " + this.getResponse().statusCode());
            }
            Problem problem = new Problem(this.readJsonResponse(), this.getResponse().request().uri().toURL());
            String error = AcmeUtils.stripErrorPrefix(problem.getType().toString());
            if ("unauthorized".equals(error)) {
                throw new AcmeUnauthorizedException(problem);
            }
            if ("userActionRequired".equals(error)) {
                URI tos = this.collectLinks("terms-of-service").stream().findFirst().map(this::resolveUri).orElse(null);
                throw new AcmeUserActionRequiredException(problem, tos);
            }
            if ("rateLimited".equals(error)) {
                Optional<Instant> retryAfter = this.getRetryAfter();
                Collection<URL> rateLimits = this.getLinks("help");
                throw new AcmeRateLimitedException(problem, retryAfter.orElse(null), rateLimits);
            }
            throw new AcmeServerException(problem);
        }
        catch (IOException ex) {
            throw new AcmeNetworkException(ex);
        }
    }

    private void expectContentType(Set<String> expectedTypes) {
        String contentType = this.getResponse().headers().firstValue(CONTENT_TYPE_HEADER).map(AcmeUtils::getContentType).orElseThrow(() -> new AcmeProtocolException("No content type header found"));
        if (!expectedTypes.contains(contentType)) {
            throw new AcmeProtocolException("Unexpected content type: " + contentType);
        }
    }

    private HttpResponse<InputStream> getResponse() {
        if (this.lastResponse == null) {
            throw new IllegalStateException("Not connected.");
        }
        return this.lastResponse;
    }

    private void assertConnectionIsClosed() {
        if (this.lastResponse != null) {
            throw new IllegalStateException("Previous connection is not closed.");
        }
    }

    private void logHeaders() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        this.getResponse().headers().map().forEach((key, headers) -> headers.forEach(value -> LOG.debug("HEADER {}: {}", key, value)));
    }

    private Collection<String> collectLinks(String relation) {
        Pattern p = Pattern.compile("<(.*?)>\\s*;\\s*rel=\"?" + Pattern.quote(relation) + "\"?");
        return this.getResponse().headers().allValues(LINK_HEADER).stream().map(p::matcher).filter(Matcher::matches).map(m -> m.group(1)).peek(location -> LOG.debug("Link: {} -> {}", (Object)relation, location)).collect(Collectors.toUnmodifiableList());
    }

    private URL resolveRelative(String link) {
        try {
            return this.resolveUri(link).toURL();
        }
        catch (MalformedURLException ex) {
            throw new AcmeProtocolException("Cannot resolve relative link: " + link, ex);
        }
    }

    private URI resolveUri(String uri) {
        return this.getResponse().request().uri().resolve(uri);
    }
}

