/*
 * Decompiled with CFR 0.152.
 */
package org.simplify4u.plugins.keyserver;

import io.vavr.CheckedFunction0;
import io.vavr.control.Try;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.SecureRandom;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Named;
import lombok.Generated;
import org.bouncycastle.openpgp.PGPException;
import org.simplify4u.plugins.keyserver.KeyCacheSettings;
import org.simplify4u.plugins.keyserver.KeyServerClientSettings;
import org.simplify4u.plugins.keyserver.PGPKeyNotFound;
import org.simplify4u.plugins.keyserver.PGPKeysServerClient;
import org.simplify4u.plugins.pgp.KeyId;
import org.simplify4u.plugins.pgp.PublicKeyRingPack;
import org.simplify4u.plugins.pgp.PublicKeyUtils;
import org.simplify4u.plugins.utils.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Named
public class PGPKeysCache {
    @Generated
    private static final Logger LOGGER = LoggerFactory.getLogger(PGPKeysCache.class);
    private static final String NL = System.lineSeparator();
    private static final Object LOCK = new Object();
    private static final Pattern KEY_SERVERS_SPLIT_PATTERN = Pattern.compile("[;,\\s]");
    private File cachePath;
    private int notFoundRefreshHours;
    private KeyServerList keyServerList;
    private boolean offLine;

    PGPKeysCache() {
    }

    public void init(KeyCacheSettings cacheSettings, KeyServerClientSettings clientSettings) throws IOException {
        this.init(cacheSettings, this.prepareClients(cacheSettings.getKeyServers(), clientSettings));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void init(KeyCacheSettings cacheSettings, List<PGPKeysServerClient> pgpKeysServerClients) throws IOException {
        this.cachePath = cacheSettings.getCachePath();
        this.notFoundRefreshHours = cacheSettings.getNotFoundRefreshHours();
        this.keyServerList = PGPKeysCache.createKeyServerList(pgpKeysServerClients, cacheSettings.isLoadBalance());
        this.offLine = cacheSettings.isOffLine();
        LOGGER.info("Key server(s) - {}", (Object)this.keyServerList);
        Object object = LOCK;
        synchronized (object) {
            if (this.cachePath.exists()) {
                if (!this.cachePath.isDirectory()) {
                    throw new IOException("PGP keys cache path exist but is not a directory: " + this.cachePath);
                }
            } else if (this.cachePath.mkdirs()) {
                LOGGER.info("Create cache directory for PGP keys: {}", (Object)this.cachePath);
            } else {
                throw new IOException("Cache directory create error");
            }
        }
    }

    List<PGPKeysServerClient> prepareClients(String keyServers, KeyServerClientSettings clientSettings) {
        List keyServersList = Arrays.stream(KEY_SERVERS_SPLIT_PATTERN.split(keyServers)).map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toList());
        return keyServersList.stream().map(keyserver -> (PGPKeysServerClient)Try.of((CheckedFunction0 & Serializable)() -> PGPKeysServerClient.getClient(keyserver, clientSettings)).get()).collect(Collectors.toList());
    }

    static KeyServerList createKeyServerList(List<PGPKeysServerClient> pgpKeysServerClients, boolean loadBalance) {
        if (pgpKeysServerClients == null || pgpKeysServerClients.isEmpty()) {
            throw new IllegalArgumentException("Not allowed empty key server clients list ");
        }
        KeyServerList ret = pgpKeysServerClients.size() == 1 ? new KeyServerListOne() : (loadBalance ? new KeyServerListLoadBalance() : new KeyServerListFallback());
        return ret.withClients(pgpKeysServerClients);
    }

    public String getUrlForShowKey(KeyId keyID) {
        return this.keyServerList.getUriForShowKey(keyID).toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PublicKeyRingPack getKeyRing(KeyId keyID) throws IOException {
        String path = keyID.getHashPath();
        Path keyFile = new File(this.cachePath, path).toPath();
        Object object = LOCK;
        synchronized (object) {
            PublicKeyRingPack keyRing;
            if (!this.offLine) {
                this.checkNotFoundCache(path);
            }
            if (keyFile.toFile().exists() && !(keyRing = PGPKeysCache.loadKeyFromFile(keyFile, keyID)).isEmpty()) {
                return keyRing;
            }
            if (this.offLine) {
                throw new IOException("Key " + keyID + " not exits in cache under path: " + keyFile + " it is not possible to download in offline mode");
            }
            return (PublicKeyRingPack)Try.of((CheckedFunction0 & Serializable)() -> this.keyServerList.execute(keysServerClient -> PGPKeysCache.receiveKey(keyFile, keyID, keysServerClient))).onFailure(PGPKeyNotFound.class, e -> this.writeNotFoundCache(path)).get();
        }
    }

    private void writeNotFoundCache(String keyFilePath) {
        Path file = new File(this.cachePath, keyFilePath + ".404").toPath();
        try {
            Files.write(file, String.valueOf(System.currentTimeMillis()).getBytes(StandardCharsets.US_ASCII), new OpenOption[0]);
        }
        catch (IOException e) {
            LOGGER.warn("Write file: {} exception: {}", (Object)file, (Object)ExceptionUtils.getMessage(e));
            PGPKeysCache.deleteFile(file);
        }
    }

    private void checkNotFoundCache(String keyFilePath) throws PGPKeyNotFound {
        File file = new File(this.cachePath, keyFilePath + ".404");
        if (file.isFile()) {
            long markTime = (Long)Try.of((CheckedFunction0 & Serializable)() -> {
                byte[] cacheContent = Files.readAllBytes(file.toPath());
                return Long.parseLong(new String(cacheContent, StandardCharsets.US_ASCII));
            }).onFailure(e -> LOGGER.warn("Read cache file: {}", (Object)file, e)).getOrElse((Object)0L);
            Duration elapsedTime = Duration.ofMillis(System.currentTimeMillis() - markTime);
            if (elapsedTime.toHours() > (long)this.notFoundRefreshHours) {
                LOGGER.debug("KeyNotFound remove cache {} - mark time: {} elapsed: {}", new Object[]{file, markTime, elapsedTime});
                PGPKeysCache.deleteFile(file.toPath());
            } else {
                LOGGER.debug("KeyNotFound from cache {} - mark time: {} elapsed: {}", new Object[]{file, markTime, elapsedTime});
                throw new PGPKeyNotFound();
            }
        }
    }

    private static PublicKeyRingPack loadKeyFromFile(Path keyFile, KeyId keyID) throws IOException {
        PublicKeyRingPack keyRing = PublicKeyRingPack.EMPTY;
        try (InputStream keyFileStream = Files.newInputStream(keyFile, new OpenOption[0]);){
            keyRing = PublicKeyUtils.loadPublicKeyRing(keyFileStream, keyID);
        }
        catch (PGPException e) {
            throw new IOException(e);
        }
        finally {
            if (keyRing.isEmpty()) {
                PGPKeysCache.deleteFile(keyFile);
            }
        }
        return keyRing;
    }

    private static PublicKeyRingPack receiveKey(Path keyFile, KeyId keyId, PGPKeysServerClient keysServerClient) throws IOException {
        Path dir = keyFile.getParent();
        if (dir == null) {
            throw new IOException("No parent dir for: " + keyFile);
        }
        if (dir.toFile().exists() && !dir.toFile().isDirectory()) {
            throw new IOException("Path exist but it isn't directory: " + dir);
        }
        Files.createDirectories(dir, new FileAttribute[0]);
        Path partFile = Files.createTempFile(String.valueOf(keyId), "pgp-public-key", new FileAttribute[0]);
        try {
            try (BufferedOutputStream output = new BufferedOutputStream(Files.newOutputStream(partFile, new OpenOption[0]));){
                keysServerClient.copyKeyToOutputStream(keyId, output, PGPKeysCache::onRetry);
            }
            PGPKeysCache.moveFile(partFile, keyFile);
        }
        catch (IOException e) {
            PGPKeysCache.deleteFile(keyFile);
            PGPKeysCache.deleteFile(partFile);
            throw e;
        }
        LOGGER.info("Receive key: {}{}\tto {}", new Object[]{keysServerClient.getUriForGetKey(keyId), NL, keyFile});
        PublicKeyRingPack keyRingPack = PGPKeysCache.loadKeyFromFile(keyFile, keyId);
        if (keyRingPack.isEmpty()) {
            throw new IOException(String.format("Can't find public key %s in download file: %s", keyId, keyFile));
        }
        return keyRingPack;
    }

    private static void onRetry(InetAddress address, int numberOfRetryAttempts, Duration waitInterval, Throwable lastThrowable) {
        LOGGER.warn("[Retry #{} waiting: {}] Last address {} with problem: [{}] {}", new Object[]{numberOfRetryAttempts, waitInterval, address, lastThrowable.getClass().getName(), ExceptionUtils.getMessage(lastThrowable)});
    }

    private static void deleteFile(Path file) {
        Optional.ofNullable(file).ifPresent(filePath -> Try.run(() -> Files.deleteIfExists(filePath)).onFailure(e -> LOGGER.warn("Can't delete: {} with exception: {}", filePath, (Object)e.getMessage())));
    }

    private static void moveFile(Path source, Path destination) throws IOException {
        try {
            Files.move(source, destination, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (FileSystemException fse) {
            try {
                Thread.sleep(250L + (long)new SecureRandom().nextInt(1000));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            Files.move(source, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    static class KeyServerListLoadBalance
    extends KeyServerList {
        private int lastIndex = 0;

        KeyServerListLoadBalance() {
        }

        @Override
        String getName() {
            return "load balance";
        }

        @Override
        PublicKeyRingPack execute(KeyServerExecutor executor) throws IOException {
            for (int i = 0; i < this.keysServerClients.size(); ++i) {
                PGPKeysServerClient client = (PGPKeysServerClient)this.keysServerClients.get(this.lastIndex);
                this.lastIndex = (this.lastIndex + 1) % this.keysServerClients.size();
                PublicKeyRingPack pgpPublicKeys = this.executeWithClient(executor, client);
                if (pgpPublicKeys.isEmpty()) continue;
                return pgpPublicKeys;
            }
            LOGGER.error("All servers from list failed");
            throw this.lastException;
        }
    }

    static class KeyServerListFallback
    extends KeyServerList {
        KeyServerListFallback() {
        }

        @Override
        String getName() {
            return "fallback";
        }

        @Override
        PublicKeyRingPack execute(KeyServerExecutor executor) throws IOException {
            for (PGPKeysServerClient client : this.keysServerClients) {
                PublicKeyRingPack pgpPublicKeys = this.executeWithClient(executor, client);
                if (pgpPublicKeys.isEmpty()) continue;
                return pgpPublicKeys;
            }
            LOGGER.error("All servers from list failed");
            throw this.lastException;
        }
    }

    static class KeyServerListOne
    extends KeyServerList {
        KeyServerListOne() {
        }

        @Override
        String getName() {
            return "one item";
        }

        @Override
        PublicKeyRingPack execute(KeyServerExecutor executor) throws IOException {
            return executor.run(this.lastClient);
        }

        @Override
        public String toString() {
            return this.lastClient.toString();
        }
    }

    static abstract class KeyServerList {
        protected List<PGPKeysServerClient> keysServerClients = new ArrayList<PGPKeysServerClient>();
        protected PGPKeysServerClient lastClient;
        protected IOException lastException;

        KeyServerList() {
        }

        KeyServerList withClients(List<PGPKeysServerClient> keysServerClients) {
            this.keysServerClients = keysServerClients;
            this.lastClient = keysServerClients.get(0);
            return this;
        }

        URI getUriForShowKey(KeyId keyID) {
            return this.lastClient.getUriForShowKey(keyID);
        }

        protected PublicKeyRingPack executeWithClient(KeyServerExecutor executor, PGPKeysServerClient client) {
            try {
                PublicKeyRingPack ret = executor.run(client);
                this.lastClient = client;
                return ret;
            }
            catch (IOException e) {
                if (!(this.lastException instanceof PGPKeyNotFound)) {
                    this.lastException = e;
                }
                LOGGER.warn("{} throw exception: {} - {} try next client", new Object[]{client, ExceptionUtils.getMessage(e), this.getName()});
                return PublicKeyRingPack.EMPTY;
            }
        }

        public String toString() {
            return String.format("%s list: %s", this.getName(), this.keysServerClients);
        }

        abstract String getName();

        abstract PublicKeyRingPack execute(KeyServerExecutor var1) throws IOException;
    }

    @FunctionalInterface
    static interface KeyServerExecutor {
        public PublicKeyRingPack run(PGPKeysServerClient var1) throws IOException;
    }
}

