/*_############################################################################
  _## 
  _##  SNMP4J-Agent 3 - ManagedObject.java  
  _## 
  _##  Copyright (C) 2005-2024  Frank Fock (SNMP4J.org)
  _##  
  _##  Licensed under the Apache License, Version 2.0 (the "License");
  _##  you may not use this file except in compliance with the License.
  _##  You may obtain a copy of the License at
  _##  
  _##      http://www.apache.org/licenses/LICENSE-2.0
  _##  
  _##  Unless required by applicable law or agreed to in writing, software
  _##  distributed under the License is distributed on an "AS IS" BASIS,
  _##  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  _##  See the License for the specific language governing permissions and
  _##  limitations under the License.
  _##  
  _##########################################################################*/


package org.snmp4j.agent;

import org.snmp4j.smi.OID;
import org.snmp4j.agent.request.SubRequest;

import java.util.function.Function;

/**
 * The {@code ManagedObject} interface defines the basic operations
 * for all SNMP4J manageable objects.
 *
 * @author Frank Fock
 * @version 3.8.0
 */
public interface ManagedObject<SR extends SubRequest<?>> {

    /**
     * Return the scope of object identifiers this managed object is managing.
     *
     * @return the {@code MOScope} that defines a range (possibly also a single
     * or none instance OID) of object IDs managed by this managed object.
     */
    MOScope getScope();

    /**
     * Find the first object ID (OID) in the specified search range.
     *
     * @param range
     *         the {@code MOScope} for the search.
     *
     * @return the {@code OID} that is included in the search {@code range}
     * and {@code null} if no such instances could be found.
     */
    OID find(MOScope range);

    /**
     * Find the first object ID (OID) in the specified search range.
     *
     * @param query
     *         the {@code MOQuery} for the search.
     * @param filter
     *         the filter criteria that needs to be true on the OID to be returned.
     *         If the filter returns {@code false} on {@link Function#apply(Object)} then the next
     *         OID is being tested that is in the provided range.
     *
     * @return the first {@code OID} that is included in the search {@code range} and
     * for which the filter criteria is {@code true}. Otherwise {@code null} if no such instances could be found.
     * @since 3.8.0
     */
    default OID find(MOQuery query, Function<OID, Boolean> filter) {
        DefaultMOQuery mutableQuery = new DefaultMOQuery(query.getScope());
        OID oid;
        do {
            oid = find(mutableQuery);
            if (oid == null) {
                break;
            }
            if (filter.apply(oid)) {
                return oid;
            }
            mutableQuery.setNewExclusiveLowerBound(oid);
        }
        while (!mutableQuery.isEmpty());
        return null;
    }

    /**
     * Process a GET request and return the result in the supplied sub-request.
     *
     * @param request
     *         the {@code SubRequest} to process.
     */
    void get(SR request);

    /**
     * Find the successor instance for the object instance ID (OID) given
     * by the supplied sub-request and returns it within the supplied sub-request
     * object.
     *
     * @param request
     *         the {@code SubRequest} to process.
     *
     * @return {@code true} if the search request found an appropriate instance,
     * {@code false} otherwise.
     * @deprecated Use {@link #next(SubRequest, Function)} instead to apply {@link org.snmp4j.agent.security.VACM}
     * inner ManagedObject OID filtering. This is import especially for multi OID objects like tables and
     * subtrees.
     */
    boolean next(SR request);

    /**
     * Find the successor instance for the object instance ID (OID) given
     * by the supplied sub-request and returns it within the supplied sub-request
     * object.
     *
     * @param request
     *         the {@code SubRequest} to process.
     * @param filter
     *         the filter criteria that needs to be true on the OID to be returned.
     *         If the filter returns {@code false} on {@link Function#apply(Object)} then the next
     *         OID is being tested that is in the provided request's range.
     *
     * @return {@code true} if the search request found an appropriate instance,
     * {@code false} otherwise.
     * @since 3.8.0
     */
    default boolean next(SR request, Function<OID, Boolean> filter) {
        // no filtering by default! CAUTION: This default method MUST be overwritten by any multi OID ManagedObject.
        return next(request);
    }

    /**
     * Prepare a SET (sub)request. This method represents the first phase of a
     * two phase commit. During preparation all necessary resources should be
     * locked in order to be able to execute the commit without claiming
     * additional resources.
     *
     * @param request
     *         the {@code SubRequest} to process.
     */
    void prepare(SR request);

    /**
     * Commit a previously prepared SET (sub)request. This is the second phase
     * of a two phase commit. The change is committed but the resources locked
     * during prepare not freed yet.
     *
     * @param request
     *         the {@code SubRequest} to process.
     */
    void commit(SR request);

    /**
     * Compensate (undo) a (sub)request when a commit of another subrequest
     * failed with an error. This also frees any resources locked during
     * the preparation phase.
     *
     * @param request
     *         the {@code SubRequest} to process.
     */
    void undo(SR request);

    /**
     * Cleanup a (sub)request and free all resources locked during
     * the preparation phase.
     *
     * @param request
     *         the {@code SubRequest} to process.
     */
    void cleanup(SR request);

}
