/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.performance.measure;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.time.Duration;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.sonarsource.performance.measure.DurationMeasure;
import org.sonarsource.performance.measure.DurationMeasureFiles;
import org.sonarsource.performance.measure.log.Logger;

public class DurationMeasureMerger {
    private static final Logger LOG = Logger.get(DurationMeasureMerger.class);
    private static final Pattern DATE_TIME_REGEX = Pattern.compile("^\\d{4}-\\d{2}-\\d{2}-\\d{2}h\\d{2}m\\d{2}\\.\\d{3}$");
    private static final DecimalFormatSymbols SYMBOLS = DecimalFormatSymbols.getInstance(Locale.ROOT);
    private static final NumberFormat SCORE_FORMAT = new DecimalFormat("0.0'%'", SYMBOLS);
    private static final int MINUTES_PER_HOUR = 60;
    private static final int SECONDS_PER_MINUTE = 60;
    public static final String PERFORMANCE_STATISTICS_FILE_NAME = "performance.statistics.txt";
    public static final String PERFORMANCE_SCORE_FILE_NAME = "performance.score.json";
    public static final String SCORE_OVERSTEP_THRESHOLD = "scoreOverstepThreshold";
    public static final String SCORE = "score";
    public static final String LINK = "link";
    private Map<String, String> categoryNames = new HashMap<String, String>();
    private Predicate<String> groupedMeasurePredicate = name -> false;
    private String performanceFileName = "performance.measure.json";
    private String repositoryBaseUrl = "";
    private double scoreMaxThreshold = 1.02;

    public DurationMeasureMerger withMaxScoreThreshold(double scoreMaxThreshold) {
        this.scoreMaxThreshold = scoreMaxThreshold;
        return this;
    }

    public DurationMeasureMerger forPerformanceFileName(String performanceFileName) {
        this.performanceFileName = performanceFileName;
        return this;
    }

    public DurationMeasureMerger withCategoryNames(Map<String, String> categoryNames) {
        this.categoryNames = categoryNames;
        return this;
    }

    public DurationMeasureMerger groupedBy(Predicate<String> groupedMeasurePredicate) {
        this.groupedMeasurePredicate = groupedMeasurePredicate;
        return this;
    }

    public DurationMeasureMerger withRepositoryBaseUrl(String repositoryBaseUrl) {
        this.repositoryBaseUrl = repositoryBaseUrl;
        return this;
    }

    public void mergeProjectPerformances(Path latestAnalysisFolder, Path destDirectory, List<String> namesToMergeOnUpperLevel) throws IOException {
        LOG.info(() -> "Merge Project Performances of " + latestAnalysisFolder.getFileName());
        DurationMeasure mergedMeasure = null;
        for (Path performanceFile : this.findPerformanceFiles(latestAnalysisFolder)) {
            DurationMeasure measure = DurationMeasureFiles.fromJsonWithoutObservationCost(performanceFile);
            namesToMergeOnUpperLevel.forEach(measure::recursiveMergeOnUpperLevel);
            mergedMeasure = mergedMeasure == null ? measure : mergedMeasure.merge(measure);
        }
        if (mergedMeasure == null) {
            LOG.warning(() -> "Can't find any '" + this.performanceFileName + "' in " + latestAnalysisFolder);
            return;
        }
        Path mergedPerformanceFile = destDirectory.resolve(this.performanceFileName);
        LOG.info(() -> "Merged Performance File: " + mergedPerformanceFile);
        DurationMeasureFiles.writeJson(mergedPerformanceFile, mergedMeasure);
        Path performanceStatisticsFile = destDirectory.resolve(PERFORMANCE_STATISTICS_FILE_NAME);
        LOG.info(() -> "Performance Statistics File: " + performanceStatisticsFile);
        DurationMeasureFiles.writeStatistics(performanceStatisticsFile, mergedMeasure, this.categoryNames, this.groupedMeasurePredicate);
    }

    public void compareWithRelease(Path releaseFolder, Path latestAnalysisFolder, Path destDirectory, Path globalPerformanceScoreFile) throws IOException {
        String scoreValue;
        boolean scoreOverstepThreshold;
        double durationRatio;
        LOG.info(() -> "Compare Performances between release " + releaseFolder.getFileName() + " and latest " + latestAnalysisFolder.getFileName());
        Set<Path> releasePerformanceFiles = this.findPerformanceFiles(releaseFolder);
        Set<Path> latestPerformanceFiles = this.findPerformanceFiles(latestAnalysisFolder);
        Set releaseAnalysisProjects = releasePerformanceFiles.stream().map(DurationMeasureMerger::projectName).collect(Collectors.toCollection(TreeSet::new));
        Set latestAnalysisProjects = latestPerformanceFiles.stream().map(DurationMeasureMerger::projectName).collect(Collectors.toCollection(TreeSet::new));
        TreeSet allProjects = new TreeSet(releaseAnalysisProjects);
        allProjects.addAll(latestAnalysisProjects);
        JsonArray projectsMissingInRelease = new JsonArray();
        JsonArray projectsMissingInLatest = new JsonArray();
        JsonArray commonProjectsCompared = new JsonArray();
        long releaseAnalysisDuration = 0L;
        long latestAnalysisDuration = 0L;
        for (String projectName : allProjects) {
            Optional<Path> releasePerformanceFile = releasePerformanceFiles.stream().filter(path -> DurationMeasureMerger.projectName(path).equals(projectName)).findFirst();
            Optional<Path> latestAnalysisPerformanceFile = latestPerformanceFiles.stream().filter(path -> DurationMeasureMerger.projectName(path).equals(projectName)).findFirst();
            if (!releasePerformanceFile.isPresent()) {
                projectsMissingInRelease.add(projectName);
                continue;
            }
            if (!latestAnalysisPerformanceFile.isPresent()) {
                projectsMissingInLatest.add(projectName);
                continue;
            }
            commonProjectsCompared.add(projectName);
            releaseAnalysisDuration += DurationMeasureMerger.analysisDuration(releasePerformanceFile.get());
            latestAnalysisDuration += DurationMeasureMerger.analysisDuration(latestAnalysisPerformanceFile.get());
        }
        JsonObject score = new JsonObject();
        if (latestAnalysisDuration == 0L || releaseAnalysisDuration == 0L) {
            durationRatio = 0.0;
            scoreOverstepThreshold = true;
            scoreValue = "Zero Duration";
        } else {
            durationRatio = (double)Math.round((double)latestAnalysisDuration * 10000.0 / (double)releaseAnalysisDuration) / 10000.0;
            scoreOverstepThreshold = durationRatio > this.scoreMaxThreshold;
            scoreValue = SCORE_FORMAT.format(durationRatio * 100.0);
        }
        score.addProperty(SCORE_OVERSTEP_THRESHOLD, Boolean.valueOf(scoreOverstepThreshold));
        score.addProperty(SCORE, scoreValue);
        score.addProperty("durationRatioCompareToRelease", (Number)durationRatio);
        score.addProperty("comparedWithRelease", releaseFolder.getFileName().toString());
        score.addProperty("releaseAnalysisDuration", DurationMeasureMerger.durationNanosToString(releaseAnalysisDuration));
        score.addProperty("latestAnalysisDuration", DurationMeasureMerger.durationNanosToString(latestAnalysisDuration));
        score.addProperty("releaseAnalysisDurationNanos", (Number)releaseAnalysisDuration);
        score.addProperty("latestAnalysisDurationNanos", (Number)latestAnalysisDuration);
        score.add("projectsMissingInRelease", (JsonElement)projectsMissingInRelease);
        score.add("projectsMissingInLatest", (JsonElement)projectsMissingInLatest);
        score.add("comparedProjects", (JsonElement)commonProjectsCompared);
        Path performanceScoreFile = destDirectory.resolve(PERFORMANCE_SCORE_FILE_NAME);
        DurationMeasureMerger.writeJson(performanceScoreFile, score);
        JsonObject globalScore = new JsonObject();
        globalScore.add(SCORE_OVERSTEP_THRESHOLD, score.get(SCORE_OVERSTEP_THRESHOLD));
        globalScore.add(SCORE, score.get(SCORE));
        String linkRelativePart = globalPerformanceScoreFile.getParent().relativize(performanceScoreFile).toString().replace(File.separatorChar, '/');
        globalScore.addProperty(LINK, this.repositoryBaseUrl + linkRelativePart);
        DurationMeasureMerger.writeJson(globalPerformanceScoreFile, globalScore);
    }

    private static void writeJson(Path path, JsonObject jsonObject) throws IOException {
        String json = new GsonBuilder().setPrettyPrinting().create().toJson((JsonElement)jsonObject);
        LOG.info(() -> "Writing: " + path);
        Files.write(path, json.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
    }

    private static String durationNanosToString(long durationNanos) {
        Duration duration = Duration.ofNanos(durationNanos);
        return String.format("%dh%02dm%02ds", duration.toHours(), duration.toMinutes() % 60L, duration.getSeconds() % 60L);
    }

    private static String projectName(Path performanceFilePath) {
        return performanceFilePath.getParent().getParent().getFileName().toString();
    }

    private static long analysisDuration(Path performanceFile) throws IOException {
        DurationMeasure measure = DurationMeasureFiles.fromJsonWithoutObservationCost(performanceFile);
        return measure.durationNanos();
    }

    private Set<Path> findPerformanceFiles(Path analysisPath) throws IOException {
        TreeSet<Path> performanceFiles = new TreeSet<Path>();
        for (Path projectDirectory : DurationMeasureMerger.getSubDirectories(analysisPath)) {
            DurationMeasureMerger.getSubDirectories(projectDirectory).stream().filter(path -> DATE_TIME_REGEX.matcher(path.getFileName().toString()).find()).sorted(Comparator.comparing(Object::toString).reversed()).map(path -> path.resolve(this.performanceFileName)).filter(x$0 -> Files.exists(x$0, new LinkOption[0])).findFirst().ifPresent(performanceFiles::add);
        }
        return performanceFiles;
    }

    private static List<Path> getSubDirectories(Path parentDirectory) throws IOException {
        try (Stream<Path> stream = Files.list(parentDirectory);){
            List<Path> list = stream.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).collect(Collectors.toList());
            return list;
        }
    }
}

