/*
 * SonarQube Java
 * Copyright (C) 2012-2020 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.java.checks.naming;

import org.sonar.check.Rule;
import org.sonar.check.RuleProperty;
import org.sonar.java.checks.helpers.UnitTestUtils;
import org.sonar.plugins.java.api.IssuableSubscriptionVisitor;
import org.sonar.plugins.java.api.JavaFileScannerContext;
import org.sonar.plugins.java.api.tree.ClassTree;
import org.sonar.plugins.java.api.tree.IdentifierTree;
import org.sonar.plugins.java.api.tree.MethodTree;
import org.sonar.plugins.java.api.tree.Tree;

import javax.annotation.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

@Rule(key = "S3577")
public class BadTestClassNameCheck extends IssuableSubscriptionVisitor {

  private static final String DEFAULT_FORMAT = "^((Test|IT)[a-zA-Z0-9]+|[A-Z][a-zA-Z0-9]*(Test|IT|TestCase|ITCase))$";

  @RuleProperty(
    key = "format",
    description = "Regular expression against which test class names are checked.",
    defaultValue = "" + DEFAULT_FORMAT)
  public String format = DEFAULT_FORMAT;

  private Pattern pattern = null;

  @Override
  public List<Tree.Kind> nodesToVisit() {
    return Collections.singletonList(Tree.Kind.CLASS);
  }

  @Override
  public void setContext(JavaFileScannerContext context) {
    if (pattern == null) {
      pattern = Pattern.compile(format, Pattern.DOTALL);
    }
    super.setContext(context);
  }

  @Override
  public void visitNode(Tree tree) {
    if (!hasSemantic()) {
      return;
    }
    ClassTree classTree = (ClassTree) tree;
    IdentifierTree simpleName = classTree.simpleName();
    if (hasInvalidName(simpleName) && isTestClass(classTree)) {
      reportIssue(simpleName, "Rename class \"" + simpleName.name() + "\" to match the regular expression: '" + format + "'");
    }
  }

  private boolean hasInvalidName(@Nullable IdentifierTree className) {
    return className != null && !pattern.matcher(className.name()).matches();
  }

  private static boolean isTestClass(ClassTree classTree) {
    return isTopLevelClass(classTree)
      && (hasTestMethod(classTree.members()) || hasNestedClass(classTree));
  }

  private static boolean isTopLevelClass(ClassTree classTree) {
    return classTree.symbol().owner().isPackageSymbol();
  }

  private static boolean hasTestMethod(List<Tree> members) {
    return members.stream()
      .filter(member -> member.is(Tree.Kind.METHOD))
      .map(MethodTree.class::cast)
      .anyMatch(UnitTestUtils::hasTestAnnotation);
  }

  private static boolean hasNestedClass(ClassTree classTree) {
    return classTree.members()
      .stream()
      .filter(member -> member.is(Tree.Kind.CLASS))
      .map(ClassTree.class::cast)
      .anyMatch(UnitTestUtils::hasNestedAnnotation);
  }

}
