/*
 * SonarScanner Java Library
 * Copyright (C) 2011-2025 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonarsource.scanner.lib.internal.facade.forked;

import com.google.gson.Gson;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Map;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonarsource.scanner.downloadcache.CachedFile;
import org.sonarsource.scanner.downloadcache.DownloadCache;
import org.sonarsource.scanner.downloadcache.Downloader;
import org.sonarsource.scanner.downloadcache.HashMismatchException;
import org.sonarsource.scanner.lib.internal.MessageException;
import org.sonarsource.scanner.lib.internal.http.ScannerHttpClient;
import org.sonarsource.scanner.lib.internal.util.Either;
import org.sonarsource.scanner.lib.internal.util.ProcessWrapperFactory;
import org.sonarsource.scanner.lib.internal.util.System2;

import static org.sonarsource.scanner.lib.ScannerProperties.SCANNER_ENGINE_JAR_PATH;

public class ScannerEngineLauncherFactory {

  private static final Logger LOG = LoggerFactory.getLogger(ScannerEngineLauncherFactory.class);

  static final String API_PATH_ENGINE = "/analysis/engine";
  private final JavaRunnerFactory javaRunnerFactory;

  public ScannerEngineLauncherFactory(System2 system) {
    this.javaRunnerFactory = new JavaRunnerFactory(system, new ProcessWrapperFactory());
  }

  ScannerEngineLauncherFactory(JavaRunnerFactory javaRunnerFactory) {
    this.javaRunnerFactory = javaRunnerFactory;
  }

  public ScannerEngineLauncher createLauncher(ScannerHttpClient scannerHttpClient, DownloadCache downloadCache, Map<String, String> properties) {
    JavaRunner javaRunner = javaRunnerFactory.createRunner(scannerHttpClient, downloadCache, properties);
    jreSanityCheck(javaRunner);
    var scannerEngine = getScannerEngine(scannerHttpClient, downloadCache, properties);
    return new ScannerEngineLauncher(javaRunner, scannerEngine);
  }

  private static void jreSanityCheck(JavaRunner javaRunner) {
    javaRunner.execute(Collections.singletonList("--version"), null, LOG::debug);
  }

  private static Either<CachedFile, Path> getScannerEngine(ScannerHttpClient scannerHttpClient, DownloadCache downloadCache, Map<String, String> properties) {
    String scannerEngineJarPathPropValue = properties.get(SCANNER_ENGINE_JAR_PATH);
    if (scannerEngineJarPathPropValue != null) {
      var path = Paths.get(scannerEngineJarPathPropValue).toAbsolutePath();
      if (!Files.isRegularFile(path)) {
        throw new MessageException("Scanner Engine jar path '" + scannerEngineJarPathPropValue + "' does not exist. Please check property '" + SCANNER_ENGINE_JAR_PATH + "'.");
      }
      LOG.info("Using the configured Scanner Engine '{}'", path);
      return Either.forRight(path);
    }
    return Either.forLeft(provisionScannerEngine(scannerHttpClient, downloadCache, true));
  }

  private static CachedFile provisionScannerEngine(ScannerHttpClient scannerHttpClient, DownloadCache downloadCache, boolean retry) {
    try {
      var scannerEngineMetadata = getScannerEngineMetadata(scannerHttpClient);
      return downloadCache.getOrDownload(scannerEngineMetadata.getFilename(), scannerEngineMetadata.getSha256(), "SHA-256",
        new ScannerEngineDownloader(scannerHttpClient, scannerEngineMetadata));
    } catch (HashMismatchException e) {
      if (retry) {
        // A new scanner-engine might have been published between the metadata fetch and the download
        LOG.warn("Failed to get the scanner-engine, retrying...");
        return provisionScannerEngine(scannerHttpClient, downloadCache, false);
      }
      throw new IllegalStateException("Unable to provision the Scanner Engine", e);
    }
  }

  private static ScannerEngineMetadata getScannerEngineMetadata(ScannerHttpClient scannerHttpClient) {
    try {
      String response = scannerHttpClient.callRestApi(API_PATH_ENGINE);
      return new Gson().fromJson(response, ScannerEngineMetadata.class);
    } catch (Exception e) {
      throw new MessageException("Failed to get the scanner-engine metadata: " + e.getMessage(), e);
    }
  }

  static class ScannerEngineMetadata extends ResourceMetadata {
    public ScannerEngineMetadata(String filename, String sha256, @Nullable String downloadUrl) {
      super(filename, sha256, downloadUrl);
    }
  }

  static class ScannerEngineDownloader implements Downloader {
    private final ScannerHttpClient connection;
    private final ScannerEngineMetadata scannerEngineMetadata;

    ScannerEngineDownloader(ScannerHttpClient connection, ScannerEngineMetadata scannerEngineMetadata) {
      this.connection = connection;
      this.scannerEngineMetadata = scannerEngineMetadata;
    }

    @Override
    public void download(String filename, Path toFile) throws IOException {
      if (StringUtils.isNotBlank(scannerEngineMetadata.getDownloadUrl())) {
        connection.downloadFromExternalUrl(scannerEngineMetadata.getDownloadUrl(), toFile);
      } else {
        connection.downloadFromRestApi(API_PATH_ENGINE, toFile);
      }
    }
  }
}
