/*
 * Decompiled with CFR 0.152.
 */
package org.spdx.utility;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spdx.Configuration;

public final class DownloadCache {
    private static final Logger logger = LoggerFactory.getLogger(DownloadCache.class);
    private static final int READ_TIMEOUT = 5000;
    private static final int IO_BUFFER_SIZE = 8192;
    private static final long DEFAULT_CACHE_CHECK_INTERVAL_SECS = 86400L;
    static final List<String> WHITE_LIST = Collections.unmodifiableList(Arrays.asList("spdx.org", "spdx.dev", "spdx.com", "spdx.info"));
    private static DownloadCache singleton;
    private final String cacheDir = (System.getenv("XDG_CACHE_HOME") == null || System.getenv("XDG_CACHE_HOME").trim().isEmpty() ? System.getProperty("user.home") + File.separator + ".cache" : System.getenv("XDG_CACHE_HOME")) + File.separator + "Spdx-Java-Library";
    private static final String CONFIG_PROPERTY_CACHE_ENABLED = "org.spdx.downloadCacheEnabled";
    private static final String CONFIG_PROPERTY_CACHE_CHECK_INTERVAL_SECS = "org.spdx.downloadCacheCheckIntervalSecs";
    private final boolean cacheEnabled;
    private final long cacheCheckIntervalSecs;
    private final DateTimeFormatter iso8601 = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.000'Z'").withZone(ZoneOffset.UTC);

    private DownloadCache() {
        boolean tmpCacheEnabled = Boolean.parseBoolean(Configuration.getInstance().getProperty(CONFIG_PROPERTY_CACHE_ENABLED, "false"));
        if (tmpCacheEnabled) {
            try {
                File cacheDirectory = new File(this.cacheDir);
                Files.createDirectories(cacheDirectory.toPath(), new FileAttribute[0]);
            }
            catch (IOException ioe) {
                logger.warn("Unable to create cache directory '{}'; continuing with cache disabled.", (Object)this.cacheDir, (Object)ioe);
                tmpCacheEnabled = false;
            }
        }
        this.cacheEnabled = tmpCacheEnabled;
        long tmpCacheCheckIntervalSecs = 86400L;
        try {
            tmpCacheCheckIntervalSecs = Long.parseLong(Configuration.getInstance().getProperty(CONFIG_PROPERTY_CACHE_CHECK_INTERVAL_SECS));
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        this.cacheCheckIntervalSecs = tmpCacheCheckIntervalSecs;
    }

    public static DownloadCache getInstance() {
        if (singleton == null) {
            singleton = new DownloadCache();
        }
        return singleton;
    }

    private static void rmdir(File dir) throws IOException {
        if (Objects.isNull(dir) || !dir.exists()) {
            return;
        }
        File[] contents = dir.listFiles();
        if (Objects.nonNull(contents)) {
            for (File f : contents) {
                DownloadCache.rmdir(f);
            }
        }
        Files.delete(dir.toPath());
    }

    public void resetCache() throws IOException {
        File cacheDirectory = new File(this.cacheDir);
        DownloadCache.rmdir(cacheDirectory);
        Files.createDirectories(cacheDirectory.toPath(), new FileAttribute[0]);
    }

    public InputStream getUrlInputStream(URL url) throws IOException {
        return this.getUrlInputStream(url, true);
    }

    public InputStream getUrlInputStream(URL url, boolean restrictRedirects) throws IOException {
        InputStream result = null;
        if (url != null) {
            result = this.cacheEnabled ? this.getUrlInputStreamThroughCache(url, restrictRedirects) : this.getUrlInputStreamDirect(url, restrictRedirects);
        }
        return result;
    }

    private InputStream getUrlInputStreamDirect(URL url, boolean restrictRedirects) throws IOException {
        int status;
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setReadTimeout(5000);
        URL redirectUrl = this.processPossibleRedirect(connection, restrictRedirects);
        if (redirectUrl != null) {
            url = redirectUrl;
            connection = (HttpURLConnection)redirectUrl.openConnection();
            connection.setReadTimeout(5000);
        }
        if ((status = connection.getResponseCode()) != 200) {
            throw new IOException("Unexpected HTTP status code from " + url + ": " + status);
        }
        InputStream result = connection.getInputStream();
        return result;
    }

    private InputStream getUrlInputStreamThroughCache(URL url, boolean restrictRedirects) throws IOException {
        String cacheKey = this.base64Encode(url);
        File cachedFile = new File(this.cacheDir, cacheKey);
        File cachedMetadataFile = new File(this.cacheDir, cacheKey + ".metadata.json");
        if (cachedFile.exists() && cachedMetadataFile.exists()) {
            try {
                this.checkCache(url, restrictRedirects);
            }
            catch (IOException iOException) {}
        } else {
            this.cacheMiss(url, restrictRedirects);
        }
        return new BufferedInputStream(Files.newInputStream(cachedFile.toPath(), new OpenOption[0]));
    }

    private void checkCache(URL url, boolean restrictRedirects) throws IOException {
        String cacheKey = this.base64Encode(url);
        File cachedMetadataFile = new File(this.cacheDir, cacheKey + ".metadata.json");
        HashMap<String, String> cachedMetadata = this.readMetadataFile(cachedMetadataFile);
        if (cachedMetadata != null) {
            long difference;
            Instant lastChecked = this.parseISO8601String(cachedMetadata.get("lastChecked"));
            long l = difference = lastChecked != null ? Math.abs(ChronoUnit.SECONDS.between(Instant.now(), lastChecked)) : Long.MAX_VALUE;
            if (difference > this.cacheCheckIntervalSecs) {
                logger.debug("Cache check interval exceeded; checking for updates to {}", (Object)url);
                String eTag = cachedMetadata.get("eTag");
                HttpURLConnection connection = (HttpURLConnection)url.openConnection();
                connection.setReadTimeout(5000);
                connection.setRequestProperty("If-None-Match", eTag);
                int status = connection.getResponseCode();
                if (status != 304) {
                    this.cacheMiss(url, connection, restrictRedirects);
                } else {
                    logger.debug("Cache hit for {}", (Object)url);
                    cachedMetadata.put("lastChecked", this.iso8601.format(Instant.now()));
                    this.writeMetadataFile(cachedMetadataFile, cachedMetadata);
                }
            } else {
                logger.debug("Within cache check interval; skipping check of updates to {}", (Object)url);
            }
        } else {
            this.cacheMiss(url, restrictRedirects);
        }
    }

    private void cacheMiss(URL url, HttpURLConnection connection, boolean restrictRedirects) throws IOException {
        int status;
        logger.debug("Cache miss for {}", (Object)url);
        URL redirectUrl = this.processPossibleRedirect(connection, restrictRedirects);
        if (redirectUrl != null) {
            url = redirectUrl;
            connection = (HttpURLConnection)redirectUrl.openConnection();
        }
        if ((status = connection.getResponseCode()) != 200) {
            throw new IOException("Unexpected HTTP status code from " + url.toString() + ": " + status);
        }
        String cacheKey = this.base64Encode(url);
        File cachedFile = new File(this.cacheDir, cacheKey);
        this.writeContentFile(connection.getInputStream(), cachedFile);
        File cachedMetadataFile = new File(this.cacheDir, cacheKey + ".metadata.json");
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put("eTag", connection.getHeaderField("ETag"));
        metadata.put("downloadedAt", this.iso8601.format(Instant.now()));
        metadata.put("lastChecked", this.iso8601.format(Instant.now()));
        metadata.put("sourceUrl", url.toString());
        this.writeMetadataFile(cachedMetadataFile, metadata);
    }

    private void cacheMiss(URL url, boolean restrictRedirects) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setReadTimeout(5000);
        this.cacheMiss(url, connection, restrictRedirects);
    }

    private URL processPossibleRedirect(HttpURLConnection connection, boolean restrictRedirects) throws IOException {
        URL result = null;
        int status = connection.getResponseCode();
        if (status == 302 || status == 301 || status == 303) {
            String redirectUrlStr = connection.getHeaderField("Location");
            if (Objects.isNull(redirectUrlStr) || redirectUrlStr.isEmpty()) {
                throw new IOException("Empty redirect URL response");
            }
            try {
                result = new URL(redirectUrlStr);
            }
            catch (Exception ex) {
                throw new IOException("Invalid redirect URL", ex);
            }
            if (!result.getProtocol().toLowerCase().startsWith("http")) {
                throw new IOException("Invalid redirect protocol");
            }
            if (restrictRedirects && !WHITE_LIST.contains(result.getHost())) {
                throw new IOException("Invalid redirect host - not on the allowed 'white list'");
            }
        }
        return result;
    }

    private HashMap<String, String> readMetadataFile(File metadataFile) {
        HashMap result;
        try {
            BufferedReader r = new BufferedReader(new FileReader(metadataFile));
            result = (HashMap)new Gson().fromJson((Reader)r, new TypeToken<HashMap<String, String>>(){}.getType());
        }
        catch (IOException ioe) {
            result = null;
        }
        return result;
    }

    private void writeMetadataFile(File metadataFile, HashMap<String, String> metadata) throws IOException {
        try (BufferedWriter w = new BufferedWriter(new FileWriter(metadataFile));){
            new Gson().toJson(metadata, new TypeToken<HashMap<String, String>>(){}.getType(), (Appendable)w);
            ((Writer)w).flush();
        }
    }

    private void writeContentFile(InputStream is, File cachedFile) throws IOException {
        try (BufferedOutputStream cacheFileOutputStream = new BufferedOutputStream(Files.newOutputStream(cachedFile.toPath(), new OpenOption[0]));){
            int length;
            byte[] ioBuffer = new byte[8192];
            while ((length = is.read(ioBuffer)) != -1) {
                ((OutputStream)cacheFileOutputStream).write(ioBuffer, 0, length);
            }
            ((OutputStream)cacheFileOutputStream).flush();
        }
        is.close();
    }

    private Instant parseISO8601String(String s) {
        Instant result = null;
        if (s != null) {
            try {
                result = Instant.parse(s);
            }
            catch (DateTimeParseException dtpe) {
                result = null;
            }
        }
        return result;
    }

    private String base64Encode(String s) {
        String result = null;
        if (s != null) {
            result = Base64.getEncoder().encodeToString(s.getBytes(StandardCharsets.UTF_8));
        }
        return result;
    }

    private String base64Encode(URL u) {
        String result = null;
        if (u != null) {
            result = this.base64Encode(u.toString());
        }
        return result;
    }
}

