/**
 * Copyright (c) 2024 Source Auditor Inc.
 *
 * SPDX-License-Identifier: Apache-2.0
 * 
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */
 
package org.spdx.library.model.v3_0_1.build;

import javax.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.spdx.core.CoreModelObject;
import org.spdx.core.DefaultModelStore;
import org.spdx.core.InvalidSPDXAnalysisException;
import org.spdx.core.IModelCopyManager;
import org.spdx.core.IndividualUriValue;
import org.spdx.library.model.v3_0_1.ModelObjectV3;
import org.spdx.storage.IModelStore;
import org.spdx.storage.IModelStore.IdType;
import org.spdx.storage.IModelStore.IModelStoreLock;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import org.spdx.library.model.v3_0_1.SpdxConstantsV3;
import org.spdx.library.model.v3_0_1.core.CreationInfo;
import org.spdx.library.model.v3_0_1.core.DictionaryEntry;
import org.spdx.library.model.v3_0_1.core.Element;
import org.spdx.library.model.v3_0_1.core.ExternalIdentifier;
import org.spdx.library.model.v3_0_1.core.ExternalRef;
import org.spdx.library.model.v3_0_1.core.Hash;
import org.spdx.library.model.v3_0_1.core.IntegrityMethod;
import org.spdx.library.model.v3_0_1.core.ProfileIdentifierType;
import org.spdx.library.model.v3_0_1.extension.Extension;

/**
 * DO NOT EDIT - this file is generated by the Owl to Java Utility 
 * See: https://github.com/spdx/tools-java 
 * 
 * Class that describes a build instance of software/artifacts. 
 */
public  class Build extends Element  {

	Collection<DictionaryEntry> environments = Collections.emptyList();
	Collection<Hash> configSourceDigests = Collections.emptyList();
	Collection<DictionaryEntry> parameters = Collections.emptyList();
	Collection<String> configSourceEntrypoints = Collections.emptyList();
	Collection<String> configSourceUris = Collections.emptyList();
	
	/**
	 * Create the Build with default model store and generated anonymous ID
	 * @throws InvalidSPDXAnalysisException when unable to create the Build
	 */
	public Build() throws InvalidSPDXAnalysisException {
		this(DefaultModelStore.getDefaultModelStore().getNextId(IdType.Anonymous));
	}

	/**
	 * @param objectUri URI or anonymous ID for the Build
	 * @throws InvalidSPDXAnalysisException when unable to create the Build
	 */
	public Build(String objectUri) throws InvalidSPDXAnalysisException {
		this(DefaultModelStore.getDefaultModelStore(), objectUri, DefaultModelStore.getDefaultCopyManager(), 
				true, DefaultModelStore.getDefaultDocumentUri() + "#");
	}

	/**
	 * @param modelStore Model store where the Build is to be stored
	 * @param objectUri URI or anonymous ID for the Build
	 * @param copyManager Copy manager for the Build - can be null if copying is not required
	 * @param create true if Build is to be created
	 * @param idPrefix - prefix to be used when generating new SPDX IDs
	 * @throws InvalidSPDXAnalysisException when unable to create the Build
	 */
	 @SuppressWarnings("unchecked")
	public Build(IModelStore modelStore, String objectUri, @Nullable IModelCopyManager copyManager,
			boolean create, String idPrefix) throws InvalidSPDXAnalysisException {
		super(modelStore, objectUri, copyManager, create, idPrefix);
		if (!isExternal()) {
			environments = (Collection<DictionaryEntry>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_ENVIRONMENT, DictionaryEntry.class);
		}
		if (!isExternal()) {
			configSourceDigests = (Collection<Hash>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_DIGEST, Hash.class);
		}
		if (!isExternal()) {
			parameters = (Collection<DictionaryEntry>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_PARAMETER, DictionaryEntry.class);
		}
		if (!isExternal()) {
			configSourceEntrypoints = (Collection<String>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_ENTRYPOINT, String.class);
		}
		if (!isExternal()) {
			configSourceUris = (Collection<String>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_URI, String.class);
		}
	}

	/**
	 * Create the Build from the builder - used in the builder class
	 * @param builder Builder to create the Build from
	 * @throws InvalidSPDXAnalysisException when unable to create the Build
	 */
	 @SuppressWarnings("unchecked")
	protected Build(BuildBuilder builder) throws InvalidSPDXAnalysisException {
		super(builder);
		environments = (Collection<DictionaryEntry>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_ENVIRONMENT, DictionaryEntry.class);
		configSourceDigests = (Collection<Hash>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_DIGEST, Hash.class);
		parameters = (Collection<DictionaryEntry>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_PARAMETER, DictionaryEntry.class);
		configSourceEntrypoints = (Collection<String>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_ENTRYPOINT, String.class);
		configSourceUris = (Collection<String>)(Collection<?>)this.getObjectPropertyValueCollection(SpdxConstantsV3.PROP_CONFIG_SOURCE_URI, String.class);
		getEnvironments().addAll(builder.environments);
		getConfigSourceDigests().addAll(builder.configSourceDigests);
		getParameters().addAll(builder.parameters);
		getConfigSourceEntrypoints().addAll(builder.configSourceEntrypoints);
		getConfigSourceUris().addAll(builder.configSourceUris);
		setBuildId(builder.buildId);
		setBuildEndTime(builder.buildEndTime);
		setBuildStartTime(builder.buildStartTime);
		if (Objects.nonNull(builder.buildType)) {
			setBuildType(builder.buildType);
		}
	}

	/* (non-Javadoc)
	 * @see org.spdx.library.core.CoreModelObject#getType()
	 */
	@Override
	public String getType() {
		return "Build.Build";
	}
	
	// Getters and Setters
	public Collection<DictionaryEntry> getEnvironments() {
		return environments;
	}
	public Collection<Hash> getConfigSourceDigests() {
		return configSourceDigests;
	}
	public Collection<DictionaryEntry> getParameters() {
		return parameters;
	}
	public Collection<String> getConfigSourceEntrypoints() {
		return configSourceEntrypoints;
	}
	public Collection<String> getConfigSourceUris() {
		return configSourceUris;
	}
	
	/**
	 * @param creationInfo the creationInfo to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Build setCreationInfo(@Nullable CreationInfo creationInfo) throws InvalidSPDXAnalysisException {
		super.setCreationInfo(creationInfo);
		return this;
	}	
		/**
	 * @return the buildId
	 */
	public Optional<String> getBuildId() throws InvalidSPDXAnalysisException {
		return getStringPropertyValue(SpdxConstantsV3.PROP_BUILD_ID);
	}
	/**
	 * @param buildId the buildId to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Build setBuildId(@Nullable String buildId) throws InvalidSPDXAnalysisException {
		setPropertyValue(SpdxConstantsV3.PROP_BUILD_ID, buildId);
		return this;
	}
		/**
	 * @return the buildEndTime
	 */
	public Optional<String> getBuildEndTime() throws InvalidSPDXAnalysisException {
		return getStringPropertyValue(SpdxConstantsV3.PROP_BUILD_END_TIME);
	}
	/**
	 * @param buildEndTime the buildEndTime to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Build setBuildEndTime(@Nullable String buildEndTime) throws InvalidSPDXAnalysisException {
		if (isStrict() && Objects.nonNull(buildEndTime) && !Pattern.matches("^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$", buildEndTime)) {
			throw new InvalidSPDXAnalysisException("buildEndTime value '" + buildEndTime + 
				"' does not match the pattern '^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$'");
		}
		setPropertyValue(SpdxConstantsV3.PROP_BUILD_END_TIME, buildEndTime);
		return this;
	}
	/**
	 * @param summary the summary to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Build setSummary(@Nullable String summary) throws InvalidSPDXAnalysisException {
		super.setSummary(summary);
		return this;
	}	
	/**
	 * @param description the description to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Build setDescription(@Nullable String description) throws InvalidSPDXAnalysisException {
		super.setDescription(description);
		return this;
	}	
	/**
	 * @param comment the comment to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Build setComment(@Nullable String comment) throws InvalidSPDXAnalysisException {
		super.setComment(comment);
		return this;
	}	
	/**
	 * @param name the name to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	 @Override
	public Build setName(@Nullable String name) throws InvalidSPDXAnalysisException {
		super.setName(name);
		return this;
	}	
		/**
	 * @return the buildStartTime
	 */
	public Optional<String> getBuildStartTime() throws InvalidSPDXAnalysisException {
		return getStringPropertyValue(SpdxConstantsV3.PROP_BUILD_START_TIME);
	}
	/**
	 * @param buildStartTime the buildStartTime to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Build setBuildStartTime(@Nullable String buildStartTime) throws InvalidSPDXAnalysisException {
		if (isStrict() && Objects.nonNull(buildStartTime) && !Pattern.matches("^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$", buildStartTime)) {
			throw new InvalidSPDXAnalysisException("buildStartTime value '" + buildStartTime + 
				"' does not match the pattern '^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$'");
		}
		setPropertyValue(SpdxConstantsV3.PROP_BUILD_START_TIME, buildStartTime);
		return this;
	}
	/**
	 * @return the buildType
	 */
	public @Nullable String getBuildType() throws InvalidSPDXAnalysisException {
		Optional<String> retval = getStringPropertyValue(SpdxConstantsV3.PROP_BUILD_TYPE);
		return retval.isPresent() ? retval.get() : null;
	}
		/**
	 * @param buildType the buildType to set
	 * @return this to chain setters
	 * @throws InvalidSPDXAnalysisException 
	 */
	public Build setBuildType(@Nullable String buildType) throws InvalidSPDXAnalysisException {
		if (isStrict() && Objects.isNull(buildType)) {
			throw new InvalidSPDXAnalysisException("buildType is a required property");
		}
		setPropertyValue(SpdxConstantsV3.PROP_BUILD_TYPE, buildType);
		return this;
	}
	
	
	@Override
	public String toString() {
				return super.toString();
	}
	
	/* (non-Javadoc)
	 * @see org.spdx.library.model.v3.ModelObjectV3#_verify(java.util.List)
	 */
	@Override
	public List<String> _verify(Set<String> verifiedIds, String specVersionForVerify, List<IndividualUriValue> profiles) {
		List<String> retval = new ArrayList<>();
		retval.addAll(super._verify(verifiedIds, specVersionForVerify, profiles));
		try {
			@SuppressWarnings("unused")
			Optional<String> buildId = getBuildId();
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting buildId for Build: "+e.getMessage());
		}
		try {
			Optional<String> buildEndTime = getBuildEndTime();
			if (buildEndTime.isPresent() && !Pattern.matches("^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$", buildEndTime.get())) {
				retval.add("buildEndTime value '" + buildEndTime.get() + 
					"' does not match the pattern '^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$'");
			}
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting buildEndTime for Build: "+e.getMessage());
		}
		try {
			Optional<String> buildStartTime = getBuildStartTime();
			if (buildStartTime.isPresent() && !Pattern.matches("^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$", buildStartTime.get())) {
				retval.add("buildStartTime value '" + buildStartTime.get() + 
					"' does not match the pattern '^\\d\\d\\d\\d-\\d\\d-\\d\\dT\\d\\d:\\d\\d:\\d\\dZ$'");
			}
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting buildStartTime for Build: "+e.getMessage());
		}
		try {
			String buildType = getBuildType();
			if (Objects.isNull(buildType) &&
					Collections.disjoint(profiles, Arrays.asList(new ProfileIdentifierType[] { ProfileIdentifierType.BUILD }))) {
				retval.add("Missing buildType in Build");
			}
		} catch (InvalidSPDXAnalysisException e) {
			retval.add("Error getting buildType for Build: "+e.getMessage());
		}
		for (DictionaryEntry environment:environments) {
			retval.addAll(environment.verify(verifiedIds, specVersionForVerify, profiles));
		}
		for (Hash configSourceDigest:configSourceDigests) {
			retval.addAll(configSourceDigest.verify(verifiedIds, specVersionForVerify, profiles));
		}
		for (DictionaryEntry parameter:parameters) {
			retval.addAll(parameter.verify(verifiedIds, specVersionForVerify, profiles));
		}
		return retval;
	}
	
	
	
	public static class BuildBuilder extends ElementBuilder {
	
		/**
		 * Create an BuildBuilder from another model object copying the modelStore and copyManager and using an anonymous ID
		 * @param from model object to copy the model store and copyManager from
		 * @throws InvalidSPDXAnalysisException
		 */
		public BuildBuilder(ModelObjectV3 from) throws InvalidSPDXAnalysisException {
			this(from, from.getModelStore().getNextId(IdType.Anonymous));
		}
	
		/**
		 * Create an BuildBuilder from another model object copying the modelStore and copyManager
		 * @param from model object to copy the model store and copyManager from
		 * @param objectUri URI for the object
		 * @throws InvalidSPDXAnalysisException if creationInfo can not be fetched from the from
		 */
		public BuildBuilder(ModelObjectV3 from, String objectUri) throws InvalidSPDXAnalysisException {
			this(from.getModelStore(), objectUri, from.getCopyManager());
			setStrict(from.isStrict());
			setIdPrefix(from.getIdPrefix());
			if (from instanceof Element) {
				setCreationInfo(((Element)from).getCreationInfo());
			} else if (from instanceof CreationInfo) {
				setCreationInfo((CreationInfo)from);
			}
		}
		
		/**
		 * Creates a BuildBuilder
		 * @param modelStore model store for the built Build
		 * @param objectUri objectUri for the built Build
		 * @param copyManager optional copyManager for the built Build
		 */
		public BuildBuilder(IModelStore modelStore, String objectUri, @Nullable IModelCopyManager copyManager) {
			super(modelStore, objectUri, copyManager);
		}
		
		protected Collection<DictionaryEntry> environments = new ArrayList<>();
		protected Collection<Hash> configSourceDigests = new ArrayList<>();
		protected Collection<DictionaryEntry> parameters = new ArrayList<>();
		protected Collection<String> configSourceEntrypoints = new ArrayList<>();
		protected Collection<String> configSourceUris = new ArrayList<>();
		protected String buildId = null;
		protected String buildEndTime = null;
		protected String buildStartTime = null;
		protected String buildType = null;
		
		
		/**
		 * Adds a environment to the initial collection
		 * @parameter environment environment to add
		 * @return this for chaining
		**/
		public BuildBuilder addEnvironment(DictionaryEntry environment) {
			if (Objects.nonNull(environment)) {
				environments.add(environment);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial environment collection
		 * @parameter environmentCollection collection to initialize the environment
		 * @return this for chaining
		**/
		public BuildBuilder addAllEnvironment(Collection<DictionaryEntry> environmentCollection) {
			if (Objects.nonNull(environmentCollection)) {
				environments.addAll(environmentCollection);
			}
			return this;
		}
		
		/**
		 * Adds a extension to the initial collection
		 * @parameter extension extension to add
		 * @return this for chaining
		**/
		public BuildBuilder addExtension(Extension extension) {
			if (Objects.nonNull(extension)) {
				super.extensions.add(extension);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial extension collection
		 * @parameter extensionCollection collection to initialize the extension
		 * @return this for chaining
		**/
		public BuildBuilder addAllExtension(Collection<Extension> extensionCollection) {
			if (Objects.nonNull(extensionCollection)) {
				super.extensions.addAll(extensionCollection);
			}
			return this;
		}
		
		/**
		 * Adds a verifiedUsing to the initial collection
		 * @parameter verifiedUsing verifiedUsing to add
		 * @return this for chaining
		**/
		public BuildBuilder addVerifiedUsing(IntegrityMethod verifiedUsing) {
			if (Objects.nonNull(verifiedUsing)) {
				super.verifiedUsings.add(verifiedUsing);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial verifiedUsing collection
		 * @parameter verifiedUsingCollection collection to initialize the verifiedUsing
		 * @return this for chaining
		**/
		public BuildBuilder addAllVerifiedUsing(Collection<IntegrityMethod> verifiedUsingCollection) {
			if (Objects.nonNull(verifiedUsingCollection)) {
				super.verifiedUsings.addAll(verifiedUsingCollection);
			}
			return this;
		}
		
		/**
		 * Adds a externalRef to the initial collection
		 * @parameter externalRef externalRef to add
		 * @return this for chaining
		**/
		public BuildBuilder addExternalRef(ExternalRef externalRef) {
			if (Objects.nonNull(externalRef)) {
				super.externalRefs.add(externalRef);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial externalRef collection
		 * @parameter externalRefCollection collection to initialize the externalRef
		 * @return this for chaining
		**/
		public BuildBuilder addAllExternalRef(Collection<ExternalRef> externalRefCollection) {
			if (Objects.nonNull(externalRefCollection)) {
				super.externalRefs.addAll(externalRefCollection);
			}
			return this;
		}
		
		/**
		 * Adds a configSourceDigest to the initial collection
		 * @parameter configSourceDigest configSourceDigest to add
		 * @return this for chaining
		**/
		public BuildBuilder addConfigSourceDigest(Hash configSourceDigest) {
			if (Objects.nonNull(configSourceDigest)) {
				configSourceDigests.add(configSourceDigest);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial configSourceDigest collection
		 * @parameter configSourceDigestCollection collection to initialize the configSourceDigest
		 * @return this for chaining
		**/
		public BuildBuilder addAllConfigSourceDigest(Collection<Hash> configSourceDigestCollection) {
			if (Objects.nonNull(configSourceDigestCollection)) {
				configSourceDigests.addAll(configSourceDigestCollection);
			}
			return this;
		}
		
		/**
		 * Adds a parameter to the initial collection
		 * @parameter parameter parameter to add
		 * @return this for chaining
		**/
		public BuildBuilder addParameter(DictionaryEntry parameter) {
			if (Objects.nonNull(parameter)) {
				parameters.add(parameter);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial parameter collection
		 * @parameter parameterCollection collection to initialize the parameter
		 * @return this for chaining
		**/
		public BuildBuilder addAllParameter(Collection<DictionaryEntry> parameterCollection) {
			if (Objects.nonNull(parameterCollection)) {
				parameters.addAll(parameterCollection);
			}
			return this;
		}
		
		/**
		 * Adds a externalIdentifier to the initial collection
		 * @parameter externalIdentifier externalIdentifier to add
		 * @return this for chaining
		**/
		public BuildBuilder addExternalIdentifier(ExternalIdentifier externalIdentifier) {
			if (Objects.nonNull(externalIdentifier)) {
				super.externalIdentifiers.add(externalIdentifier);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial externalIdentifier collection
		 * @parameter externalIdentifierCollection collection to initialize the externalIdentifier
		 * @return this for chaining
		**/
		public BuildBuilder addAllExternalIdentifier(Collection<ExternalIdentifier> externalIdentifierCollection) {
			if (Objects.nonNull(externalIdentifierCollection)) {
				super.externalIdentifiers.addAll(externalIdentifierCollection);
			}
			return this;
		}
		
		/**
		 * Adds a configSourceEntrypoint to the initial collection
		 * @parameter configSourceEntrypoint configSourceEntrypoint to add
		 * @return this for chaining
		**/
		public BuildBuilder addConfigSourceEntrypoint(String configSourceEntrypoint) {
			if (Objects.nonNull(configSourceEntrypoint)) {
				configSourceEntrypoints.add(configSourceEntrypoint);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial configSourceEntrypoint collection
		 * @parameter configSourceEntrypointCollection collection to initialize the configSourceEntrypoint
		 * @return this for chaining
		**/
		public BuildBuilder addAllConfigSourceEntrypoint(Collection<String> configSourceEntrypointCollection) {
			if (Objects.nonNull(configSourceEntrypointCollection)) {
				configSourceEntrypoints.addAll(configSourceEntrypointCollection);
			}
			return this;
		}
		
		/**
		 * Adds a configSourceUri to the initial collection
		 * @parameter configSourceUri configSourceUri to add
		 * @return this for chaining
		**/
		public BuildBuilder addConfigSourceUri(String configSourceUri) {
			if (Objects.nonNull(configSourceUri)) {
				configSourceUris.add(configSourceUri);
			}
			return this;
		}
		
		/**
		 * Adds all elements from a collection to the initial configSourceUri collection
		 * @parameter configSourceUriCollection collection to initialize the configSourceUri
		 * @return this for chaining
		**/
		public BuildBuilder addAllConfigSourceUri(Collection<String> configSourceUriCollection) {
			if (Objects.nonNull(configSourceUriCollection)) {
				configSourceUris.addAll(configSourceUriCollection);
			}
			return this;
		}
		
		/**
		 * Sets the initial value of creationInfo
		 * @parameter creationInfo value to set
		 * @return this for chaining
		**/
		public BuildBuilder setCreationInfo(CreationInfo creationInfo) {
			super.creationInfo = creationInfo;
			return this;
		}
		
		/**
		 * Sets the initial value of buildId
		 * @parameter buildId value to set
		 * @return this for chaining
		**/
		public BuildBuilder setBuildId(String buildId) {
			this.buildId = buildId;
			return this;
		}
		
		/**
		 * Sets the initial value of buildEndTime
		 * @parameter buildEndTime value to set
		 * @return this for chaining
		**/
		public BuildBuilder setBuildEndTime(String buildEndTime) {
			this.buildEndTime = buildEndTime;
			return this;
		}
		
		/**
		 * Sets the initial value of summary
		 * @parameter summary value to set
		 * @return this for chaining
		**/
		public BuildBuilder setSummary(String summary) {
			super.summary = summary;
			return this;
		}
		
		/**
		 * Sets the initial value of description
		 * @parameter description value to set
		 * @return this for chaining
		**/
		public BuildBuilder setDescription(String description) {
			super.description = description;
			return this;
		}
		
		/**
		 * Sets the initial value of comment
		 * @parameter comment value to set
		 * @return this for chaining
		**/
		public BuildBuilder setComment(String comment) {
			super.comment = comment;
			return this;
		}
		
		/**
		 * Sets the initial value of name
		 * @parameter name value to set
		 * @return this for chaining
		**/
		public BuildBuilder setName(String name) {
			super.name = name;
			return this;
		}
		
		/**
		 * Sets the initial value of buildStartTime
		 * @parameter buildStartTime value to set
		 * @return this for chaining
		**/
		public BuildBuilder setBuildStartTime(String buildStartTime) {
			this.buildStartTime = buildStartTime;
			return this;
		}
		
		/**
		 * Sets the initial value of buildType
		 * @parameter buildType value to set
		 * @return this for chaining
		**/
		public BuildBuilder setBuildType(String buildType) {
			this.buildType = buildType;
			return this;
		}
	
		/**
		 * @return the Build
		 * @throws InvalidSPDXAnalysisException on any errors during build
		 */
		public Build build() throws InvalidSPDXAnalysisException {
			IModelStoreLock lock = getModelStore().enterCriticalSection(false);
			try {
				return new Build(this);
			} finally {
				getModelStore().leaveCriticalSection(lock);
			}
		}
	}
}
